/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// EeStore.h: interface for the EeStore class.
//
//////////////////////////////////////////////////////////////////////

#ifndef EESTORE_H
#define EESTORE_H


#include "NewPKIStore.h"
#include "ClientLDAP.h"


/*!
	This class is the store for an EE
*/
class EeStore : public NewPKIStore  
{
public:
	/*! \brief This is the constructor.
	 *  \param EntityName [IN] The name of the entity.
	 *  \param e [IN] The ENGINE, can be NULL.
	 */
	EeStore(const mString & EntityName, ENGINE * e);

	/*! \brief This is the destructor.
	 */
	virtual ~EeStore();

	bool CreateTables(const SQL_Connection * DbConn);

	/*! \brief This function sets the LDAP authentication options.
	 *  \param DnSpecs [IN] The list of DN attributes.
	 *  \param Policies [IN] The policies.
	 *  \param ldap_auth [IN] LDAP auth is on or off.
	 *  \param ldap_server [IN] The ldap server's address.
	 *  \param ldap_port [IN] The ldap server's port.
	 *  \param ldap_username [IN] The username.
	 *  \param ldap_password [IN] The password.
	 *  \param ldap_base [IN] The base to search in.
	 *  \param ldap_attr_name [IN] The name of the attribute that hold the uid.
	 *  \param ldap_filters [IN] The LDAP filters.
	 *  \param utf8 [IN] The LDAP server uses UTF-8.
	 *  \return true on success, false on failure.
	 */
	bool SetOptions(const mVector<DnSpecs> & DnSpecs,
					const HashTable_String & Policies,
					bool ldap_auth,
					const mString & ldap_server,
					unsigned long ldap_port,
					const mString & ldap_username, 
					const mString & ldap_password,
					const mString & ldap_base,
					const mString & ldap_attr_name,
					const mString & ldap_filters,
					unsigned long utf8);

	/*! \brief This function inserts a new DN validation request.
	 *  \param transactionId [IN] The transaction ID.
	 *  \param ra_id [IN] The RA id.
	 *  \param dn [IN] The DN.
	 *  \return true on success, false on failure.
	 */
	bool InsertDnValidationRequest(const Asn1OctetString & transactionId, unsigned long ra_id, const X509_NAME * dn);

	/*! \brief This function deletes a DN validation request.
	 *  \param transactionId [IN] The transaction ID.
	 *  \return true on success, false on failure.
	 */
	bool DeleteDnValidationRequest(const Asn1OctetString & transactionId);

	/*! \brief This function returns a DN validation request.
	 *  \param transactionId [IN] The transaction ID.
	 *  \return the DN on success, NULL on failure.
	 */
	X509_NAME * GetDnValidationRequest(const Asn1OctetString & transactionId);

	/*! \brief This function returns the RA ID associated with a EE validation request.
	 *  \param transactionId [IN] The transaction ID.
	 *  \param ra_id [OUT] The RA id.
	 *  \return true on success, false on failure.
	 */
	bool GetDnValidationRequestRaId(const Asn1OctetString & transactionId, unsigned long & ra_id);

	/*! \brief This function set the site name that is sent in the users' notification emails.
	 *  \param siteName [IN] The site name.
	 */
	void set_SiteName(const mString & siteName);

	/*! \brief This function is used to resend notification emails.
	 */
	void ResendNotificationMails();

	/*! \brief This function creates a web user.
	 *  \param email [IN] The email address.
	 *  \param password [IN] The password.
	 *  \param dn [IN] The DN.
	 *  \return The user id on sucess, 0 on failure.
	 */
	unsigned long WebUserCreate(const mString & email, const mString & password, const X509_NAME * dn);

	/*! \brief This function returns the user id.
	 *  \param otp [IN] The one time password sent by email.
	 *  \param user_id [OUT] The user id.
	 *  \return true on success, false on failure.
	 */
	bool WebUserGetUserId(const mString & otp, unsigned long & user_id);

	/*! \brief This function returns the user DN.
	 *  \param user_id [IN] The user ID.
	 *  \return The DN on success, NULL on failure.
	 */
	X509_NAME * WebUserGetUserDn(unsigned long user_id);

	/*! \brief This function activates a web user.
	 *  \param otp [IN] The one time password sent by email.
	 *  \return true on success, false on failure.
	 */
	bool WebUserActivate(const mString & otp);

	/*! \brief This function deactivates a web user.
	 *  \param otp [IN] The one time password sent by email.
	 *  \return true on success, false on failure.
	 */
	bool WebUserDeactivate(const mString & otp);

	/*! \brief This function logs in a web user.
	 *  \param email [IN] The email address.
	 *  \param password [IN] The password.
	 *  \param created_user_id [OUT] The user id created by LDAP importation.
	 *  \return The user_id on success, 0 on failure.
	 */
	unsigned long WebUserLogin(const mString & email, const mString & password, unsigned long & created_user_id);

	/*! \brief This function deletes a web user.
	 *  \param user_id [IN] The user id to delete.
	 *  \return true on success, false on failure.
	 */
	bool WebUserDelete(unsigned long user_id);

	/*! \brief This function returns the email address associated to a user id.
	 *  \param user_id [IN] The user id.
	 *  \param email [OUT] The email address.
	 *  \return true on success, false on failure.
	 */
	bool WebUserGetEmail(unsigned long user_id, mString & email);

	/*! \brief This function marks a web user as RA validated.
	 *  \param user_id [IN] The user id.
	 *  \return true on success, false on failure.
	 */
	bool WebUserRaValidated(unsigned long user_id);

	/*! \brief This function inserts a certificate request for a web user.
	 *  \param request [IN] The request.
	 *  \param CertReqId [OUT] The certificate request id.
	 *  \return true on success, false on failure.
	 */
	bool WebUserInsertCertReq(const NewpkiEeRequestCert & request, unsigned long & CertReqId);

	/*! \brief This function marks a certificate as witing for revokation.
	 *  \param user_id [IN] The owner's ID.
	 *  \param cert_id [IN] The certificate's ID.
	 *  \param ra_id [OUT] The RA ID for this certificate.
	 *  \return true on success, false on failure.
	 */
	bool WebUserMarkCertRev(unsigned long user_id, unsigned long cert_id, unsigned long & ra_id);

	/*! \brief This function marks a certificate as witing for revokation.
	 *  \param ra_id [IN] The RA ID.
	 *  \param status [IN] The status.
	 *  \param lastCrl [IN] The last CRL from the CA.
	 *  \return true on success, false on failure.
	 */
	bool WebUserSetStatusByRaId(unsigned long ra_id, int status, const PKI_CRL & lastCrl);

	/*! \brief This function unmarks a certificate as witing for revokation.
	 *  \param cert_id [IN] The certificate's ID.
	 *  \return true on success, false on failure.
	 */
	bool WebUserUnmarkCertRev(unsigned long cert_id);

	/*! \brief This function removes a certificate request for a web user.
	 *  \param CertReqId [IN] The certificate request id.
	 *  \return true on success, false on failure.
	 */
	bool WebUserDeleteCertReq(unsigned long CertReqId);

	/*! \brief This function inserts a certificate response for a web user.
	 *  \param CertReqId [IN] The certificate request id.
	 *  \param cert_response [IN] The response.
	 *  \param errors [IN] Optional errors.
	 *  \param user_id [OUT] The owner's id.
	 *  \return true on success, false on failure.
	 */
	bool WebUserImportCertResponse(unsigned long CertReqId, const NewpkiEeCertResponse & cert_response, const mVector< ErrorEntry > & errors, unsigned long & user_id);

	/*! \brief This function inserts a certificate for a web user.
	 *  \param Request [IN] The certifciate.
	 *  \return true on success, false on failure.
	 */
	bool WebUserPublishCert(const NewpkiEeCertPublish & Request);

	/*! \brief This function inserts a revocation response for a web user.
	 *  \param CertReqId [IN] The certificate request id.
	 *  \param rev_response [IN] The response.
	 *  \param errors [IN] Optional errors.
	 *  \param user_id [OUT] The owner's id.
	 *  \return true on success, false on failure.
	 */
	bool WebUserImportRevResponse(unsigned long CertReqId, const NewpkiEeRevResponse & rev_response, const mVector< ErrorEntry > & errors, unsigned long & user_id);

	/*! \brief This function gets a certificate request for a web user.
	 *  \param cert_id [IN] The certificate request id.
	 *  \param Cert [OUT] The certificate request.
	 *  \param user_id [OUT] The owner's id.
	 *  \return true on success, false on failure.
	 */
	bool WebUserGetCertReq(unsigned long cert_id, NewpkiProfileDatasCert & Cert, unsigned long & user_id);

	/*! \brief This function enumerates the certs of a web user.
	 *  \param user_id [IN] The web user id.
	 *  \param Certs [OUT] The certificates.
	 *  \return true on success, false on failure.
	 */
	bool WebUserEnumCerts(unsigned long user_id, mVector<NewpkiProfileDatasCert> & Certs);

	/*! \brief This function returns the user DN.
	 *  \param user_id [IN] The web user id.
	 *  \return The DN on success, NULL on failure.
	 */
	X509_NAME * WebUserGetDn(unsigned long user_id);

	/*! \brief This function changes the password of a web user.
	 *  \param chgPasswd [IN] The password change info.
	 *  \param check_password [IN] The old password should be checked.
	 *  \return true on success, false on failure.
	 */
	bool WebUserChangePassword(const WebuserChangePasswd & chgPasswd, bool check_password);

	/*! \brief This function gets the list of web users.
	 *  \param Users [OUT] The users list.
	 *  \param index [IN] The index for the enumeration.
	 *  \param num [IN] The maximum number of entries to return.
	 *  \param state [IN] The users state.
	 *  \param filter [IN] The a string to search for in the users' dn.
	 *  \return true on success, false on failure.
	 */
	bool EnumWebUsers(mVector<NewpkiEeUser> & Users, long index, long num, long state, const mString & filter);

	/*! \brief This function deletes the PKCS12 associated with a certificate.
	 *  \param id [IN] The certificate's internal Id.
	 *  \return true on success, false on failure.
	 */
	bool DeletePKCS12(unsigned long id);
private:
	#define EESTORE_DN_VAL_TABLE		"dn_validations"
	#define EESTORE_USERS_TABLE			"webusers"
	#define EESTORE_CERTS_TABLE			"webuserscerts"
	#define EESTORE_CREATE_1			"create table "EESTORE_DN_VAL_TABLE" (dn_val_id INT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY, id VARCHAR(41) NOT NULL UNIQUE, ra_id INT UNSIGNED NOT NULL, dn LONGBLOB NOT NULL, email TEXT NOT NULL, lastmail INT UNSIGNED NOT NULL, mailcount INT UNSIGNED NOT NULL, INDEX(id));"
	#define EESTORE_CREATE_2			"create table "EESTORE_USERS_TABLE" (user_id INT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY, email CHAR(250) NOT NULL UNIQUE, password TEXT NOT NULL, dn LONGBLOB NOT NULL, dn_string LONGBLOB NOT NULL, dn_hash VARCHAR(41) NOT NULL, ra_validated INT NOT NULL, activated INT NOT NULL, otp TEXT NOT NULL, last_send INT UNSIGNED NOT NULL, send_count INT UNSIGNED NOT NULL, INDEX(send_count), INDEX(last_send));"
	#define EESTORE_CREATE_3			"create table "EESTORE_CERTS_TABLE" (id INT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY, ra_id INT UNSIGNED NOT NULL, user_id INT UNSIGNED NOT NULL, ca_name BLOB NOT NULL, status INT UNSIGNED NOT NULL, type INT UNSIGNED NOT NULL, error LONGBLOB NOT NULL, serial INT UNSIGNED NOT NULL, p12 LONGBLOB NOT NULL, x509 LONGBLOB NOT NULL, p7b LONGBLOB NOT NULL, INDEX (user_id));"

	#define EESTORE_INSERT_USER			"INSERT INTO "EESTORE_USERS_TABLE" (email, password, dn, dn_string, dn_hash, ra_validated, activated, otp, last_send, send_count) VALUES ('%s', '%s', '%s', '%s', '%s', 0, 0, '%s', '%ld', 1);"
	#define EESTORE_LOGIN_USER			"SELECT user_id, activated, ra_validated, password FROM "EESTORE_USERS_TABLE" WHERE email = '%s';"
	#define EESTORE_GET_USER			"SELECT * FROM "EESTORE_USERS_TABLE" WHERE user_id = '%ld';"
	#define EESTORE_ACTIVATE_USER		"UPDATE "EESTORE_USERS_TABLE" SET activated='1', otp='' WHERE otp='%s';"
	#define EESTORE_DEACTIVATE_USER		"UPDATE "EESTORE_USERS_TABLE" SET activated='0', otp='' WHERE otp='%s';"
	#define EESTORE_DELETE_USER			"DELETE FROM "EESTORE_USERS_TABLE" WHERE user_id='%ld';"
	#define EESTORE_RA_VALIDATED_USER	"UPDATE "EESTORE_USERS_TABLE" SET ra_validated='1' WHERE user_id='%ld';"
	#define EESTORE_GET_USERS_MAIL		"SELECT * FROM "EESTORE_USERS_TABLE" WHERE last_send < '%ld' AND send_count = '%d' AND activated='0';"
	#define EESTORE_UPDATE_USERS_MAIL	"UPDATE "EESTORE_USERS_TABLE" SET last_send='%ld', send_count='%d' WHERE user_id='%ld';"
	#define EESTORE_GET_USER_OTP		"SELECT * FROM "EESTORE_USERS_TABLE" WHERE otp='%s';"
	#define EESTORE_SEARCH_USER_DN		"SELECT user_id FROM "EESTORE_USERS_TABLE" WHERE dn_hash='%s' %s;"
	#define EESTORE_GET_USERS			"SELECT * FROM "EESTORE_USERS_TABLE" %s ORDER BY user_id DESC LIMIT %ld,%ld;"
	#define EESTORE_SET_PASSWORD		"UPDATE "EESTORE_USERS_TABLE" SET password='%s' WHERE user_id='%ld';"
	#define EESTORE_ACTIVATE_USER_ID	"UPDATE "EESTORE_USERS_TABLE" SET activated='1', otp='' WHERE user_id='%ld';"
	
	#define EESTORE_INSERT_DN_VAL		"INSERT INTO "EESTORE_DN_VAL_TABLE" (id, ra_id, dn, email, lastmail, mailcount) VALUES ('%s', '%ld', '%s', '%s', '%d', 1);"
	#define EESTORE_GET_DN_VAL			"SELECT * FROM "EESTORE_DN_VAL_TABLE" WHERE id='%s';"
	#define EESTORE_DEL_DN_VAL			"DELETE FROM "EESTORE_DN_VAL_TABLE" WHERE id='%s';"

	#define EESTORE_INSERT_CERT				"INSERT INTO "EESTORE_CERTS_TABLE" (ra_id, user_id, ca_name, type, status, error, serial, p12, x509, p7b) VALUES ('0', '%ld', '%s', '%d', '%d', '', 0, '', '', '');"
	#define EESTORE_INSERT_CERT_ALL			"INSERT INTO "EESTORE_CERTS_TABLE" (ra_id, user_id, ca_name, type, status, error, serial, p12, x509, p7b) VALUES ('%ld', '%ld', '%s', '%d', '%d', '', %ld, '%s', '%s', '%s');"
	#define EESTORE_GET_CERTS				"SELECT * FROM "EESTORE_CERTS_TABLE" WHERE user_id=%ld ORDER BY id DESC;"
	#define EESTORE_SET_CERT_STATUS			"UPDATE "EESTORE_CERTS_TABLE" SET status=%d WHERE id='%ld';"
	#define EESTORE_SET_CERT_STATUS_RA_ID	"UPDATE "EESTORE_CERTS_TABLE" SET status=%d WHERE ra_id='%ld';"
	#define EESTORE_GET_CERT				"SELECT * FROM "EESTORE_CERTS_TABLE" WHERE id=%ld;"
	#define EESTORE_GET_CERT_BY_RA_ID		"SELECT * FROM "EESTORE_CERTS_TABLE" WHERE ra_id=%ld;"
	#define EESTORE_DELETE_CERT				"DELETE FROM "EESTORE_CERTS_TABLE" WHERE id='%ld';"
	#define EESTORE_SET_CERT_RESP			"UPDATE "EESTORE_CERTS_TABLE" SET ra_id='%ld', status='%d', x509='%s', p7b='%s', error='%s', p12='%s', serial='%ld' WHERE id='%ld';"
	#define EESTORE_GET_CERT_4_REV			"SELECT x509, ca_name, status, FROM "EESTORE_CERTS_TABLE" WHERE id ='%ld'"
	#define EESTORE_UPDATE_FROM_CRL_REV		"UPDATE "EESTORE_CERTS_TABLE" SET x509='', p7b='', p12='', status='%d', serial='0', ra_id='0' WHERE ca_name='%s' AND serial='%ld';"
	#define EESTORE_UPDATE_FROM_CRL_SUSP	"UPDATE "EESTORE_CERTS_TABLE" SET status='%d' WHERE ca_name='%s' AND serial='%ld';"
	#define EESTORE_UPDATE_FROM_CRL_UNSUSP	"UPDATE "EESTORE_CERTS_TABLE" SET status='%d' WHERE ca_name='%s' AND status='%d' AND serial NOT IN (%s);"
	#define EESTORE_UPDATE_NO_SUSP			"UPDATE "EESTORE_CERTS_TABLE" SET status='%d' WHERE ca_name='%s' AND status='%d';"
	#define EESTORE_DELETE_PKCS12			"UPDATE "EESTORE_CERTS_TABLE" SET p12='' WHERE id='%ld';"

	bool SendUserMailNotification(const mString & email, const mString & otp, bool deleteNotification = false);
	bool SendUserMailNotifications(time_t t, int scount);
	bool UpdateFromCRL(const mString & ca_name, const PKI_CRL &Crl);
	bool ProfileExists(const mString & dn_hash, bool & exists, unsigned long & UserId);
	bool WebUserLdapLogin(const mString & email, const mString & password, LdapResult & Result);
	unsigned long WebUserLdapCreate(const mString & email, const LdapResult & Result);
	unsigned long WebUserInsert(const mString & email, const mString & password, const X509_NAME * dn, const mString & otp);


	bool Sql2Cert(SQL *sql, NewpkiProfileDatasCert & Cert, int index);
	bool SqlToNewpkiEeProfile(SQL & sql, long pos, NewpkiEeUser & Profile);

	ClientLDAP m_LdapClient;
	bool m_ldap_auth;
	bool m_ldap_available;
	mString m_ldap_server;
	unsigned long m_ldap_port;
	mString m_ldap_filters;
	mString m_ldap_attr_name;
	mVector<DnSpecs> m_DnSpecs;
	HashTable_String m_Policies;
	CriticalSection ConfAccessLock;
	mString m_siteName;
};

#endif
