/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/



// CA_Handler.h: interface for the CA_Handler class.
//
//////////////////////////////////////////////////////////////////////

#ifndef CA_HANDLER_H
#define CA_HANDLER_H

#include <PKI_CERT.h>
#include <PKI_CRL.h>
#include <CriticalSection.h>
#include "SQL/SQL.h"
#include <ASN1/Asn1Resp.h>
#include "CA_Handler_ASN1.h"
#include "NewPKIStore.h"


/*!
	This class represents one or more CAs
*/
class CA_Handler  
{
public:	
	/*! \brief This is the constructor.
	 *  \param EntityName [IN] The name of the entity.
	 *  \param CaType [IN] Since there is only one table for all the CAs, CaType is used to know on which one we're working.
	 */
	CA_Handler(const mString & EntityName, int CaType);

	/*! \brief This is the destructor.
	 */
	virtual ~CA_Handler();

	/*! \brief This function creates the tables needed to work.
	 *  \param DbConn [IN] A SQL connection.
	 *  \return true on success, false on failure.
	 */
	static bool CreateTables(SQL_Connection * DbConn);

	/*! \brief This function sets the SQL connection.
	 *  \param DbConn [IN] A SQL connection.
	 *  \return true on success, false on failure.
	 */
	bool SetDbConn(const SQL_Connection * DbConn);

	/*! \brief This function generate a private key.
	 *  \param KeyGen [IN] The private key info.
	 *  \param Key [OUT] The private key.
	 *  \param InternalKey [OUT] The private key, in a parsable format.
	 *  \param e [IN] An optional ENGINE.
	 *  \return true on success, false on failure.
	 */
	static bool GEN_PRIVATE_KEY_load(const GenPrivateKey & KeyGen, PKI_RSA & Key, InternalCaKey & InternalKey, ENGINE * e);

	/*! \brief This function sets a private key.
	 *  \param Key [IN] The private key.
	 *  \param InternalKey [OUT] The private key, in a parsable format.
	 *  \return true on success, false on failure.
	 */
	static bool INTERNAL_CA_KEY_set(InternalCaKey & InternalKey, const PKI_RSA & Key);

	/*! \brief This function loads a private key.
	 *  \param ca_key [IN] The private key, in a parsable format.
	 *  \param key [OUT] The private key.
	 *  \param e [IN] An optional ENGINE.
	 *  \return true on success, false on failure.
	 */
	static bool INTERNAL_CA_KEY_load(const InternalCaKey & ca_key, PKI_RSA & key, ENGINE * e);

	/*! \brief This function creates the CA initial datas.
	 *  \param cert [IN] The CA certificate.
	 *  \param privkey [IN] The CA private key.
	 *  \return true on success, false on failure.
	 */
	bool Create(const PKI_CERT & cert, const InternalCaKey & privkey);

	/*! \brief This function loads the intial datas.
	 *  \return true on success, false on failure.
	 */
	bool Load();
	
	/*! \brief This function returns the CA certificate.
	 *  \return The CA certificate, or NULL if CA hasn't been intializaed yet.
	 */
	PKI_CERT * get_CaCert();
	
	/*! \brief This function returns the ENGINE.
	 *  \return The ENGINE, or NULL if ENGINE wasn't set.
	 */
	ENGINE * get_ENGINE();

	/*! \brief This function sets the ENGINE.
	 *  \param e [IN] The ENGINE.
	 */
	void set_ENGINE(ENGINE * e);
	
	/*! \brief This function returns the list of certificates.
	 *  \param Certs [OUT] The certs.
	 *  \param state [IN] The state of certificates to list.
	 *  \param index [IN] The index for the enumeration.
	 *  \param num [IN] The maximum number of entries to return.
	 *  \return The list of certificates, or NULL on failure.
	 */
	bool get_Certs(mVector<InternalCaCert> & Certs, CERT_STATE state=(CERT_STATE)0, long index=0, long num=0);

	/*! \brief This function returns the list of CRLs.
	 *  \param Crls [OUT] The index for the enumeration.
	 *  \param index [IN] The index for the enumeration.
	 *  \param num [IN] The maximum number of entries to return.
	 *  \return The list of CRLs, or NULL on failure.
	 */
	bool get_Crls(mVector<PKI_CRL> & Crls, long index, long num);
	
	/*! \brief This function signs a CSR.
	 *  \param csr [IN] The certificate request.
	 *  \param ResultCert [OUT] The resulting certificate.
	 *  \param uid [IN] The LDAP uid, might be needed for publication.
	 *  \param Days [IN] The number of days for the certificate validity.
	 *  \param Exts [IN] The available extensions for the certificate.
	 *  \param keepCsrExts [IN] Whether to keep the CSR's extensions.
	 *  \param csrExtsOverwrite [IN] Whether to overwrite the CSR's extensions.
	 *  \param check_sig [IN] Whether to check the CSR signature.
	 *  \return true on success, false on failure.
	 */
	bool Sign(const PKI_CSR & csr, PKI_CERT & ResultCert, const mString & uid, int Days, const HashTable_String *Exts, bool keepCsrExts, bool csrExtsOverwrite, bool check_sig);

	/*! \brief This function revokes a certificate.
	 *  \param serial [IN] The serial of the certificate to revoke.
	 *  \param Cert [OUT] The revoked certificate.
	 *  \param CertStatus [OUT] The status of the certificate.
	 *  \param rev_date [OUT] The revocation date.
	 *  \param uid [OUT] The LDAP uid.
	 *  \return true on success, false on failure.
	 */
	bool Revoke(unsigned long serial, PKI_CERT & Cert, CERT_STATE & CertStatus, time_t & rev_date, mString & uid);

	/*! \brief This function suspends a certificate.
	 *  \param serial [IN] The serial of the certificate to suspend.
	 *  \param Cert [OUT] The suspended certificate.
	 *  \param CertStatus [OUT] The status of the certificate.
	 *  \param susp_date [OUT] The suspension date.
	 *  \param uid [OUT] The LDAP uid.
	 *  \return true on success, false on failure.
	 */
	bool Suspend(unsigned long serial, PKI_CERT & Cert, CERT_STATE & CertStatus, time_t & susp_date, mString & uid);

	/*! \brief This function unsuspends a certificate.
	 *  \param serial [IN] The serial of the certificate to unsuspend.
	 *  \param Cert [OUT] The unsuspended certificate.
	 *  \param CertStatus [OUT] The status of the certificate.
	 *  \param uid [OUT] The LDAP uid.
	 *  \return true on success, false on failure.
	 */
	bool Unsuspend(unsigned long serial, PKI_CERT & Cert, CERT_STATE & CertStatus, mString & uid);

	/*! \brief This function generates a CRL.
	 *  \param resCrl [OUT] The new CRL.
	 *  \param Days [IN] The number of days for validity.
	 *  \param Hours [IN] The number of hours for validity.
	 *  \param Exts [IN] The extensions to include to the CRL.
	 *  \param md [IN] The hash algorithm to use for the signature.
	 *  \param InsertCRL [IN] Should the CRL be inserted.
	 *  \return true on success, false on failure.
	 */
	bool Generate_CRL(PKI_CRL & resCrl, int Days, int Hours, const HashTable_String * Exts, const char * md, bool InsertCRL = true);

	/*! \brief This function retieves the last CRL.
	 *  \param crl [OUT] The last CRL.
	 *  \return true on success, false on failure.
	 */
	bool GetLastCrl(PKI_CRL & crl);

	/*! \brief This function upgrades the entity.
	 *  \param Version [IN] The current version.
	 *  \param DbConn [IN] A SQL connection.
	 *  \return true on success, false on failure.
	 */
	static bool Upgrade(const char * Version, SQL_Connection* DbConn);

	/*! \brief This function return 1 when the CA is operable and 0 when not.
	 *  \return return 1 when the CA is operable and 0 when not.
	 */
	operator int();
private:
	#define CA_HANDLER_CERTS_TABLE			"internal_certs"
	#define CA_HANDLER_CREATE_1				"create table internal_cas (ca_type INT UNSIGNED NOT NULL PRIMARY KEY, datas LONGBLOB NOT NULL);"
	#define CA_HANDLER_CREATE_2				"create table "CA_HANDLER_CERTS_TABLE" (unique_id INT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY, ca_type INT UNSIGNED NOT NULL, serial INT UNSIGNED NOT NULL, state INT UNSIGNED NOT NULL, dn_hash VARCHAR(41) NOT NULL, dn BLOB NOT NULL, uid BLOB NOT NULL, begin_date INT UNSIGNED NOT NULL, end_date INT UNSIGNED NOT NULL, rev_date INT UNSIGNED NOT NULL, susp_date INT UNSIGNED NOT NULL, cert LONGBLOB NOT NULL, signature LONGBLOB NOT NULL, INDEX (serial), INDEX(state), INDEX(begin_date), INDEX(dn_hash));"
	#define CA_HANDLER_CREATE_3				"create table internal_crls (unique_id INT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY, ca_type INT UNSIGNED NOT NULL, crl LONGBLOB NOT NULL);"

	#define CA_HANDLER_INSERT_CA			"REPLACE INTO internal_cas (ca_type, datas) VALUES (%d, '%s');"
	#define CA_HANDLER_GET_CA				"SELECT * FROM internal_cas WHERE ca_type=%d;"
	
	#define CA_HANDLER_INSERT_CERT			"INSERT INTO "CA_HANDLER_CERTS_TABLE" (ca_type, serial, state, dn_hash, dn, uid, begin_date, end_date, rev_date, cert, signature) VALUES (%d, %ld, %d, '%s', '%s', '%s', %ld, %ld, 0, '%s', '%s');"
	#define CA_HANDLER_GET_CERT				"SELECT * FROM "CA_HANDLER_CERTS_TABLE" WHERE serial=%ld AND ca_type=%d;"
	#define CA_HANDLER_GET_CERT_UNIQUE		"SELECT serial FROM "CA_HANDLER_CERTS_TABLE" WHERE dn_hash='%s' AND state=%d AND end_date > %ld AND ca_type=%d;"
	#define CA_HANDLER_GET_CERT_BY_STATE	"SELECT serial, rev_date, susp_date FROM "CA_HANDLER_CERTS_TABLE" WHERE state=%d AND ca_type=%d;"
	#define CA_HANDLER_REV_CERT				"UPDATE "CA_HANDLER_CERTS_TABLE" SET state=%d, signature='%s', rev_date=%ld WHERE serial=%ld AND ca_type=%d;"
	#define CA_HANDLER_SUSP_CERT			"UPDATE "CA_HANDLER_CERTS_TABLE" SET state=%d, signature='%s', susp_date=%ld WHERE serial=%ld AND ca_type=%d;"
	#define CA_HANDLER_UNSUSP_CERT			"UPDATE "CA_HANDLER_CERTS_TABLE" SET state=%d, signature='%s', susp_date=0 WHERE serial=%ld AND ca_type=%d;"
	#define CA_HANDLER_GET_CERTS			"SELECT * FROM "CA_HANDLER_CERTS_TABLE" WHERE ca_type=%d ORDER BY begin_date DESC"
	#define CA_HANDLER_GET_CERTS_BY_STATE	"SELECT * FROM "CA_HANDLER_CERTS_TABLE" WHERE state=%d AND ca_type=%d ORDER BY begin_date DESC"
	
	#define CA_HANDLER_INSERT_CRL			"INSERT INTO internal_crls (ca_type, crl) VALUES (%d, '%s');"
	#define CA_HANDLER_GET_CRLS				"SELECT * FROM internal_crls WHERE ca_type=%d ORDER BY unique_id DESC"

	bool Flush();
	void Reset();
	void CleanMemory();
	bool get_InternalCertSig(const InternalCaCert & cert, mString & sig_pem);
	bool verify_InternalCertSig(const InternalCaCert & cert, const char * sig_pem);
	bool get_Cert(unsigned long serial, InternalCaCert & cert);
	bool SQL2InternalCert(SQL * sql, long i, InternalCaCert & cert, bool checkSig = true);
	bool Private_Upgrade(const char * Version);
	
	CriticalSection CaLock;
	
	PKI_CERT m_CaCert;
	PKI_RSA m_CaKey;
	InternalCaBody m_InternCaBody;
	InternalCa m_InternCa;
	ENGINE * m_Engine;
	SQL_Connection * m_DbConn;
	int m_CaType;
	mString m_EntityName;
	static const char * m_Version;
};

#endif
