/*
 	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com).
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


#include "Asn1Helper.h"
#include <PKI_ERR.h>
#include <openssl/asn1t.h>

ASN1_SEQUENCE(ASN1_ENCRYPT_SIGN) = {
	ASN1_SIMPLE(ASN1_ENCRYPT_SIGN, sym_algo, ASN1_OBJECT),
	ASN1_SIMPLE(ASN1_ENCRYPT_SIGN, crypteddatas, ASN1_OCTET_STRING),
	ASN1_SIMPLE(ASN1_ENCRYPT_SIGN, sessionkey, ASN1_OCTET_STRING),
	ASN1_SIMPLE(ASN1_ENCRYPT_SIGN, sig, X509_SIG),
}ASN1_SEQUENCE_END(ASN1_ENCRYPT_SIGN)
Asn1EncryptSign Asn1EncryptSign::EmptyInstance;
bool Asn1EncryptSign::set_crypteddatas(const Asn1OctetString & c_crypteddatas)
{
	m_crypteddatas = c_crypteddatas;
	return true;
}

const Asn1OctetString & Asn1EncryptSign::get_crypteddatas() const
{
	return m_crypteddatas;
}

Asn1OctetString & Asn1EncryptSign::get_crypteddatas()
{
	return m_crypteddatas;
}

bool Asn1EncryptSign::set_sessionkey(const Asn1OctetString & c_sessionkey)
{
	m_sessionkey = c_sessionkey;
	return true;
}

const Asn1OctetString & Asn1EncryptSign::get_sessionkey() const
{
	return m_sessionkey;
}

Asn1OctetString & Asn1EncryptSign::get_sessionkey()
{
	return m_sessionkey;
}

bool Asn1EncryptSign::set_sig(const X509_SIG * c_sig)
{
	if(m_sig)
		ASN1_item_free((ASN1_VALUE*)m_sig, ASN1_ITEM_rptr(X509_SIG));
	m_sig = (X509_SIG*)ASN1_item_dup(ASN1_ITEM_rptr(X509_SIG), (void*)c_sig);
	if(!m_sig)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

const X509_SIG * Asn1EncryptSign::get_sig() const
{
	if(!m_sig)
		((Asn1EncryptSign*)this)->m_sig = (X509_SIG*)ASN1_item_new(ASN1_ITEM_rptr(X509_SIG));
	return m_sig;
}

X509_SIG * Asn1EncryptSign::get_sig()
{
	if(!m_sig)
		m_sig = (X509_SIG*)ASN1_item_new(ASN1_ITEM_rptr(X509_SIG));
	return m_sig;
}

bool Asn1EncryptSign::set_symAlgo(const ASN1_OBJECT * c_symAlgo)
{
	if(m_symAlgo)
		ASN1_item_free((ASN1_VALUE*)m_symAlgo, ASN1_ITEM_rptr(ASN1_OBJECT));
	m_symAlgo = (ASN1_OBJECT*)ASN1_item_dup(ASN1_ITEM_rptr(ASN1_OBJECT), (void*)c_symAlgo);
	if(!m_symAlgo)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

const ASN1_OBJECT * Asn1EncryptSign::get_symAlgo() const
{
	if(!m_symAlgo)
		((Asn1EncryptSign*)this)->m_symAlgo = (ASN1_OBJECT*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_OBJECT));
	return m_symAlgo;
}

ASN1_OBJECT * Asn1EncryptSign::get_symAlgo()
{
	if(!m_symAlgo)
		m_symAlgo = (ASN1_OBJECT*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_OBJECT));
	return m_symAlgo;
}

bool Asn1EncryptSign::to_PEM(mString & PemDatas) const
{
	ASN1_ENCRYPT_SIGN * c_localvar = NULL;
	if(!give_Datas(&c_localvar))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!Private_toPEM("NEWPKI SIGNED ENCRYPTED DATAS", get_ASN1_ITEM(), (ASN1_VALUE*)c_localvar, PemDatas))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
		ASN1_item_free((ASN1_VALUE*)c_localvar, get_ASN1_ITEM());
	}
	ASN1_item_free((ASN1_VALUE*)c_localvar, get_ASN1_ITEM());
	return true;
}

bool Asn1EncryptSign::from_PEM(const mString & PemDatas)
{
	ASN1_ENCRYPT_SIGN * c_localvar = NULL;
	if(!Private_fromPEM("NEWPKI SIGNED ENCRYPTED DATAS", get_ASN1_ITEM(), (ASN1_VALUE**)&c_localvar, PemDatas))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!load_Datas(c_localvar))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		ASN1_item_free((ASN1_VALUE*)c_localvar, get_ASN1_ITEM());
		return false;
	}
	ASN1_item_free((ASN1_VALUE*)c_localvar, get_ASN1_ITEM());
	return true;
}

Asn1EncryptSign::Asn1EncryptSign():NewPKIObject()
{
	resetAll();
}

Asn1EncryptSign::Asn1EncryptSign(const Asn1EncryptSign & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

Asn1EncryptSign::~Asn1EncryptSign()
{
	Clear();
}

void Asn1EncryptSign::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void Asn1EncryptSign::freeAll()
{
	if(m_sig)
	{
		ASN1_item_free((ASN1_VALUE*)m_sig, ASN1_ITEM_rptr(X509_SIG));
	}
	if(m_symAlgo)
	{
		ASN1_item_free((ASN1_VALUE*)m_symAlgo, ASN1_ITEM_rptr(ASN1_OBJECT));
	}
}

void Asn1EncryptSign::resetAll()
{
	m_crypteddatas.Clear();
	m_sessionkey.Clear();
	m_sig = NULL;
	m_symAlgo = NULL;
}

bool Asn1EncryptSign::load_Datas(const ASN1_ENCRYPT_SIGN * Datas)
{
	Clear();
	if(Datas->crypteddatas)
	{
		if(!m_crypteddatas.load_Datas(Datas->crypteddatas))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(Datas->sessionkey)
	{
		if(!m_sessionkey.load_Datas(Datas->sessionkey))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(Datas->sig)
	{
		if(m_sig)
			ASN1_item_free((ASN1_VALUE*)m_sig, ASN1_ITEM_rptr(X509_SIG));
		m_sig = (X509_SIG*)ASN1_item_dup(ASN1_ITEM_rptr(X509_SIG), Datas->sig);
		if(!m_sig)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
	}
	if(Datas->sym_algo)
	{
		if(m_symAlgo)
			ASN1_item_free((ASN1_VALUE*)m_symAlgo, ASN1_ITEM_rptr(ASN1_OBJECT));
		m_symAlgo = (ASN1_OBJECT*)ASN1_item_dup(ASN1_ITEM_rptr(ASN1_OBJECT), Datas->sym_algo);
		if(!m_symAlgo)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
	}
	m_isOk=true;
	return true;
}

bool Asn1EncryptSign::give_Datas(ASN1_ENCRYPT_SIGN ** Datas) const
{
	if(!(*Datas) && !(*Datas = (ASN1_ENCRYPT_SIGN*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->crypteddatas && !((*Datas)->crypteddatas = (ASN1_OCTET_STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_OCTET_STRING))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_crypteddatas.give_Datas(&(*Datas)->crypteddatas))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->crypteddatas, ASN1_ITEM_rptr(ASN1_OCTET_STRING));
		(*Datas)->crypteddatas = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->sessionkey && !((*Datas)->sessionkey = (ASN1_OCTET_STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_OCTET_STRING))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_sessionkey.give_Datas(&(*Datas)->sessionkey))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->sessionkey, ASN1_ITEM_rptr(ASN1_OCTET_STRING));
		(*Datas)->sessionkey = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(m_sig)
	{
		if((*Datas)->sig)
			ASN1_item_free((ASN1_VALUE*)(*Datas)->sig, ASN1_ITEM_rptr(X509_SIG));
		if(!((*Datas)->sig = (X509_SIG*)ASN1_item_dup(ASN1_ITEM_rptr(X509_SIG), (ASN1_VALUE*)m_sig)))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
	}
	else
	{
		if(!(*Datas)->sig)
		{
			(*Datas)->sig = (X509_SIG*)ASN1_item_new(ASN1_ITEM_rptr(X509_SIG));
			if(!(*Datas)->sig)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
		}
	}
	if(m_symAlgo)
	{
		if((*Datas)->sym_algo)
			ASN1_item_free((ASN1_VALUE*)(*Datas)->sym_algo, ASN1_ITEM_rptr(ASN1_OBJECT));
		if(!((*Datas)->sym_algo = (ASN1_OBJECT*)ASN1_item_dup(ASN1_ITEM_rptr(ASN1_OBJECT), (ASN1_VALUE*)m_symAlgo)))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
	}
	else
	{
		if(!(*Datas)->sym_algo)
		{
			(*Datas)->sym_algo = (ASN1_OBJECT*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_OBJECT));
			if(!(*Datas)->sym_algo)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
		}
	}
	return true;
}

bool Asn1EncryptSign::operator=(const Asn1EncryptSign & other)
{
	Clear();
	m_crypteddatas = other.m_crypteddatas;
	m_sessionkey = other.m_sessionkey;
	if(other.m_sig)
	{
		if(m_sig)
			ASN1_item_free((ASN1_VALUE*)m_sig, ASN1_ITEM_rptr(X509_SIG));
		m_sig = (X509_SIG*)ASN1_item_dup(ASN1_ITEM_rptr(X509_SIG), (void*)other.m_sig);
		if(!m_sig)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(other.m_symAlgo)
	{
		if(m_symAlgo)
			ASN1_item_free((ASN1_VALUE*)m_symAlgo, ASN1_ITEM_rptr(ASN1_OBJECT));
		m_symAlgo = (ASN1_OBJECT*)ASN1_item_dup(ASN1_ITEM_rptr(ASN1_OBJECT), (void*)other.m_symAlgo);
		if(!m_symAlgo)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_isOk=true;
	return true;
}



const ASN1_ITEM * Asn1EncryptSign::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(ASN1_ENCRYPT_SIGN);
}

Asn1OctetString Asn1OctetString::EmptyInstance;

Asn1OctetString::Asn1OctetString():mBuffer()
{
}

Asn1OctetString::Asn1OctetString(const Asn1OctetString & other):mBuffer(other)
{
}

Asn1OctetString::~Asn1OctetString()
{
}

bool Asn1OctetString::load_Datas(const ASN1_OCTET_STRING *Datas)
{
	if(!Copy(Datas->data, Datas->length))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool Asn1OctetString::give_Datas(ASN1_OCTET_STRING **Datas) const
{
	if(!(*Datas) && !(*Datas = ASN1_OCTET_STRING_new()))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(isOK())
	{
		if(ASN1_OCTET_STRING_set(*Datas, (unsigned char *)get_Buffer(), get_BufferLen()) <= 0)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	return true;
}





#define CIPHER_IV (unsigned char*)"NEWPKI_IV"

bool Private_toSignEncrypt(Asn1EncryptSign & cryptinfo, const ASN1_ITEM * it, ASN1_VALUE * a, const EVP_PKEY * sig_pkey, const EVP_PKEY * crypt_pkey, const EVP_MD * sig_md, const EVP_CIPHER * crypt_cypher)
{
	unsigned char rnd_datas[50];
	unsigned char * c_key;
	int c_key_len;
	unsigned char key[EVP_MAX_KEY_LENGTH];
	int keylen;
	EVP_CIPHER_CTX ctx;
	unsigned char * der_body;
	unsigned char * p;
	int der_body_len;
	int der_body_len2;
	unsigned char * crypted_body;
	int crypted_body_len;
	unsigned char iv[EVP_MAX_IV_LENGTH+1];
	strcpy((char*)iv, (char*)CIPHER_IV);
	ASN1_OCTET_STRING * datas;

	if(!it || !a || !sig_pkey || !crypt_pkey || !sig_md || !crypt_cypher)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}
	if(!cryptinfo.set_symAlgo(OBJ_nid2obj(EVP_CIPHER_nid(crypt_cypher))))
	{
		ASN1err(ASN1_F_ASN1_SIGN,ASN1_R_UNKNOWN_OBJECT_TYPE);
		return false;
	}

	//generate session key, that will encrypt the body
	RAND_bytes(rnd_datas, sizeof(rnd_datas));
	if( (keylen = EVP_BytesToKey(crypt_cypher, sig_md, NULL, rnd_datas, sizeof(rnd_datas), 1, key, iv)) <= 0 )
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	//Not needed anymore
	OPENSSL_cleanse(rnd_datas, sizeof(rnd_datas));

	//Encrypt session key with public key
	c_key = (unsigned char*)malloc(EVP_PKEY_size((EVP_PKEY*)crypt_pkey));
	if(!c_key)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if( (c_key_len = EVP_PKEY_encrypt(c_key, key, keylen, (EVP_PKEY*)crypt_pkey)) <= 0 )
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		free(c_key);
		return false;
	}
	cryptinfo.get_sessionkey().Assign(c_key, c_key_len);


	//get the body in DER
	der_body_len = ASN1_item_i2d(a, NULL, it);
	if(der_body_len <= 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	der_body = (unsigned char*)malloc(der_body_len);
	if(!der_body)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	p = der_body;

	der_body_len = ASN1_item_i2d(a, &p, it);
	if(der_body_len <= 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		free(der_body);
		return false;
	}

	//Allocate decryption buffer
	crypted_body = (unsigned char *)malloc(keylen + der_body_len + 10);
	if(!crypted_body)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		free(der_body);
		return false;
	}


	//Do the encryption
	EVP_CIPHER_CTX_init(&ctx);
	if(EVP_EncryptInit(&ctx, crypt_cypher, key, CIPHER_IV) <= 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		free(der_body);
		free(crypted_body);
		EVP_CIPHER_CTX_cleanup(&ctx);
		return false;
	}
	if(EVP_EncryptUpdate(&ctx, crypted_body, &crypted_body_len, der_body, der_body_len) <= 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		free(der_body);
		free(crypted_body);
		EVP_CIPHER_CTX_cleanup(&ctx);
		return false;
	}
	if(EVP_EncryptFinal(&ctx, &crypted_body[crypted_body_len], &der_body_len2) <= 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		free(der_body);
		free(crypted_body);
		EVP_CIPHER_CTX_cleanup(&ctx);
		return false;
	}
	crypted_body_len += der_body_len2;
	EVP_CIPHER_CTX_cleanup(&ctx);
	free(der_body);
	OPENSSL_cleanse(key, keylen);
	
	cryptinfo.get_crypteddatas().Assign(crypted_body, crypted_body_len);

	datas = NULL;
	if(!cryptinfo.get_crypteddatas().give_Datas(&datas))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	//Do signature
	if(ASN1_sign((int (*)())i2d_ASN1_OCTET_STRING, cryptinfo.get_sig()->algor, NULL, cryptinfo.get_sig()->digest, (char*)datas, (EVP_PKEY*)sig_pkey, sig_md) <= 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		ASN1_OCTET_STRING_free(datas);
		return false;
	}
	ASN1_OCTET_STRING_free(datas);
	return true;
}


bool Private_fromSignEncrypt(const Asn1EncryptSign & cryptinfo, const ASN1_ITEM * it, ASN1_VALUE ** a, const EVP_PKEY * sig_pkey, const EVP_PKEY * crypt_pkey)
{
	const EVP_MD * sig_md;
	const EVP_CIPHER * crypt_cypher;
	unsigned char * c_key;
	unsigned char key[EVP_MAX_KEY_LENGTH];
	int keylen;
	EVP_CIPHER_CTX ctx;
	unsigned char * der_body;
	unsigned char * p;
	int der_body_len;
	int der_body_len2;
	unsigned char iv[EVP_MAX_IV_LENGTH+1];
	ASN1_OCTET_STRING * datas;
	strcpy((char*)iv, (char*)CIPHER_IV);
	
	if(!sig_pkey)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}

	sig_md = EVP_get_digestbyobj(cryptinfo.get_sig()->algor->algorithm);
	if(!sig_md)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	crypt_cypher = EVP_get_cipherbyobj(cryptinfo.get_symAlgo());
	if(!crypt_cypher)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}

	datas = NULL;
	if(!cryptinfo.get_crypteddatas().give_Datas(&datas))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	//We can verify the signature
	if (ASN1_verify((int(*)())i2d_ASN1_OCTET_STRING, cryptinfo.get_sig()->algor, cryptinfo.get_sig()->digest, (char*)datas, (EVP_PKEY*)sig_pkey) <= 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		ASN1_OCTET_STRING_free(datas);
		return false;
	}
	ASN1_OCTET_STRING_free(datas);

	//We can be asked to only verify the datas
	if(!crypt_pkey)
	{
		return true;
	}

	//We now decrypt the session key
	c_key = (unsigned char*)malloc(EVP_PKEY_size((EVP_PKEY*)crypt_pkey)+30);
	if(!c_key)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if( (keylen = EVP_PKEY_decrypt(c_key, (unsigned char*)cryptinfo.get_sessionkey().get_Buffer(), cryptinfo.get_sessionkey().get_BufferLen(), (EVP_PKEY*)crypt_pkey)) <= 0 )
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		free(c_key);
		return false;
	}

	if(keylen > (int)sizeof(key))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		free(c_key);
		return false;
	}
	memcpy(key, c_key, keylen);
	OPENSSL_cleanse(c_key, keylen);
	free(c_key);

	//With the session key we can decrypt the body !

	//Allocate decryption buffer
	der_body = (unsigned char *)malloc(EVP_MAX_KEY_LENGTH + cryptinfo.get_crypteddatas().get_BufferLen() + 50);
	if(!der_body)
	{
		OPENSSL_cleanse(key, keylen);
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}


	//Do the decryption
	EVP_CIPHER_CTX_init(&ctx);
	if(EVP_DecryptInit(&ctx, crypt_cypher, key, CIPHER_IV) <= 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		EVP_CIPHER_CTX_cleanup(&ctx);
		free(der_body);
		OPENSSL_cleanse(key, keylen);
		return false;
	}
	if(EVP_DecryptUpdate(&ctx, der_body, &der_body_len, cryptinfo.get_crypteddatas().get_Buffer(), cryptinfo.get_crypteddatas().get_BufferLen()) <= 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		EVP_CIPHER_CTX_cleanup(&ctx);
		free(der_body);
		OPENSSL_cleanse(key, keylen);
		return false;
	}
	if(EVP_DecryptFinal(&ctx, &der_body[der_body_len], &der_body_len2) <= 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		EVP_CIPHER_CTX_cleanup(&ctx);
		free(der_body);
		OPENSSL_cleanse(key, keylen);
		return false;
	}
	der_body_len += der_body_len2;
	EVP_CIPHER_CTX_cleanup(&ctx);
	OPENSSL_cleanse(key, keylen);
	
	p = der_body;
	*a = ASN1_item_d2i(NULL, &p, der_body_len, it);
	if(!*a)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		free(der_body);
		return false;
	}
	free(der_body);

	return true;
}

