/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// DlgSearchLDAP.cpp: implementation of the DlgSearchLDAP class.
//
//////////////////////////////////////////////////////////////////////

#include "DlgSearchLDAP.h"
#include "dlgs_wdr.h"
#include "clintl.h"

#ifndef _WIN32
	#define stricmp strcasecmp

	#define strupr(x) \
	{ \
		int dtlen; \
		for(dtlen=0; dtlen<strlen(x); dtlen++) x[dtlen]=toupper(x[dtlen]); \
	}

	#define strlwr(x) \
	{ \
		int dtlen; \
		for(dtlen=0; dtlen<strlen(x); dtlen++) x[dtlen]=tolower(x[dtlen]); \
	}
#endif


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

BEGIN_EVENT_TABLE(DlgSearchLDAP, wxDialog)
    EVT_BUTTON(IDC_OK, DlgSearchLDAP::OnOk)
    EVT_BUTTON(IDC_CANCEL, DlgSearchLDAP::OnCancel)
	EVT_BUTTON(IDC_ADD, DlgSearchLDAP::OnAdd)
    EVT_BUTTON(IDC_SEARCH, DlgSearchLDAP::OnSearch)
    EVT_BUTTON(IDC_DEL, DlgSearchLDAP::OnDelete)
END_EVENT_TABLE()


DlgSearchLDAP::DlgSearchLDAP(wxWindow * wParent, PkiClient * ClientPki):wxDialog(wParent, (wxWindowID)-1, _("LDAP search"), wxDefaultPosition)
{
	m_ClientPki = ClientPki;
	m_IsOK = false;
	DlgSearchLDAP_SetWindow(this);
	m_listCtrl = ((wxListCtrl *)FindWindow(IDC_LIST));


	//Intialisation de la listview
	m_imageListSmall = new wxImageList(16, 16, TRUE);
	wxIcon ico;
	ico.CopyFromBitmap(DlgSearchLDAP_GetBitmap(0));
	m_imageListSmall->Add( ico );

	
	m_listCtrl->SetImageList(m_imageListSmall, wxIMAGE_LIST_SMALL);
	m_listCtrl->InsertColumn(0, _("Name"), wxLIST_FORMAT_LEFT, 160);
	m_listCtrl->InsertColumn(1, _("Email address"), wxLIST_FORMAT_LEFT, 150);

	int nItemCN;
	int nItemSN;
	int nItemG;
	int nItemE;
	int nItemO;

	// Initialize the list of objects
	nItemCN = ((wxComboBox *)FindWindow(IDC_OBJECT))->GetCount();
	((wxComboBox *)FindWindow(IDC_OBJECT))->Append(_("Name"), (void*)"cn");
	
	nItemSN = ((wxComboBox *)FindWindow(IDC_OBJECT))->GetCount();
	((wxComboBox *)FindWindow(IDC_OBJECT))->Append(_("Last name"), (void*)"sn");
	
	nItemG = ((wxComboBox *)FindWindow(IDC_OBJECT))->GetCount();
	((wxComboBox *)FindWindow(IDC_OBJECT))->Append(_("First name"), (void*)"givenName");

	nItemE = ((wxComboBox *)FindWindow(IDC_OBJECT))->GetCount();
	((wxComboBox *)FindWindow(IDC_OBJECT))->Append(_("Email address"), (void*)"mail");
	
	nItemO = ((wxComboBox *)FindWindow(IDC_OBJECT))->GetCount();
	((wxComboBox *)FindWindow(IDC_OBJECT))->Append(_("Organisation"), (void*)"o");
	
	
	// Initialize the list of flags
	((wxComboBox *)FindWindow(IDC_CRIT))->Append(_("contains"), (void*)"(%s=*%s*)");
	((wxComboBox *)FindWindow(IDC_CRIT))->Append(_("is"), (void*)"(%s=%s)");
	((wxComboBox *)FindWindow(IDC_CRIT))->Append(_("starts with"), (void*)"(%s=%s*)");
	((wxComboBox *)FindWindow(IDC_CRIT))->Append(_("ends with"), (void*)"(%s=*%s)");
	((wxComboBox *)FindWindow(IDC_CRIT))->Append(_("sounds like"), (void*)"(%s~=%s)");

	((wxComboBox *)FindWindow(IDC_CRIT))->SetSelection(0);

	((wxComboBox *)FindWindow(IDC_OBJECT))->SetSelection(nItemCN);
	
	CenterOnScreen();
	ShowModal();
}

DlgSearchLDAP::~DlgSearchLDAP()
{
	int nItem;
	int i;
	char * Entry;

	nItem = ((wxListBox *)FindWindow(IDC_LST_SEARCH))->GetCount();

	for(i=0; i<nItem; i++)
	{
		Entry = (char *)((wxListBox *)FindWindow(IDC_LST_SEARCH))->GetClientData(i);
		if(!Entry) continue;
		free(Entry);
	}

	ClearList();

	if(m_imageListSmall) delete m_imageListSmall;
}

const HashTable_Dn & DlgSearchLDAP::GetDN()
{
	return mDn;
}


const mString & DlgSearchLDAP::GetUID()
{
	return m_strUID;
}

void DlgSearchLDAP::OnOk(wxCommandEvent &event)
{
	LDAP_ENTRY * Entry;
	long SelectedItem;

	SelectedItem = m_listCtrl->GetNextItem(-1, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
	if(SelectedItem == -1) return;
	Entry = (LDAP_ENTRY *)m_listCtrl->GetItemData(SelectedItem);

	if(Entry->Dn) mDn = *(Entry->Dn);
	if(Entry->strDn)
		m_strDn = Entry->strDn;
	if(Entry->strUid)
		m_strUID = Entry->strUid;

	m_IsOK = true;
	Close(TRUE);
}

void DlgSearchLDAP::OnAdd(wxCommandEvent &event)
{
	int nItem;
	char * Obj;
	char * Crit;
	wxString Value;
	char AddValue[250];
	
	char strAddValue[250];
	wxString strObj;
	wxString strCrit;

	Value = ((wxTextCtrl *)FindWindow(IDC_VALUE))->GetValue();
	if(!Value.Len()) return;

	// The LDAP way !
	nItem = ((wxComboBox *)FindWindow(IDC_OBJECT))->GetSelection();	
	Obj = (char*)((wxComboBox *)FindWindow(IDC_OBJECT))->GetClientData(nItem);
	if(nItem == -1) return;

	nItem = ((wxComboBox *)FindWindow(IDC_CRIT))->GetSelection();	
	Crit = (char*)((wxComboBox *)FindWindow(IDC_CRIT))->GetClientData(nItem);
	if(nItem == -1) return;

	snprintf(AddValue, sizeof(AddValue), Crit, Obj, Value.GetData());


	// The string way !
	strObj = ((wxComboBox *)FindWindow(IDC_OBJECT))->GetValue();
	strCrit = ((wxComboBox *)FindWindow(IDC_CRIT))->GetValue();
	snprintf(strAddValue, sizeof(strAddValue), "%s %s %s", strObj.GetData(), strCrit.GetData(), Value.GetData());

	// We add it to the list
	((wxListBox *)FindWindow(IDC_LST_SEARCH))->Append(strAddValue, (void*)strdup(AddValue));

	((wxTextCtrl *)FindWindow(IDC_VALUE))->SetValue("");
}

void DlgSearchLDAP::OnSearch(wxCommandEvent &event)
{
	mString m_Value;
	int nItem;
	size_t i;
	char * Entry;
	DlgMessage DlgMsg(this);
	mVector<LdapResult> Results;

	nItem = ((wxListBox *)FindWindow(IDC_LST_SEARCH))->GetCount();
	if(!nItem) return;

	m_Value = "(&";
	for(i=0; i<(size_t)nItem; i++)
	{
		Entry = (char *)((wxListBox *)FindWindow(IDC_LST_SEARCH))->GetClientData(i);
		if(!Entry) continue;
		m_Value += Entry;
	}
	m_Value += ")";

	DlgMsg.wShow(_("Searching LDAP..."));
	if(!m_ClientPki->SearchLdap(m_Value.c_str(), Results))
	{
		DlgMsg.wHide();
		HandleError(m_ClientPki->GetError(), this);
		return;
	}
	DlgMsg.wHide();
	

	ClearList();

	for(i=0; i<Results.size(); i++)
	{
		AddOneLdapEntry(Results[i]);
	}

	if(!m_listCtrl->GetItemCount())
	{
		HandleError(_("Could not find any matching entry."), this);
	}
	Results.clear();
	Enable(TRUE);
}


void DlgSearchLDAP::ClearList()
{
	int nItem;
	int ItemCount;
	LDAP_ENTRY * Entry;

	ItemCount = m_listCtrl->GetItemCount();

	for(nItem=0; nItem<ItemCount; nItem++)
	{
		Entry = (LDAP_ENTRY *)m_listCtrl->GetItemData(nItem);
		if(Entry)
		{
			if(Entry->Dn) delete Entry->Dn;
			if(Entry->strDn) free(Entry->strDn);
			if(Entry->strUid) free(Entry->strUid);
			free(Entry);
		}
	}
	m_listCtrl->DeleteAllItems();
}

void DlgSearchLDAP::OnDelete(wxCommandEvent &event)
{
	int nItem;
	char * Entry;

	nItem = ((wxListBox *)FindWindow(IDC_LST_SEARCH))->GetSelection();
	if(nItem == -1) return;

	Entry = (char *)((wxListBox *)FindWindow(IDC_LST_SEARCH))->GetClientData(nItem);
	((wxListBox *)FindWindow(IDC_LST_SEARCH))->Delete(nItem);
	if(!Entry) return;
	
	free(Entry);
}

void DlgSearchLDAP::AddOneLdapEntry(const LdapResult & currResult)
{
	LDAP_ENTRY * newEntry;
	size_t i;
	int index;
	long pos;
	const char * Name;

	newEntry = (LDAP_ENTRY *)malloc(sizeof(LDAP_ENTRY));
	if(!newEntry)
	{
		return;
	}

	newEntry->Dn = new HashTable_Dn();
	if(!newEntry->Dn)
	{
		free(newEntry);
		return;
	}

	index = m_listCtrl->GetItemCount();


	// The RDN
	newEntry->strDn = strdup(currResult.get_rdn().c_str());
	if(!newEntry->strDn)
	{
		delete newEntry->Dn;
		free(newEntry);
		return;
	}

	// The UID
	if(currResult.get_uid().size())
	{
		newEntry->strUid = strdup(currResult.get_uid().c_str());
		if(!newEntry->strUid)
		{
			delete newEntry->Dn;
			free(newEntry->strDn);
			free(newEntry);
			return;
		}
	}
		
	for(i=0; i<currResult.get_objects().size(); i++)
	{
		newEntry->Dn->Add(currResult.get_objects()[i].get_name().c_str(),
							currResult.get_objects()[i].get_value().c_str());
	}


	pos = newEntry->Dn->SeekEntryName("commonName", -1);
	if( pos == HASHTABLE_NOT_FOUND)
	{
		delete newEntry->Dn;
		free(newEntry->strDn);
		free(newEntry->strUid);
		free(newEntry);
		return;
	}
	Name = newEntry->Dn->Get(pos);
	if(!Name)
	{
		delete newEntry->Dn;
		free(newEntry->strDn);
		free(newEntry->strUid);
		free(newEntry);
		return;
	}


	if(m_listCtrl->InsertItem(index, Name, 0) == -1)
	{
		delete newEntry->Dn;
		free(newEntry->strDn);
		free(newEntry->strUid);
		free(newEntry);
		return;
	}

	if(!m_listCtrl->SetItemData(index, (long)newEntry))
	{
		m_listCtrl->DeleteItem(index);
		delete newEntry->Dn;
		free(newEntry->strDn);
		free(newEntry->strUid);
		free(newEntry);
		return;
	}


	pos = newEntry->Dn->SeekEntryName("emailAddress", -1);
	if( pos != HASHTABLE_NOT_FOUND && (Name = newEntry->Dn->Get(pos)) )
	{
		m_listCtrl->SetItem(index, 1, Name);
	}	
}

bool DlgSearchLDAP::IsOK()
{
	return m_IsOK;
}

