/*
 * Copyright (c) 2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using System;
using System.Runtime.InteropServices;

interface IReadLine {
  ReadLine (prompt : string) : string;
  AddHistory (line : string) : void; 
}

class ConsoleReadLine : IReadLine
{
  public ReadLine (prompt : string) : string
  {
    Console.Write (prompt);    
    Console.ReadLine ()
  }

  public AddHistory (_ : string) : void { }
}

class NativeReadLine : IReadLine
{
  [DllImport ("readline")]
  static extern readline (_ : string) : string;

  [DllImport ("readline")]
  static extern add_history (_ : string) : void;

  [DllImport ("readline")]
  static extern isatty (_ : int) : bool;

  private this () {
  }
  
  public static GetInstance () : IReadLine {
    try {
      _ = isatty (0);
      NativeReadLine ()
    }
    catch {
      | _ is System.EntryPointNotFoundException => ConsoleReadLine ()
      | _ is System.DllNotFoundException => ConsoleReadLine ()
    }
  }

  public ReadLine (prompt : string) : string {
    readline (prompt);
  }

  public AddHistory (line : string) : void {
    add_history (line);
  }
}

#region HACKS
module HacksFramework
{
  public Run () : void {
    // libreadline is broken on my machine, we need to call something
    // in libtermcap.so to initialize it properly
    try { _ = tgetnum (""); } catch { _ => () }    
  }
  
  // helper call for making system load needed libs
  [DllImport ("termcap")]
  extern tgetnum (_ : string) : int;
}
#endregion HACKS
