using System
using Nemerle

struct ColoredChar
  public Char : char
  public Foreground : ConsoleColor
  public Background : ConsoleColor

  public this (ch : Char, 
               fg : ConsoleColor = ConsoleColor.Gray, 
               bg : ConsoleColor = ConsoleColor.Black)
    this.Char = ch
    Foreground = fg
    Background = bg

  [OverrideObjectEquals] \
  public Equals (other : ColoredChar) : bool
    (Char == other.Char 
      && Foreground == other.Foreground
      && Background == other.Background)

module ConsoleBuffer
  mutable current_fg : ConsoleColor = ConsoleColor.Gray
  mutable current_bg : ConsoleColor = ConsoleColor.Black

  mutable cur_x : int
  mutable cur_y : int

  screen : array [2, ColoredChar]
  real_screen : array [2, ColoredChar]

  width : int
  height : int

  public Flush () : void
    for (mutable y = 0; y < height; ++y)
      mutable first_x = -1
      mutable last_x = -1
      
      for (mutable x = 0; x < width; ++x)
        when (! screen [x, y].Equals (real_screen [x, y]))
          when (first_x == -1) first_x = x
          last_x = x

      when (last_x != -1)
        Console.SetCursorPosition (1 + first_x, y + 1)
        while (first_x <= last_x)
          def ch = screen [first_x, y]
          real_screen [first_x, y] = ch
          DoSetColor (ch.Foreground, ch.Background)
          Console.Write (ch.Char)
          first_x++
    
    Console.SetCursorPosition (width - 1, height - 1)

  public Clear () : void
    for (mutable y = 0; y < height; ++y)
      for (mutable x = 0; x < width; ++x)
        screen [x, y] = ColoredChar (' ')

  public Goto (x : int, y : int) : void
    cur_x = x
    cur_y = y

  public DrawChar (ch : ColoredChar) : void
    screen [cur_x, cur_y] = ch
    cur_x++
    when (cur_x == width)
      cur_x = 0
      cur_y++
      when (cur_y == height)
        cur_y = 0

  public DrawString (s : string,
                     fg : ConsoleColor = ConsoleColor.Gray, 
                     bg : ConsoleColor = ConsoleColor.Black) : void
    foreach (ch in s)
      DrawChar (ColoredChar (ch, fg, bg))

  DoSetColor (fg : ConsoleColor, bg : ConsoleColor) : void
    when (fg != current_fg)
      def is_high = fg :> int > ConsoleColor.Gray :> int

      when (!is_high)
        Console.Write ("[0m")
      Console.ForegroundColor = fg
      when (is_high)
        Console.Write ("[1m")
      current_fg = fg
    when (bg != current_bg)
      Console.BackgroundColor = fg
      current_bg = bg

  this ()
    width = Console.WindowWidth - 1
    height = Console.WindowHeight - 1
    screen = array (width, height)
    real_screen = array (width, height)
