/*
 * Copyright (c) 2003-2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Collections;
using Nemerle.IO;
using Nemerle.Utility;

using Nemerle.Compiler;
using Nemerle.Compiler.Util;
using Nemerle.Compiler.Typedtree;

namespace Nemerle.Compiler {

  public class ICE : System.Exception
  {
    public msg : string;
    public this (m : string) { this.msg = m }
  }

  public class Recovery : System.Exception
  {
    public this () {}
  }

  public class BailOutException : System.Exception
  {
    public this () {}
  }

  public module Util 
  {
    mutable current_id : int;

    public Init () : void
    {
      current_id = 0;

    }

    public next_id () : int
    {
      current_id = current_id + 1;
      current_id
    }

    public ice['a] (msg : string) : 'a { throw ICE (msg) }
    public ice['a] () : 'a { throw ICE ("(see backtrace)") }

    public is_capitalized (s : string) : bool
    {
      def idx = s.LastIndexOf('.');
      s[idx + 1] >= 'A' && s[idx + 1] <= 'Z'
    }

    public tmpname (kind : string) : string
    {
      def sb = System.Text.StringBuilder ("_N_");
      foreach (c in kind) {
        if (System.Char.IsLetterOrDigit (c))
          _ = sb.Append (c)
        else
          _ = sb.Append ('_')
      }
      sb.Append (Util.next_id ()).ToString ()
    }

    public StripGenericMark (str : string) : string {
      def occur = str.LastIndexOf ('`');
      if (occur != -1)
        str.Substring (0, occur)
      else
        str
    }
    
    public QidOfExpr (expr : Parsetree.PExpr) : option [list [string] * Parsetree.Name]
    {
      qidl_of_expr (expr)
    }

    public ExprOfQid (fullname : string) : Parsetree.PExpr
    {
      ExprOfList (NString.Split (fullname, array ['.']))
    }

    public ExprOfList (parts : list [string]) : Parsetree.PExpr
    {
      expr_of_qidl (parts)
    }
    
    internal qidl_of_expr (expr : Parsetree.PExpr) 
           : option [list [string] * Parsetree.Name]
    {
      def loop (e, acc) {
        match (e) {
          | <[ $(x : name) ]> => Some ((x.idl :: acc, x))
          | <[ $obj . $(fld : dyn) ]> => loop (obj, fld :: acc)
          | Member (obj, HalfId (fld)) => loop (obj, fld.idl :: acc)
          | ToComplete (name) => Some (name.Id :: acc, name)
          | _ => None ()
        }
      };
      loop (expr, [])
    }

    public QidOfList (l : list [string]) : string
    {
      qid_of_list (l);
    }
    
    internal qid_of_list (l : list [string]) : string
    {
      Nemerle.Utility.NString.Concat (".", l)
    }

    internal expr_of_qidl (parts : list [string]) : Parsetree.PExpr
    {
      List.FoldLeft (parts, null, fun (name : string, expr) {
        if (expr == null)
          <[ $(name : dyn) ]>
        else
          <[ $expr . $(name : dyn) ]>
      })
    }
  }

  public module Message 
  {
    mutable error_cnt : int;
    mutable warning_cnt : int;
    mutable emitted_hints : Hashtable [string, int];

    mutable output : System.IO.TextWriter;

    public delegate MessageEventHandler (loc : Location, msg : string) : void;
    public event ErrorOccured : MessageEventHandler;
    public event WarningOccured : MessageEventHandler;
    public event MessageOccured : MessageEventHandler;    
  
    public InitOutput (o : System.IO.TextWriter) : void
    {
      output = o;
    }

    public Init () : void {
      error_cnt = 0;
      warning_cnt = 0;
      unless (emitted_hints == null) emitted_hints.Clear ();
    }

    public Error (loc : Location, m : string) : void
    {
      def loc =
        if (loc == Location.Default) Location_stack.top()
        else loc;

      when (ErrorOccured != null)
        ErrorOccured (loc, m);

      Message.error_cnt = Message.error_cnt + 1;
      if (Options.ColorMessages)
        Message.report (loc, "\e[01;31merror\e[0m: " + m)
      else
        Message.report (loc, "error: " + m);

      when (Options.ThrowOnError)
        throw BailOutException ();
    }

    public Error (m : string) : void
    {
      Message.Error (Location.Default, m)
    }

    public Warning (loc : Location, m : string) : void
    {
      Warning (-1, loc, m)
    }

    public Warning (m : string) : void { Warning (-1, Location.Default, m) }

    public Warning (code : int, loc : Location, m : string) : void
    {
      def loc =
        if (loc == Location.Default) Location_stack.top()
        else loc;

      when (code == -1 || WarningOptions.IsEnabledAt (loc, code)) {
        when (WarningOccured != null)
          WarningOccured (loc, m);
  
        Message.warning_cnt = Message.warning_cnt + 1;

        def m = if (code == -1) m else $"N$code: $m";
        
        if (Options.ColorMessages)
          Message.report (loc, "\e[01;33mwarning\e[0m: " + m)
        else
          Message.report (loc, "warning: " + m);
      }
    }

    public Warning (code : int, m : string) : void
    {
      Warning (code, Location.Default, m)
    }

    
    public Hint (loc : Location, m : string) : void
    {
      if (Options.ColorMessages)
        Message.report (loc, "\e[01;32mhint\e[0m: " + m)
      else
        Message.report (loc, "hint: " + m);
    }

    public Hint (m : string) : void
    {
      Message.Hint (Location.Default, m)
    }

    public HintOnce (loc : Location, m : string) : void {
      when (emitted_hints == null)
        emitted_hints = Hashtable (20);
      unless (emitted_hints.Contains (m)) {
        if (Options.ColorMessages)
          Message.report (loc, "\e[01;32mhint\e[0m: " + m)
        else
          Message.report (loc, "hint: " + m);
        emitted_hints.Add (m, 0)
      }
    }

    public HintOnce (code : int, loc : Location, m : string) : void
    {
      when (WarningOptions.IsEnabled (code))
        HintOnce (loc, m);
    }

    public HintOnce (code : int, m : string) : void
    {
      HintOnce (code, Location.Default, m);
    }
    
    public HintOnce (m : string) : void
    {
      Message.HintOnce (Location.Default, m)
    }

    public Debug (loc : Location, m : string) : void
    {
      Message.report (loc, "debug: " + m);
    }

    public Debug (m : string) : void
    {
      Message.Debug (Location.Default, m)
    }

    public FatalError['a] (loc : Location, m : string) : 'a
    {
      Message.Error (loc, m);
      throw Recovery ()
    }

    public FatalError['a] (m : string) : 'a
    {
      Message.FatalError (Location.Default, m) 
    }

    public MaybeBailout (fscked_up : bool) : void
    {
      unless (Options.IgnoreConfusion && fscked_up)
        when (SeenError) {
          when (fscked_up)
            printf ("confused by earlier errors bailing out\n");
          if (Options.IgnoreConfusion)
            throw Recovery ()
          else
            System.Environment.Exit (1)
        }
    }

    public SeenError : bool
    {
      get { error_cnt != 0 }
    }

    public ErrorCount : int
    {
      get { error_cnt }
    }

    public MaybeBailout () : void
    {
      Message.MaybeBailout (false)
    }
    
    internal report (l : Location, m : string) : void
    {
      def l =
        if (l == Location.Default) Location_stack.top()
        else l;
      Passes.KillProgressBar ();
      def msg = l.ToString () + m;
      when (MessageOccured != null) MessageOccured (l, msg);
      when (output == null)
        output = System.Console.Out;
      fprintf (output, "%s\n", msg)
    }
  }

  module Location_stack
  {
    mutable s : Vector [Location];

    public push (l : Location) : void
    {
      s.Add (l)
    }

    [Nemerle.Assertions.Requires (!s.IsEmpty)]
    public pop () : void
    {
      s.RemoveLast ();
    }

    public top () : Location
    {
      if (s.IsEmpty)
        Location.Default
      else
        s [s.Count - 1]
    }

    this () {
      s = Vector (32)
    }
  }
}
