/*
 * Copyright (c) 2003, 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using System.Text;
using Nemerle.Collections;
 
namespace Nemerle.Utility
{
  public module NString
  {
    /**
     * Splits the string at positions of occurrence of one
     * of the characters from the given array.
     */
    public Split (str : string, params sep : array [char]) : list [string]
    {
      def seplen = sep.Length - 1;
      
      mutable last = str.Length - 1;
      mutable res = [];
      
      for (mutable i = str.Length - 1; i >= 0; --i)
      {
        def isseparator (j) {
          if (sep[j] == str[i]) {
            when (last - i > 0)
              res = str.Substring (i + 1, last - i) :: res;
            last = i - 1;
          }
          else
            when (j < seplen) isseparator (j + 1)
        }
        
        isseparator (0)
      }
      
      if (last + 1 > 0)
        str.Substring (0, last + 1) :: res
      else
        res
    }

    
    /**
     * Splits the string at positions of occurrence of one
     * of the characters from the given list.
     */
    public Split (str : string, sep : list [char]) : list [string]
    {
      mutable last = str.Length - 1;
      mutable res = [];
      
      for (mutable i = str.Length - 1; i >= 0; --i)
      {
        def isseparator (sep : list [char])
        {
          | sep :: rest =>
            if (str [i] == sep) {
              when (last - i > 0)
                res = str.Substring (i + 1, last - i) :: res;
              last = i - 1;
            }
            else
              isseparator (rest)

          | [] => ()
        }
        
        isseparator (sep)
      }
      
      if (last + 1 > 0)
        str.Substring (0, last + 1) :: res
      else
        res
      
    }

    
    /**
     * Concatenates strings from the list, inserting given
     * separator between them.
     */
    public Concat (sep : string, l : list [string]) : string
    {
      def loop (l : list[string], acc : System.Text.StringBuilder)
      {
        match (l) {
          | [x] => acc.Append (x)
          | x :: xs => loop (xs, acc.Append (x).Append (sep))
          | [] => acc
        }
      }
      
      loop (l, System.Text.StringBuilder ("")).ToString ()
    }

    
    /**
     * Calls the given function on elements of given list, appending
     * given separator to string builder between those calls.
     *
     * NOTE: the `f' function should be taking `acc' as a parameter,
     *       so that this function looks more Fold-like.
     */
    public SeparatedCalls ['a] (sep : string, l : list ['a], f : 'a -> void,
                                acc : System.Text.StringBuilder) : void
    {
      def loop (l)
      {
        | [x] => f (x)
        | x :: xs => f (x); ignore (acc.Append (sep)); loop (xs)
        | [] => ()
      }
      
      loop (l)
    }


    /** Same as [Implode (List.Map (Explode (s), f))] but a lot faster.  */
    public Map (this s : string, f : char -> char) : string
    {
      def res = StringBuilder (s.Length);
      for (mutable i = 0; i < s.Length; ++i)
        _ = res.Append (f (s [i]));
      res.ToString ()
    }


    /** Same as [Concat ("", List.Map (Explode (s), f))] but a lot faster.  */
    public MapCS (this s : string, f : char -> string) : string
    {
      def res = StringBuilder (s.Length);
      for (mutable i = 0; i < s.Length; ++i)
        _ = res.Append (f (s [i]));
      res.ToString ()
    }


    /** Call [f] for all characters in [s] in turn.  */
    public Iter (this s : string, f : char -> void) : void
    {
      for (mutable i = 0; i < s.Length; ++i)
        f (s [i]);
    }


    /** Call [f] for all characters in [s] in turn, passing the current 
        index as the additional paramter.  */
    public IterI (this s : string, f : char * int -> void) : void
    {
      for (mutable i = 0; i < s.Length; ++i)
        f (s [i], i);
    }


    public Fold['a] (this s : string, ini : 'a, f : char * 'a -> 'a) : 'a
    {
      def loop (acc, i) {
        if (i >= s.Length) acc
        else loop (f (s [i], acc), i + 1)
      }

      loop (ini, 0)
    }

    public Fold2['a] (s1 : string, s2 : string, ini : 'a, f : char * char * 'a -> 'a) : 'a
    {
      def loop (acc, i) {
        if (i >= s1.Length) acc
        else loop (f (s1 [i], s2 [i], acc), i + 1)
      }

      if (s1.Length != s2.Length)
        throw System.ArgumentException ("NString.Fold2")
      else
        loop (ini, 0)
    }

    /** Return [true] if [f] is returns [true] for all of the characters 
        in the string [s].  */
    public ForAll (this s : string, f : char -> bool) : bool
    {
      def loop (i) {
        if (i >= s.Length) true
        else
          f (s [i]) && loop (i + 1)
      }
      loop (0)
    }


    /** Return [true] if [f] is returns [true] for any of the characters 
        in the string [s].  */
    public Exists (this s : string, f : char -> bool) : bool
    {
      def loop (i) {
        if (i >= s.Length) false
        else
          f (s [i]) || loop (i + 1)
      }
      loop (0)
    }


    /** Changes string into a corresponding list of characters.

        Warning: this should not be used in performance critical parts of
        the program, because of list's memory overheads.  */
    public Explode (this s : string) : list [char]
    {
      def loop (i, acc) {
        if (i < 0) acc
        else
          loop (i - 1, s [i] :: acc)
      }
      loop (s.Length - 1, [])
    }


    /** Constructs a string out of a list of characters.

        Warning: this should not be used in performance critical parts of
        the program, because of list's memory overheads.  */
    public Implode (s : list [char]) : string
    {
      def sb = StringBuilder ();
      foreach (ch in s)
        _ = sb.Append (ch);
      sb.ToString ()
    }
    
  }
}
