/*
 * NASPRO - The NASPRO Architecture for Sound Processing
 * Core library
 *
 * Copyright (C) 2007-2011 NASPRO core development team
 *
 * See the COPYING file for license conditions.
 */

#include "internal.h"

struct _nacore_list
  {
	nacore_list_elem	head;
	nacore_list_elem	tail;
	size_t			count;
	nacore_get_size_cb	gs_cb;
  };

struct _nacore_list_elem
  {
	void			*value;
	nacore_list_elem	 prev;
	nacore_list_elem	 next;
  };

_NACORE_DEF nacore_list
nacore_list_new(nacore_get_size_cb gs_cb)
{
	nacore_list ret;

	ret = malloc(sizeof(struct _nacore_list));
	if (ret == NULL)
		return NULL;

	ret->head	= NULL;
	ret->tail	= NULL;
	ret->count	= 0;
	ret->gs_cb	= gs_cb;

	return ret;
}

_NACORE_DEF void
nacore_list_free(nacore_list list, nacore_op_cb free_cb, void *free_opaque)
{
	nacore_list_elem elem, prev;

	if (free_cb != NULL)
		for (elem = list->head; elem != NULL; elem = elem->next)
			free_cb(elem->value, free_opaque);

	if (list->gs_cb != NULL)
		for (elem = list->head; elem != NULL; elem = elem->next)
			free(elem->value);

	for (prev = NULL, elem = list->head; elem != NULL;
	     prev = elem, elem = elem->next)
		if (prev != NULL)
			free(prev);

	if (prev != NULL)
		free(prev);

	free(list);
}

_NACORE_DEF size_t
nacore_list_get_n_elems(nacore_list list)
{
	return list->count;
}

_NACORE_DEF nacore_list_elem
nacore_list_get_head(nacore_list list)
{
	return list->head;
}

_NACORE_DEF nacore_list_elem
nacore_list_get_tail(nacore_list list)
{
	return list->tail;
}

static nacore_list_elem
elem_new(nacore_list list, void *gs_opaque, void *value)
{
	nacore_list_elem ret;
	size_t size;

	ret = malloc(sizeof(struct _nacore_list_elem));
	if (ret == NULL)
		return NULL;

	if (list->gs_cb == NULL)
	  {
		ret->value = value;
		return ret;
	  }

	size = list->gs_cb(value, gs_opaque);

	ret->value = malloc(size);
	if (ret->value == NULL)
	  {
		free(ret);
		return NULL;
	  }

	memcpy(ret->value, value, size);

	return ret;
}

_NACORE_DEF nacore_list_elem
nacore_list_prepend(nacore_list list, void *gs_opaque, void *value)
{
	nacore_list_elem ret;

	ret = elem_new(list, gs_opaque, value);
	if (ret == NULL)
		return NULL;

	ret->prev = NULL;

	if (list->head == NULL)
	  {
		list->head = ret;
		list->tail = ret;
		ret->next = NULL;
	  }
	else
	  {
		ret->next = list->head;
		list->head->prev = ret;
		list->head = ret;
	  }

	list->count++;

	return ret;
}

_NACORE_DEF nacore_list_elem
nacore_list_append(nacore_list list, void *gs_opaque, void *value)
{
	nacore_list_elem ret;

	ret = elem_new(list, gs_opaque, value);
	if (ret == NULL)
		return NULL;

	ret->next = NULL;

	if (list->tail == NULL)
	  {
		list->head = ret;
		list->tail = ret;
		ret->prev = NULL;
	  }
	else
	  {
		ret->prev = list->tail;
		list->tail->next = ret;
		list->tail = ret;
	  }

	list->count++;

	return ret;
}

_NACORE_DEF nacore_list_elem
nacore_list_insert_before(nacore_list list, nacore_list_elem elem,
			  void *gs_opaque, void *value)
{
	nacore_list_elem ret;

	ret = elem_new(list, gs_opaque, value);
	if (ret == NULL)
		return NULL;

	ret->next = elem;

	ret->prev = elem->prev;
	elem->prev = ret;

	if (ret->prev == NULL)
		list->head = ret;
	else
		ret->prev->next = ret;

	list->count++;

	return ret;
}

_NACORE_DEF nacore_list_elem
nacore_list_insert_after(nacore_list list, nacore_list_elem elem,
			 void *gs_opaque, void *value)
{
	nacore_list_elem ret;

	ret = elem_new(list, gs_opaque, value);
	if (ret == NULL)
		return NULL;

	ret->prev = elem;

	ret->next = elem->next;
	elem->next = ret;

	if (ret->next == NULL)
		list->tail = ret;
	else
		ret->next->prev = ret;

	list->count++;

	return ret;
}

_NACORE_DEF void
nacore_list_move_before(nacore_list list, nacore_list_elem dest,
			nacore_list_elem src)
{
	if (dest == src)
		return;

	if (dest->prev != src)
	  {
		if (src->prev == NULL)
		  {
			list->head = src->next;
			src->next->prev = NULL;
		  }
		else
			src->prev->next = src->next;

		if (src->next == NULL)
		  {
			list->tail = src->prev;
			src->prev->next = NULL;
		  }
		else
			src->next->prev = src->prev;

		src->next = dest;
		src->prev = dest->prev;

		if (dest->prev == NULL)
			list->head = src;
		else
			dest->prev->next = src;

		dest->prev = src;
	  }
}

_NACORE_DEF void
nacore_list_move_after(nacore_list list, nacore_list_elem dest,
			nacore_list_elem src)
{
	if (dest == src)
		return;

	if (dest->next != src)
	  {
		if (src->prev == NULL)
		  {
			list->head = src->next;
			src->next->prev = NULL;
		  }
		else
			src->prev->next = src->next;

		if (src->next == NULL)
		  {
			list->tail = src->prev;
			src->prev->next = NULL;
		  }
		else
			src->next->prev = src->prev;

		src->prev = dest;
		src->next = dest->next;

		if (dest->next == NULL)
			list->tail = src;
		else
			dest->next->prev = src;

		dest->next = src;
	  }
}

_NACORE_DEF void *
nacore_list_pop(nacore_list list, nacore_list_elem elem)
{
	void *ret;

	if (elem->prev != NULL)
		elem->prev->next = elem->next;
	else
		list->head = elem->next;

	if (elem->next != NULL)
		elem->next->prev = elem->prev;
	else
		list->tail = elem->prev;

	list->count--;

	ret = elem->value;

	free(elem);

	return ret;
}

_NACORE_DEF nacore_list_elem
nacore_list_find_first(nacore_list list, nacore_cmp_cb cmp_cb, void *cmp_opaque,
		       nacore_filter_cb filter_cb, void *filter_opaque,
		       void *value)
{
	nacore_list_elem ret;

	for (ret = list->head; ret != NULL; ret = ret->next)
		if (cmp_cb(ret->value, value, cmp_opaque) == 0)
		  {
			if (filter_cb != NULL)
				if (filter_cb(ret->value, filter_opaque) == 0)
					continue;
			break;
		  }

	return ret;
}

_NACORE_DEF nacore_list_elem
nacore_list_find_last(nacore_list list, nacore_cmp_cb cmp_cb, void *cmp_opaque,
		      nacore_filter_cb filter_cb, void *filter_opaque,
		      void *value)
{
	nacore_list_elem ret;

	for (ret = list->tail; ret != NULL; ret = ret->prev)
		if (cmp_cb(ret->value, value, cmp_opaque) == 0)
		  {
			if (filter_cb != NULL)
				if (filter_cb(ret->value, filter_opaque) == 0)
					continue;
			break;
		  }

	return ret;
}

_NACORE_DEF nacore_list_elem
nacore_list_find_before(nacore_list list, nacore_list_elem elem,
			nacore_cmp_cb cmp_cb, void *cmp_opaque,
			nacore_filter_cb filter_cb, void *filter_opaque,
			void *value)
{
	nacore_list_elem ret;

	for (ret = elem->prev; ret != NULL; ret = ret->prev)
		if (cmp_cb(ret->value, value, cmp_opaque) == 0)
		  {
			if (filter_cb != NULL)
				if (filter_cb(ret->value, filter_opaque) == 0)
					continue;
			break;
		  }

	return ret;
}

_NACORE_DEF nacore_list_elem
nacore_list_find_after(nacore_list list, nacore_list_elem elem,
		       nacore_cmp_cb cmp_cb, void *cmp_opaque,
		       nacore_filter_cb filter_cb, void *filter_opaque,
		       void *value)
{
	nacore_list_elem ret;

	for (ret = elem->next; ret != NULL; ret = ret->next)
		if (cmp_cb(ret->value, value, cmp_opaque) == 0)
		  {
			if (filter_cb != NULL)
				if (filter_cb(ret->value, filter_opaque) == 0)
					continue;
			break;
		  }

	return ret;
}

_NACORE_DEF nacore_list
nacore_list_dup(nacore_list list, nacore_get_size_cb gs_cb, void *gs_opaque,
		nacore_filter_cb filter_cb, void *filter_opaque,
		nacore_op_cb dup_cb, void *dup_opaque)
{
	nacore_list ret;
	nacore_list_elem elem;

	ret = nacore_list_new(gs_cb);

	for (elem = list->head; elem != NULL; elem = elem->next)
	  {
		if (filter_cb != NULL)
			if (filter_cb(elem->value, filter_opaque) == 0)
				continue;

		if (nacore_list_append(ret, gs_opaque, elem->value) == NULL)
		  {
			nacore_list_free(ret, NULL, NULL);
			return NULL;
		  }
	  }

	if (dup_cb != NULL)
		for (elem = ret->head; elem != NULL; elem = elem->next)
			dup_cb(elem->value, dup_opaque);

	return ret;
}

_NACORE_DEF nacore_list
nacore_list_merge(nacore_list dest, nacore_list src)
{
	if (dest->head == NULL)
	  {
		nacore_list_free(dest, NULL, NULL);
		return src;
	  }

	if (src->head == NULL)
	  {
		nacore_list_free(src, NULL, NULL);
		return dest;
	  }

	dest->tail->next = src->head;
	src->head->prev = dest->tail->next;
	dest->tail = src->tail;

	dest->count += src->count;

	src->head = NULL;
	src->tail = NULL;

	nacore_list_free(src, NULL, NULL);

	return dest;
}

_NACORE_DEF void
nacore_list_dump(nacore_list list, nacore_to_string_cb to_string_cb,
		 void *to_string_opaque)
{
	nacore_list_elem elem;
	char *desc;

	fprintf(stderr, "List: %p, head: %p, tail: %p, # elems: %"
		NACORE_LIBC_SIZE_FORMAT_LM "u\n", (void *)list,
		(void *)list->head, (void *)list->tail,
		(NACORE_LIBC_SIZE_FORMAT_TYPE)list->count);

	for (elem = list->head; elem != NULL; elem = elem->next)
	  {
		fprintf(stderr, "  Elem: %p, value ptr: %p, prev: %p, next: %p",
			(void *)elem, (void *)elem->value, (void *)elem->prev,
			(void *)elem->next);

		if (to_string_cb != NULL)
		  {
			desc = to_string_cb(elem->value, to_string_opaque);
			if (desc != NULL)
			  {
				fprintf(stderr, ", value: %s", desc);
				free(desc);
			  }
		  }

		fprintf(stderr, "\n");
	  }

	fprintf(stderr, "List dump end\n");
}

_NACORE_DEF void *
nacore_list_elem_get_value(nacore_list list, nacore_list_elem elem)
{
	return elem->value;
}

_NACORE_DEF int
nacore_list_elem_set_value(nacore_list list, nacore_list_elem elem,
			   nacore_op_cb free_cb, void *free_opaque,
			   void *gs_opaque, void *value)
{
	void *new_val;
	size_t size;

	new_val = NULL;  /* make gcc happy */

	if (list->gs_cb != NULL)
	  {
		size = list->gs_cb(value, gs_opaque);
		new_val = malloc(size);
		if (new_val == NULL)
			return ENOMEM;

		memcpy(new_val, value, size);
	  }

	if (list->gs_cb == NULL)
		elem->value = value;
	else
	  {
		if (free_cb != NULL)
			free_cb(elem->value, free_opaque);
		free(elem->value);
		elem->value = new_val;
	  }

	return 0;
}

_NACORE_DEF nacore_list_elem
nacore_list_elem_get_prev(nacore_list list, nacore_list_elem elem)
{
	return elem->prev;
}

_NACORE_DEF nacore_list_elem
nacore_list_elem_get_next(nacore_list list, nacore_list_elem elem)
{
	return elem->next;
}
