unit LayerTree;

// Copyright (C) 2003, 2004 MySQL AB
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//----------------------------------------------------------------------------------------------------------------------
//
// This unit contains the implementation of tree that is used to display the layer and element hierarchy in
// a Generic Canvas View.
//
//----------------------------------------------------------------------------------------------------------------------

interface

uses
  Windows, Classes, Graphics, TntClasses, Controls,
  myx_grt_public_interface, Grt, PNGImage,
  VirtualTrees;

type
  PMYX_GRT_VALUE = pointer;

  PLayerTreeData = ^TLayerTreeData;
  TLayerTreeData = record
    GrtValue: PMYX_GRT_VALUE;
    ValueType: GrtValueType;
    Caption: WideString;
    StructName: WideString;
    LayerCount: Integer;
    GroupCount: Integer;
    ElementCount: Integer;
  end;

  // A tree displaying the current layer, element and groups structure of a GRT view.
  TLayerTree = class(TVirtualStringTree)
  private
    FGrt: TGrt;
    FRootData: TLayerTreeData;
    FInternalDataOffset: Cardinal;
    FTreeBtnOpenPNGImg,
    FTreeBtnClosedPNGImg,
    FTreeBtnBoxPNGImg,

    FGrtValueDictPNGImg: TPNGObject;
    FGrtValueListPNGImg: TPNGObject;
    FGrtValueStructPNGImg: TPNGObject;

    FTreeImageList: TImageList;

    procedure SetRootLayer(const Value: PMYX_GRT_VALUE);
  protected
    procedure DetermineDetails(Data: PLayerTreeData; Value: PMYX_GRT_VALUE);
    procedure DoAfterCellPaint(Canvas: TCanvas; Node: PVirtualNode; Column: TColumnIndex; CellRect: TRect); override;
    procedure DoFreeNode(Node: PVirtualNode); override;
    procedure DoGetText(Node: PVirtualNode; Column: TColumnIndex; TextType: TVSTTextType;
      var Text: WideString); override;
    procedure DoInitChildren(Node: PVirtualNode; var ChildCount: Cardinal); override;
    procedure DoInitNode(Parent, Node: PVirtualNode; var InitStates: TVirtualNodeInitStates); override;
    function ElementFromList(Value: PMYX_GRT_VALUE; ListName: WideString; Index: Integer): PMYX_GRT_VALUE;
    function InternalNodeData(Node: PVirtualNode): Pointer; reintroduce;
    procedure LayerTreeGetImageIndex(Sender: TBaseVirtualTree;
      Node: PVirtualNode; Kind: TVTImageKind; Column: TColumnIndex;
      var Ghosted: Boolean; var ImageIndex: Integer);
    procedure LayerTreeMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure ValueChanged(Value: PMYX_GRT_VALUE);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

    property Grt: TGrt read FGrt write FGrt;
    property RootLayer: PMYX_GRT_VALUE write SetRootLayer;
  end;

//----------------------------------------------------------------------------------------------------------------------

implementation

uses
  SysUtils, Forms,
  PngTools;

//----------------------------------------------------------------------------------------------------------------------

function ListenerCallback(Grt: Pointer; Value: Pointer; reason: MYX_GRT_VALUE_CALLBACK_REASON; user_data: Pointer): Integer cdecl;

var
  LayerTree: TLayerTree;

begin
  case reason of
    MYX_GVCR_DICT_ITEM_CHANGE:
      begin
        LayerTree := TLayerTree(user_data);
        LayerTree.ValueChanged(Value);
      end;
  end;
  Result := 0;
end;

//----------------- TLayerTree -----------------------------------------------------------------------------------------

constructor TLayerTree.Create(AOwner: TComponent);

var
  Bmp: TBitmap;
  tmpStream: TTntResourceStream;
  Column: TVirtualTreeColumn;

begin
  inherited;

  FInternalDataOffset := AllocateInternalDataArea(SizeOf(TLayerTreeData));
  {TreeOptions.PaintOptions := [toShowButtons, toShowDropmark, toThemeAware, toUseBlendedImages];
  TreeOptions.SelectionOptions := [toFullRowSelect];

  TreeOptions.AutoOptions := TreeOptions.AutoOptions - [toAutoScrollOnExpand];
  TreeOptions.MiscOptions := TreeOptions.MiscOptions + [toGridExtensions];
  TreeOptions.PaintOptions := TreeOptions.PaintOptions - [toShowTreeLines] +
    [toShowHorzGridLines, toShowVertGridLines, toShowDropmark, toThemeAware, toUseBlendedImages];
  TreeOptions.SelectionOptions := TreeOptions.SelectionOptions + [toRightClickSelect];
  TreeOptions.StringOptions := TreeOptions.StringOptions - [toSaveCaptions];}

  TreeOptions.AutoOptions := TreeOptions.AutoOptions -
    [toAutoScrollOnExpand];
  TreeOptions.MiscOptions := TreeOptions.MiscOptions +
    [toGridExtensions];
  TreeOptions.PaintOptions := TreeOptions.PaintOptions -
    [toShowButtons, toShowDropmark, toShowTreeLines] +
    [toShowHorzGridLines, toShowVertGridLines];
  TreeOptions.SelectionOptions := TreeOptions.SelectionOptions +
    [toRightClickSelect];
  TreeOptions.StringOptions := TreeOptions.StringOptions -
    [toSaveCaptions];

  // Columns for various info.
  // 1. Visibility column
  Column := Header.Columns.Add;
  Column.MinWidth := 0;
  Column.Width := 0;

  // 2. Lock column
  Column := Header.Columns.Add;
  Column.MinWidth := 0;
  Column.Width := 0;

  // 3. Titel column (main column)
  Header.Columns.Add;

  // 4. Selection column.
  Column := Header.Columns.Add;
  Column.MinWidth := 0;
  Column.Width := 0;

  Header.AutoSizeIndex := 2;
  Header.MainColumn := 2;
  Header.Options := [hoAutoResize];

  FTreeBtnOpenPNGImg := LoadPNGImageFromResource('tree_button_open');
  FTreeBtnClosedPNGImg := LoadPNGImageFromResource('tree_button_closed');
  FTreeBtnBoxPNGImg := LoadPNGImageFromResource('tree_button_box');

  FGrtValueDictPNGImg := LoadPNGImageFromResource('grt_value_dict');
  FGrtValueListPNGImg := LoadPNGImageFromResource('grt_value_list');
  FGrtValueStructPNGImg := LoadPNGImageFromResource('grt_value_struct');

  tmpStream := TTntResourceStream.Create(
    HInstance, 'white_8x8', 'BMP');
  try
    Bmp := TBitmap.Create;
    try
      Bmp.LoadFromStream(tmpStream);

      FTreeImageList := TImageList.Create(nil);
      FTreeImageList.Width := 8;
      FTreeImageList.Height := 8;
      FTreeImageList.Masked := True;
      FTreeImageList.Add(Bmp, Bmp);

      Images := FTreeImageList;
    finally
      Bmp.Free;
    end;
  finally
    tmpStream.Free;
  end;

  OnGetImageIndex := LayerTreeGetImageIndex;
  OnMouseDown := LayerTreeMouseDown;
end;

//----------------------------------------------------------------------------------------------------------------------

destructor TLayerTree.Destroy;

begin
  FTreeBtnOpenPNGImg.Free;
  FTreeBtnClosedPNGImg.Free;
  FTreeBtnBoxPNGImg.Free;

  FGrtValueDictPNGImg.Free;
  FGrtValueListPNGImg.Free;
  FGrtValueStructPNGImg.Free;

  FTreeImageList.Free;

  inherited;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TLayerTree.SetRootLayer(const Value: PMYX_GRT_VALUE);

begin
  Clear;

  DetermineDetails(@FRootData, Value);
  RootNodeCount := FRootData.LayerCount + FRootData.GroupCount + FRootData.ElementCount;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TLayerTree.DetermineDetails(Data: PLayerTreeData; Value: PMYX_GRT_VALUE);

var
  Layers: PMYX_GRT_VALUE;
  Groups: PMYX_GRT_VALUE;
  Elements: PMYX_GRT_VALUE;

begin
  with Data^ do
  begin
    GrtValue := Value;
    
    ValueType := Grt.ValueType(Value);
    StructName := myx_grt_dict_struct_get_name(Value);
    if Assigned(FGrt) and Assigned(FGrt.NativeGrt) then
    begin
      Layers := Grt.DictItem[Value, 'subLayers'];
      if Assigned(Layers) then
        LayerCount := Grt.ListCount(Layers);
      Groups := Grt.DictItem[Value, 'groups'];
      if Assigned(Groups) then
        GroupCount := Grt.ListCount(Groups);
      Elements := Grt.DictItem[Value, 'elements'];
      if Assigned(Elements) then
        ElementCount := Grt.ListCount(Elements);
      Caption := Grt.DictString[Value, 'name'];
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TLayerTree.DoFreeNode(Node: PVirtualNode);

var
  Data: PLayerTreeData;

begin
  Data := InternalNodeData(Node);
  myx_grt_value_listener_remove(Data.GrtValue, Self, ListenerCallback);
  Finalize(Data^);

  inherited;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TLayerTree.DoAfterCellPaint(Canvas: TCanvas; Node: PVirtualNode; Column: TColumnIndex; CellRect: TRect);

var
  NodeData: PLayerTreeData;
  TxtRect: TRect;
  x,
  IconIndex: Integer;
  Icon: TPNGObject;
  PImageData: PChar;
  DataLength: Integer;
  PStruct: Pointer;

begin
  inherited;

  case Column of
    2: // Main column.
      begin
        NodeData := InternalNodeData(Node);
        TxtRect := GetDisplayRect(Node, Column, True);
        x := TxtRect.Left - OffsetX;


        // Draw > / v
        if HasChildren[Node] then
        begin
          if Expanded[Node] then
            FTreeBtnOpenPNGImg.Draw(Canvas, Rect(x - 29, CellRect.Top + 4, x - 21, CellRect.Top + 20))
          else
            FTreeBtnClosedPNGImg.Draw(Canvas, Rect(x - 29, CellRect.Top + 4, x - 21, CellRect.Top + 20))
        end
        else
          FTreeBtnBoxPNGImg.Draw(Canvas, Rect(x - 29, CellRect.Top + 4, x - 21, CellRect.Top + 20));

        // Draw Icons
        if NodeData.ValueType = GrtListValue then
          Icon := FGrtValueListPNGImg
        else
          if (NodeData.ValueType = GrtDictValue) and (NodeData.StructName = '') then
            Icon := FGrtValueDictPNGImg
          else
          begin
            IconIndex := FGrt.StructIconsList.IndexOf(NodeData.StructName);

            // Check if there is a cached icon for the struct already.
            if IconIndex > -1 then
            begin
              // if the struct was cached, use its icon, if there is any.
              Icon := TPNGObject(FGrt.StructIconsList.Objects[IconIndex]);

              // If there is no special icon for the struct, use the default one.
              if Icon = nil then
                Icon := FGrtValueStructPNGImg;
            end
            else
            begin
              PStruct := myx_grt_struct_get(FGrt.NativeGrt, NodeData.StructName);

              if (PStruct <> nil) then
              begin
                PImageData := _myx_grt_struct_get_icon(FGrt.NativeGrt,
                  PChar(ExtractFilePath(Application.ExeName)+'images\structs\'),
                  PStruct, MYX_IT_SMALL, @DataLength);

                Icon := LoadPNGImageFromPChar(PImageData, DataLength);

                FGrt.StructIconsList.AddObject(NodeData.StructName, Icon);
              end
              else
                Icon := FGrtValueStructPNGImg;
            end;

          end;

        if (Icon <> nil) then
          Icon.Draw(Canvas, Rect(x - 16, CellRect.Top + 1,
            x - 16, CellRect.Top + 17));
      end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TLayerTree.DoGetText(Node: PVirtualNode; Column: TColumnIndex; TextType: TVSTTextType; var Text: WideString);

var
  Data: PLayerTreeData;

begin
  case Column of
    2:
      begin
        Data := InternalNodeData(Node);
        Text := Data.Caption;
      end;
  else
    Text := '';
  end;

  inherited;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TLayerTree.DoInitChildren(Node: PVirtualNode; var ChildCount: Cardinal);

var
  Data: PLayerTreeData;

begin
  inherited;

  Data := InternalNodeData(Node);
  Inc(ChildCount, Data.LayerCount + Data.GroupCount + Data.ElementCount);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TLayerTree.DoInitNode(Parent, Node: PVirtualNode; var InitStates: TVirtualNodeInitStates);

var
  Data: PLayerTreeData;
  ParentData: PLayerTreeData;
  Index: Integer;
  Value: PMYX_GRT_VALUE;
  Name: PMYX_GRT_VALUE;

begin
  inherited;

  Data := InternalNodeData(Node);
  if Parent = nil then
    ParentData := @FRootData
  else
  begin
    ParentData := InternalNodeData(Parent);
  end;

  Index := Node.Index;

  if Index < ParentData.LayerCount then
  begin
    // Node belongs to the layers part.
    Value := ElementFromList(ParentData.GrtValue, 'subLayers', Index);
    myx_grt_value_listener_add(Grt.NativeGrt, Value, Self, ListenerCallback);
    DetermineDetails(Data, Value);
    if Data.LayerCount + Data.GroupCount + Data.ElementCount > 0 then
      Include(InitStates, ivsHasChildren);
  end
  else
  begin
    Dec(Index, ParentData.LayerCount);
    if Index < ParentData.GroupCount then
    begin
      // Node belongs to the groups.
      Value := ElementFromList(ParentData.GrtValue, 'groups', Index);
      myx_grt_value_listener_add(Grt.NativeGrt, Value, Self, ListenerCallback);
      DetermineDetails(Data, Value);
    end
    else
    begin
      Dec(Index, ParentData.GroupCount);

      // Node belongs to the elements part.
      // Register a listener to the value's name to accordingly update the display.
      Value := ElementFromList(ParentData.GrtValue, 'elements', Index);
      Name := Grt.DictItem[Value, 'name'];
      if Assigned(Name) then
        myx_grt_value_listener_add(Grt.NativeGrt, Name, Self, ListenerCallback);
      DetermineDetails(Data, Value);
    end;
  end;

end;

//----------------------------------------------------------------------------------------------------------------------

function TLayerTree.ElementFromList(Value: PMYX_GRT_VALUE; ListName: WideString; Index: Integer): PMYX_GRT_VALUE;

// Determines the index'd value from the GRT list of Value given by ListName. It is assumed that the list entry
// is a reference id to the true value.

begin
  Result := Grt.ListItem[Grt.DictItem[Value, ListName], Index];
  Result := Grt.ValueReference(Result);
end;

//----------------------------------------------------------------------------------------------------------------------

function TLayerTree.InternalNodeData(Node: PVirtualNode): Pointer;

begin
  if (Node = RootNode) or (Node = nil) then
    Result := nil
  else
    Result := PChar(Node) + FInternalDataOffset;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TLayerTree.LayerTreeGetImageIndex(Sender: TBaseVirtualTree;
  Node: PVirtualNode; Kind: TVTImageKind; Column: TColumnIndex;
  var Ghosted: Boolean; var ImageIndex: Integer);

begin
  ImageIndex := 0;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TLayerTree.LayerTreeMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);

var HitInfo: THitInfo;
  TxtRect: TRect;
  xpos: integer;

begin
  if(Sender.InheritsFrom(TBaseVirtualTree))then
  begin
    TBaseVirtualTree(Sender).GetHitTestInfoAt(X, Y, True, HitInfo);

    if(HitInfo.HitNode<>nil)then
    begin
      TxtRect:=TBaseVirtualTree(Sender).GetDisplayRect(
        HitInfo.HitNode, -1, True);

      xpos:=TxtRect.Left-TBaseVirtualTree(Sender).OffsetX;

      if(X>xpos-16-4-14)and(X<xpos+2-14)and
        ((HitInfo.HitNode.ChildCount>0)or(vsHasChildren in HitInfo.HitNode.States))then
      begin
        TBaseVirtualTree(Sender).Expanded[HitInfo.HitNode]:=
          Not(TBaseVirtualTree(Sender).Expanded[HitInfo.HitNode]);
      end;
    end
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TLayerTree.ValueChanged(Value: PMYX_GRT_VALUE);

// Called if something changes in a GRT value. Update the tree here for that.

var
  Data: PLayerTreeData;
  Run: PVirtualNode;
  
begin
  // Find the node that represents the given value.
  Run := GetFirst;

  while Assigned(Run) do
  begin
    Data := InternalNodeData(Run);
    if Data.GrtValue = Value then
    begin
      // Found the value. Invalidate the display and stop loop here.
      DetermineDetails(Data, Value);
      InvalidateNode(Run);
      Break;
    end;

    Run := GetNext(Run);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

end.

