/*
 * muscleTool parser.
 *
 * MUSCLE SmartCard Development ( http://www.linuxnet.com )
 *
 * Copyright (C) 2004
 *  Toni Andjelkovic <toni@soth.at>
 *  Ludovic Rousseau <ludovic.rousseau@free.fr>
 *  Karsten Ohme <k_o_@users.sourceforge.net>
 *
 * $Id: parser.y,v 1.1.1.1 2006-02-07 19:54:07 rousseau Exp $
 */

%{
#include <stdio.h>
#include <stdlib.h>
#include <musclecard.h>

#ifdef WIN32
#define snprintf _snprintf
#endif

#ifdef HAVE_READLINE
#include <readline/readline.h>
#include <readline/history.h>
#endif

#include "help.h"
#include "muscleTool.h"

#define MUSCLETOOL_PROMPT "muscleTool"
#define MUSCLETOOL_PROMPT_SIZ 256
#ifndef HAVE_READLINE
#define MUSCLETOOL_CMDBUF_SIZ 256
#endif

/* used for the prompt in main() */
extern MSCString connectToken;
static int bye = 0;

static void test(int num, char *str);

/* silence warnings */
extern int yylex(void);
int yyerror(char *s);
extern void my_lex_init(char *);
extern void my_lex_close(void);

%}

%union {
	int num;
	char *str;
}

%token CREATE
%token _DELETE
%token WRITE
%token READ
%token LIST
%token CHANGEACL
%token MOVE

%token CRYPT
%token FILECRYPT
%token RSA
%token RSACRT
%token DSA
%token EC
%token ECFP
%token ECF2M
%token DES
%token TDES
%token TDES3
%token AES
%token KEYS
%token PINS
%token OBJECTS

%token VERIFY
%token MAKEPIN
%token CHANGEPIN
%token UNBLOCK
%token LISTPINS
%token DELETEID

%token LISTKEYS
%token EXPORTKEY
%token IMPORTKEY
%token GENKEYS
%token DELETEKEY
%token MOVEKEY
%token CHANGEKEYACL

%token FORMAT
%token TOKENS
%token CONNECT
%token RESUME
%token RELEASE
%token STATUS
%token SETSTATUS
%token CHALLENGE
%token LOGOUT
%token _VERSION
%token EXIT
%token HELP

%token <num> NUMBER
%token <str> STRING
%token HEXNUMBER

%token SEPARATOR;
%token TEST;

%%

seq:		command
	|	seq SEPARATOR command
	;

command:	/* empty */
	|	action
	|	help
	;

action:		LIST				{ doList(); }
	|	CREATE STRING NUMBER		{ doCreate($2, $3); }
	|	_DELETE STRING			{ doDelete($2); }
	|	WRITE STRING STRING		{ doWrite($2, $3); }
	|	READ STRING STRING		{ doRead($2, $3); }
	|	CHANGEACL STRING 			{ doChangeACL($2); }
	|	VERIFY NUMBER 		{ doVerify($2); }
	|	MAKEPIN NUMBER STRING STRING	{ doMakePIN($2, $3, $4); }
	|	CHANGEPIN NUMBER { doChangePIN($2); }
	|	UNBLOCK NUMBER		{ doUnblockPIN($2); }
	|	LISTPINS			{ doListPINs(); }
	|	DELETEID NUMBER		{ doDeleteID($2); }
	|	LISTKEYS			{ doListKeys(); }
	|	EXPORTKEY NUMBER STRING		{ doExportKey($2, $3); }
	|	IMPORTKEY NUMBER STRING		{ doImportKey($2, $3); }
	|	CHANGEKEYACL NUMBER 			{ doChangeKeyACL($2); }
	|	GENKEYS	RSA NUMBER { doGenKeys("rsa", $3); }
	|	GENKEYS	DSA NUMBER 	{ doGenKeys("dsa", $3); }
	|	GENKEYS	ECFP NUMBER 	{ doGenKeys("ecfp", $3); }
	|	GENKEYS	ECF2M NUMBER 	{ doGenKeys("ecf2m", $3); }
	|	GENKEYS	RSACRT NUMBER 	{ doGenKeys("rsacrt", $3); }
	|	GENKEYS	DES NUMBER { doGenKeys("des", $3); }
	|	GENKEYS	TDES NUMBER { doGenKeys("3des", $3); }
	|	GENKEYS	TDES3 NUMBER { doGenKeys("3des3", $3); }
	|	GENKEYS	AES NUMBER { doGenKeys("aes", $3); }
	|	DELETEKEY NUMBER	{ doDeleteKey($2); }
	|	FILECRYPT NUMBER STRING	STRING	{ doCrypt($2, $3, $4); }
	|	CRYPT NUMBER 		{ doCrypt($2, NULL, NULL); }
	|	FORMAT NUMBER		{ doFormat($2); }
	|	TOKENS				{ doListTokens(); }
	|	MOVEKEY	NUMBER NUMBER	{ doMoveKey($2, $3); }
	|	MOVE STRING STRING	{ doMove($2, $3); }
	|	CONNECT NUMBER		{ doConnect($2); }
	|	RESUME				{ doResume(NULL); }
	|	RESUME CRYPT		{ doResume("crypt"); }
	|	RESUME RSA			{ doResume("rsa"); }
	|	RESUME DSA			{ doResume("dsa"); }
	|	RESUME DES			{ doResume("des"); }
	|	RESUME EC			{ doResume("ec"); }
	|	RESUME TDES			{ doResume("3des"); }
	|	RESUME TDES3		{ doResume("3des3"); }
	|	RESUME AES			{ doResume("aes"); }
	|	RESUME KEYS			{ doResume("keys"); }
	|	RESUME PINS			{ doResume("pins"); }
	|	RESUME OBJECTS			{ doResume("objects"); }
	|	RELEASE				{ doRelease(); }
	|	STATUS				{ doStatus(); }
	|	SETSTATUS				{ doSetStatus(); }
	|	CHALLENGE			{ doChallenge(8); }
	|	CHALLENGE NUMBER		{ doChallenge($2); }
	|	LOGOUT				{ doLogout(); }
	|	_VERSION				{ doVersion(); }
	|	EXIT				{ doExit(); bye=1; }
	|	TEST NUMBER STRING		{ test($2, $3); }
	;

help:		HELP				{ help(); }
	|	HELP CREATE			{ help_create(); }
	|	HELP _DELETE			{ help_delete(); }
	|	HELP WRITE			{ help_write(); }
	|	HELP DELETEKEY		{ help_deletekey(); }
	|	HELP READ			{ help_read(); }
	|	HELP LIST			{ help_list(); }
	|	HELP VERIFY			{ help_verify(); }
	|	HELP CHANGEACL		{ help_changeacl(); }
	|	HELP MAKEPIN			{ help_makepin(); }
	|	HELP CHANGEPIN			{ help_changepin(); }
	|	HELP UNBLOCK			{ help_unblock(); }
	|	HELP LISTPINS			{ help_listpins(); }
	|	HELP LISTKEYS			{ help_listkeys(); }
	|	HELP DELETEID			{ help_deleteid(); }
	|	HELP MOVEKEY			{ help_movekey(); }
	|	HELP CHANGEKEYACL		{ help_changekeyacl(); }
	|	HELP MOVE				{ help_move(); }
	|	HELP EXPORTKEY			{ help_exportkey(); }
	|	HELP IMPORTKEY			{ help_importkey(); }
	|	HELP GENKEYS			{ help_genkeys(); }
	|	HELP CRYPT			{ help_crypt(); }
	|	HELP FILECRYPT			{ help_filecrypt(); }
	|	HELP FORMAT			{ help_format(); }
	|	HELP TOKENS			{ help_tokens(); }
	|	HELP CONNECT			{ help_connect(); }
	|	HELP RESUME			{ help_resume(); }
	|	HELP RESUME CRYPT		{ help_resume_crypt(); }
	|	HELP RESUME RSA			{ help_resume_rsa(); }
	|	HELP RESUME DSA			{ help_resume_dsa(); }
	|	HELP RESUME DES			{ help_resume_des(); }
	|	HELP RESUME TDES		{ help_resume_3des(); }
	|	HELP RESUME TDES3		{ help_resume_3des3(); }
	|	HELP RESUME KEYS		{ help_resume_keys(); }
	|	HELP RESUME EC			{ help_resume_ec(); }
	|	HELP RESUME PINS		{ help_resume_pins(); }
	|	HELP RESUME OBJECTS		{ help_resume_objects(); }
	|	HELP RELEASE			{ help_release(); }
	|	HELP STATUS			{ help_status(); }
	|	HELP SETSTATUS			{ help_setstatus(); }
	|	HELP CHALLENGE			{ help_challenge(); }
	|	HELP LOGOUT			{ help_logout(); }
	|	HELP _VERSION			{ help_version(); }
	|	HELP EXIT			{ help_exit(); }
	|	HELP HELP			{ help_help(); }
	;

%%

static void test(int num, char *str)
{
	printf("test(%d, %s)\n", num, str);
}

int yyerror(char *s)
{
	fprintf(stderr, "Error: %s. Try \"help\" for help.\n", s);
	return 1;
}

int main(int argc, char **argv)
{
	char *line;
	int rv;
#ifndef HAVE_READLINE
	char cmdbuf[MUSCLETOOL_CMDBUF_SIZ];
#endif
	char prompt[MUSCLETOOL_PROMPT_SIZ];

	if ((argc > 1) && (0 == strcmp(argv[1], "-d"))) {
		if (getenv("MUSCLECARD_DEBUG") == NULL)
			putenv("MUSCLECARD_DEBUG=1");
	}

	printf("MuscleCard shell - type \"help\" for help.\n");
	while (!bye) {

		if (connectToken)
			snprintf(prompt,
				MUSCLETOOL_PROMPT_SIZ,
				"%s [%s] > ",
				MUSCLETOOL_PROMPT,
				connectToken);
		else
			snprintf(prompt,
				MUSCLETOOL_PROMPT_SIZ,
				"%s > ",
				MUSCLETOOL_PROMPT);
		prompt[MUSCLETOOL_PROMPT_SIZ - 1] = 0;

#ifdef HAVE_READLINE
		line = readline(prompt);
		if (line && *line)
			add_history(line);
#else
		printf("%s", prompt);
		line = fgets(cmdbuf, MUSCLETOOL_CMDBUF_SIZ, stdin);
		cmdbuf[MUSCLETOOL_CMDBUF_SIZ - 1] = 0;
#endif
		if (!line)
			break;
		my_lex_init(line);
		rv = yyparse();
		my_lex_close();
#ifdef HAVE_READLINE
		free(line);
#endif

	}

	return 0;
}

