/* 
 * encoding3548.c - RFC 3548 encodings
 *
 * Copyright (C) 2005 Yann Droneaud <ydroneaud@meuh.org>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

/* This file implements some of the encoding described in 
   RFC 3548: The Base16, Base32, and Base64 Data Encodings */

#include <sys/types.h>
#include "encoding3548.h"

static inline char 
get(const char *conv, size_t count, unsigned int index)
{
  if (index >= (unsigned int) count) {
    return '!';
  }

  return conv[index];
}

/*
 * RFC 3548
 *  6.  Base 16 Encoding
 *
 * base16 (aka hexadecimal)
 * a well known encoding
 *
 */

static const char base16_up[]  = "0123456789ABCDEF";
static const char base16_low[] = "0123456789abcdef";

size_t
base16_encode_len(size_t bin_size)
{
  return bin_size * 2;
}

/* the common function */
static size_t
_base16_encode(char *str,
	       const unsigned char *bin, size_t bin_size,
	       const char *conv)
{
  size_t ret;

  unsigned char val;

  ret = base16_encode_len(bin_size);

  for(;bin_size > 0; bin_size --) {
    val = *bin ++;
  
    *(str ++) = get(conv, 16, (val >> 4));
    *(str ++) = get(conv, 16, (val & 0x0f));
  }

  return ret;
}

size_t
base16_encode(char *str,
	      const unsigned char *bin, size_t bin_size)
{
  return _base16_encode(str, bin, bin_size, base16_up);
}

size_t
base16_up_encode(char *str,
		 const unsigned char *bin, size_t bin_size)
{
  return _base16_encode(str, bin, bin_size, base16_up);
}

size_t
base16_low_encode(char *str,
		  const unsigned char *bin, size_t bin_size)
{
  return _base16_encode(str, bin, bin_size, base16_low);
}

/* 
 * RFC 3548
 *  5.  Base 32 Encoding
 *
 */

static char base32_up[]  = "ABCDEFGHIJKLMNOPQRSTUVWXYZ234567";
static char base32_low[] = "abcdefghijklmnopqrstuvwxyz234567";

size_t
base32_encode_len(size_t bin_size)
{
  if (bin_size % 5 == 0) {
    return (bin_size / 5) * 8;
  } else {
    return ((bin_size + 5) / 5) * 8;
  }
}

/* encode the binary buffer into base32 */
static size_t
_base32_encode(char *str, 
	       const unsigned char *bin, size_t bin_size,
	       const char *conv)
{
  size_t ret;
  int i;
  int padding;

  ret = base32_encode_len(bin_size);

  for(i = 0; (i + (size_t) 5) <= bin_size; i += 5) {

    *(str ++) = get(conv, 32,  (bin[i]   & 0xf8) >> 3);
    *(str ++) = get(conv, 32, ((bin[i]   & 0x07) << 2 |
			       (bin[i+1] & 0xc0) >> 6));
    *(str ++) = get(conv, 32,  (bin[i+1] & 0x3e) >> 1); 
    *(str ++) = get(conv, 32, ((bin[i+1] & 0x01) << 4 |  
			       (bin[i+2] & 0xf0) >> 4)); 
    *(str ++) = get(conv, 32, ((bin[i+2] & 0x0f) << 1 | 
			       (bin[i+3] & 0x80) >> 7));
    *(str ++) = get(conv, 32,  (bin[i+3] & 0x7c) >> 2);
    *(str ++) = get(conv, 32, ((bin[i+3] & 0x03) << 3 |
			       (bin[i+4] & 0xe0) >> 5));
    *(str ++) = get(conv, 32,   bin[i+4] & 0x1f      ); 
  }

  padding = 0;

  switch(bin_size - i) {
  case 1:
    *(str ++) = get(conv, 32,  (bin[i]   & 0xf8) >> 3);
    *(str ++) = get(conv, 32, ((bin[i]   & 0x07) << 2));
    padding = 6;
    break;

  case 2:
    *(str ++) = get(conv, 32,  (bin[i]   & 0xf8) >> 3);
    *(str ++) = get(conv, 32, ((bin[i]   & 0x07) << 2 |
			       (bin[i+1] & 0xc0) >> 6));
    *(str ++) = get(conv, 32,  (bin[i+1] & 0x3e) >> 1); 
    *(str ++) = get(conv, 32, ((bin[i+1] & 0x01) << 4));
    padding = 4;
    break;

  case 3:
    *(str ++) = get(conv, 32,  (bin[i]   & 0xf8) >> 3);
    *(str ++) = get(conv, 32, ((bin[i]   & 0x07) << 2 |
			       (bin[i+1] & 0xc0) >> 6));
    *(str ++) = get(conv, 32,  (bin[i+1] & 0x3e) >> 1); 
    *(str ++) = get(conv, 32, ((bin[i+1] & 0x01) << 4 |  
			       (bin[i+2] & 0xf0) >> 4)); 
    *(str ++) = get(conv, 32, ((bin[i+2] & 0x0f) << 1));
    padding = 3;
    break;

  case 4:

    *(str ++) = get(conv, 32,  (bin[i]   & 0xf8) >> 3);
    *(str ++) = get(conv, 32, ((bin[i]   & 0x07) << 2 |
			       (bin[i+1] & 0xc0) >> 6));
    *(str ++) = get(conv, 32,  (bin[i+1] & 0x3e) >> 1); 
    *(str ++) = get(conv, 32, ((bin[i+1] & 0x01) << 4 |  
			       (bin[i+2] & 0xf0) >> 4)); 
    *(str ++) = get(conv, 32, ((bin[i+2] & 0x0f) << 1 | 
			       (bin[i+3] & 0x80) >> 7));
    *(str ++) = get(conv, 32,  (bin[i+3] & 0x7c) >> 2);
    *(str ++) = get(conv, 32, ((bin[i+3] & 0x03) << 3));
    padding = 1;
    break;
  }
  
  for(;padding > 0; padding --) {
    *(str ++) = '=';
  }

  return ret;
}

size_t
base32_encode(char *str, 
	      const unsigned char *bin, size_t bin_size)
{
  return _base32_encode(str, bin, bin_size, base32_up);
}

size_t
base32_up_encode(char *str,
		 const unsigned char *bin, size_t bin_size)
{
  return _base32_encode(str, bin, bin_size, base32_up);
}

size_t
base32_low_encode(char *str, 
		  const unsigned char *bin, size_t bin_size)
{
  return _base32_encode(str, bin, bin_size, base32_low);
}

/*
 * RFC 3548
 *  4.  Base 64 Encoding with URL and Filename Safe Alphabet
 *
 * base64ufs:
 * base64 adapted for url/filename described in RFC 3548
 *
 */

static const char base64ufs[] = 
"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-_";

size_t
base64_encode_len(size_t bin_size)
{
  if (bin_size % 3 == 0) {
    return (bin_size / 3) * 4;
  } else {
    return ((bin_size + 3) / 3) * 4;
  }
}

static size_t
_base64_encode(char *str,
	       const unsigned char *bin, size_t bin_size,
	       const char *conv)
{
  size_t ret;
  int i;
  int padding;

  ret = base64_encode_len(bin_size);

  for(i = 0; (i + (size_t) 3) <= bin_size; i += 3) {
    *(str ++) = get(conv, 64,  (bin[i]   & 0xfc) >> 2);
    *(str ++) = get(conv, 64, ((bin[i]   & 0x03) << 4 |
			       (bin[i+1] & 0xf0) >> 4));
    *(str ++) = get(conv, 64, ((bin[i+1] & 0x0f) << 2 |
			       (bin[i+2] & 0xc0) >> 6));
    *(str ++) = get(conv, 64,  (bin[i+2] & 0x3f));
  }

  padding = 0;

  switch(bin_size - i) {
  case 1:
    *(str ++) = get(conv, 64,  (bin[i]   & 0xfc) >> 2);
    *(str ++) = get(conv, 64, ((bin[i]   & 0x03) << 4));
    padding = 2;
    break;

  case 2:
    *(str ++) = get(conv, 64,  (bin[i]   & 0xfc) >> 2);
    *(str ++) = get(conv, 64, ((bin[i]   & 0x03) << 4 |
			       (bin[i+1] & 0xf0) >> 4));
    *(str ++) = get(conv, 64, ((bin[i+1] & 0x0f) << 2));
    padding = 1;
    break;
  }

  for(;padding > 0; padding --) {
    *(str ++) = '=';
  }

  return ret;
}

size_t
base64_ufs_encode(char *str, 
		  const unsigned char *bin, size_t bin_size)
{
  return _base64_encode(str, bin, bin_size, base64ufs);
}

#if defined(DO_TEST) && DO_TEST

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

int
test_base32_up(const char *str)
{
  size_t len;
  char *buffer;

  len = base32_encode_len(strlen(str));

  buffer = (char *) malloc(len + 1);

  base32_up_encode(buffer, str, strlen(str));
  buffer[len] = '\0';

  printf("'%s' -> '%s'\n", str, buffer);

  free(buffer);

  return 0;
}

int
test_base64_ufs(const char *str)
{
  size_t len;
  char *buffer;

  len = base64_encode_len(strlen(str));

  buffer = (char *) malloc(len + 1);

  base64_ufs_encode(buffer, str, strlen(str));
  buffer[len] = '\0';

  printf("'%s' -> '%s'\n", str, buffer);

  free(buffer);

  return 0;
}

#define test_func test_base64_ufs

int
main(void)
{

  test_func("a");
  test_func("ab");
  test_func("abc");
  test_func("abcd");
  test_func("abcde");
  test_func("abcdef");

  test_func("aa");
  test_func("aaa");
  test_func("aaaa");
  test_func("aaaaa");
  test_func("aaaaaa");

  test_func("Hell");
  test_func("Hello");
  test_func("Helloo");

  return 0;
}

#endif
