(* Copyright (C) 2005, HELM Team.
 * 
 * This file is part of HELM, an Hypertextual, Electronic
 * Library of Mathematics, developed at the Computer Science
 * Department, University of Bologna, Italy.
 * 
 * HELM is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * HELM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with HELM; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA  02111-1307, USA.
 * 
 * For details, see the HELM World-Wide-Web page,
 * http://helm.cs.unibo.it/
 *)

(* $Id: grafiteEngine.ml 8795 2008-07-07 21:01:17Z tassi $ *)

module PEH = ProofEngineHelpers

exception Drop
(* mo file name, ma file name *)
exception IncludedFileNotCompiled of string * string 
exception Macro of
 GrafiteAst.loc *
  (Cic.context -> GrafiteTypes.status * Cic.term GrafiteAst.macro)

type 'a disambiguator_input = string * int * 'a

type options = { 
  do_heavy_checks: bool ; 
}

(** create a ProofEngineTypes.mk_fresh_name_type function which uses given
  * names as long as they are available, then it fallbacks to name generation
  * using FreshNamesGenerator module *)
let namer_of names =
  let len = List.length names in
  let count = ref 0 in
  fun metasenv context name ~typ ->
    if !count < len then begin
      let name = match List.nth names !count with
         | Some s -> Cic.Name s
	 | None   -> Cic.Anonymous
      in
      incr count;
      name
    end else
      FreshNamesGenerator.mk_fresh_name ~subst:[] metasenv context name ~typ

let rec tactic_of_ast status ast =
  let module PET = ProofEngineTypes in
  match ast with
  (* Higher order tactics *)
  | GrafiteAst.Do (loc, n, tactic) ->
     Tacticals.do_tactic n (tactic_of_ast status tactic)
  | GrafiteAst.Seq (loc, tactics) ->  (* tac1; tac2; ... *)
     Tacticals.seq (List.map (tactic_of_ast status) tactics)
  | GrafiteAst.Repeat (loc, tactic) ->
     Tacticals.repeat_tactic (tactic_of_ast status tactic)
  | GrafiteAst.Then (loc, tactic, tactics) ->  (* tac; [ tac1 | ... ] *)
     Tacticals.thens
      (tactic_of_ast status tactic)
      (List.map (tactic_of_ast status) tactics)
  | GrafiteAst.First (loc, tactics) ->
     Tacticals.first (List.map (tactic_of_ast status) tactics)
  | GrafiteAst.Try (loc, tactic) ->
     Tacticals.try_tactic (tactic_of_ast status tactic)
  | GrafiteAst.Solve (loc, tactics) ->
     Tacticals.solve_tactics (List.map (tactic_of_ast status) tactics)
  | GrafiteAst.Progress (loc, tactic) ->
     Tacticals.progress_tactic (tactic_of_ast status tactic)
  (* First order tactics *)
  | GrafiteAst.Absurd (_, term) -> Tactics.absurd term
  | GrafiteAst.Apply (_, term) -> Tactics.apply term
  | GrafiteAst.ApplyP (_, term) -> Tactics.applyP term
  | GrafiteAst.ApplyS (_, term, params) ->
     Tactics.applyS ~term ~params ~dbd:(LibraryDb.instance ())
       ~universe:status.GrafiteTypes.universe
  | GrafiteAst.Assumption _ -> Tactics.assumption
  | GrafiteAst.AutoBatch (_,params) ->
      Tactics.auto ~params ~dbd:(LibraryDb.instance ()) 
	~universe:status.GrafiteTypes.universe
  | GrafiteAst.Cases (_, what, pattern, (howmany, names)) ->
      Tactics.cases_intros ?howmany ~mk_fresh_name_callback:(namer_of names)
        ~pattern what
  | GrafiteAst.Change (_, pattern, with_what) ->
     Tactics.change ~pattern with_what
  | GrafiteAst.Clear (_,id) -> Tactics.clear id
  | GrafiteAst.ClearBody (_,id) -> Tactics.clearbody id
  | GrafiteAst.Compose (_,t1,t2,times,(howmany, names)) -> 
      Tactics.compose times t1 t2 ?howmany
        ~mk_fresh_name_callback:(namer_of names)
  | GrafiteAst.Contradiction _ -> Tactics.contradiction
  | GrafiteAst.Constructor (_, n) -> Tactics.constructor n
  | GrafiteAst.Cut (_, ident, term) ->
     let names = match ident with None -> [] | Some id -> [Some id] in
     Tactics.cut ~mk_fresh_name_callback:(namer_of names) term
  | GrafiteAst.Decompose (_, names) ->
      let mk_fresh_name_callback = namer_of names in
      Tactics.decompose ~mk_fresh_name_callback ()
  | GrafiteAst.Demodulate (_, params) -> 
      Tactics.demodulate 
	~dbd:(LibraryDb.instance ()) ~params 
          ~universe:status.GrafiteTypes.universe
  | GrafiteAst.Destruct (_,xterms) -> Tactics.destruct xterms
  | GrafiteAst.Elim (_, what, using, pattern, (depth, names)) ->
      Tactics.elim_intros ?using ?depth ~mk_fresh_name_callback:(namer_of names)
        ~pattern what
  | GrafiteAst.ElimType (_, what, using, (depth, names)) ->
      Tactics.elim_type ?using ?depth ~mk_fresh_name_callback:(namer_of names)
        what
  | GrafiteAst.Exact (_, term) -> Tactics.exact term
  | GrafiteAst.Exists _ -> Tactics.exists
  | GrafiteAst.Fail _ -> Tactics.fail
  | GrafiteAst.Fold (_, reduction_kind, term, pattern) ->
      let reduction =
        match reduction_kind with
        | `Normalize ->
            PET.const_lazy_reduction
              (CicReduction.normalize ~delta:false ~subst:[])
        | `Simpl -> PET.const_lazy_reduction ProofEngineReduction.simpl
        | `Unfold None ->
            PET.const_lazy_reduction (ProofEngineReduction.unfold ?what:None)
        | `Unfold (Some lazy_term) ->
           (fun context metasenv ugraph ->
             let what, metasenv, ugraph = lazy_term context metasenv ugraph in
             ProofEngineReduction.unfold ~what, metasenv, ugraph)
        | `Whd ->
            PET.const_lazy_reduction (CicReduction.whd ~delta:false ~subst:[])
      in
      Tactics.fold ~reduction ~term ~pattern
  | GrafiteAst.Fourier _ -> Tactics.fourier
  | GrafiteAst.FwdSimpl (_, hyp, names) -> 
     Tactics.fwd_simpl ~mk_fresh_name_callback:(namer_of names)
      ~dbd:(LibraryDb.instance ()) hyp
  | GrafiteAst.Generalize (_,pattern,ident) ->
     let names = match ident with None -> [] | Some id -> [Some id] in
     Tactics.generalize ~mk_fresh_name_callback:(namer_of names) pattern 
  | GrafiteAst.IdTac _ -> Tactics.id
  | GrafiteAst.Intros (_, (howmany, names)) ->
      PrimitiveTactics.intros_tac ?howmany
        ~mk_fresh_name_callback:(namer_of names) ()
  | GrafiteAst.Inversion (_, term) ->
      Tactics.inversion term
  | GrafiteAst.LApply (_, linear, how_many, to_what, what, ident) ->
      let names = match ident with None -> [] | Some id -> [Some id] in
      Tactics.lapply ~mk_fresh_name_callback:(namer_of names) 
        ~linear ?how_many ~to_what what
  | GrafiteAst.Left _ -> Tactics.left
  | GrafiteAst.LetIn (loc,term,name) ->
      Tactics.letin term ~mk_fresh_name_callback:(namer_of [Some name])
  | GrafiteAst.Reduce (_, reduction_kind, pattern) ->
      (match reduction_kind with
	 | `Normalize -> Tactics.normalize ~pattern
	 | `Simpl -> Tactics.simpl ~pattern 
	 | `Unfold what -> Tactics.unfold ~pattern what
	 | `Whd -> Tactics.whd ~pattern)
  | GrafiteAst.Reflexivity _ -> Tactics.reflexivity
  | GrafiteAst.Replace (_, pattern, with_what) ->
     Tactics.replace ~pattern ~with_what
  | GrafiteAst.Rewrite (_, direction, t, pattern, names) ->
     EqualityTactics.rewrite_tac ~direction ~pattern t 
(* to be replaced with ~mk_fresh_name_callback:(namer_of names) *)
     (List.map (function Some s -> s | None -> assert false) names)
  | GrafiteAst.Right _ -> Tactics.right
  | GrafiteAst.Ring _ -> Tactics.ring
  | GrafiteAst.Split _ -> Tactics.split
  | GrafiteAst.Symmetry _ -> Tactics.symmetry
  | GrafiteAst.Transitivity (_, term) -> Tactics.transitivity term
  (* Implementazioni Aggiunte *)
  | GrafiteAst.Assume (_, id, t) -> Declarative.assume id t
  | GrafiteAst.Suppose (_, t, id, t1) -> Declarative.suppose t id t1
  | GrafiteAst.By_just_we_proved (_, just, ty, id, t1) ->
     Declarative.by_just_we_proved ~dbd:(LibraryDb.instance())
      ~universe:status.GrafiteTypes.universe just ty id t1
  | GrafiteAst.We_need_to_prove (_, t, id, t2) ->
     Declarative.we_need_to_prove t id t2
  | GrafiteAst.Bydone (_, t) ->
     Declarative.bydone ~dbd:(LibraryDb.instance())
      ~universe:status.GrafiteTypes.universe t
  | GrafiteAst.We_proceed_by_cases_on (_, t, t1) ->
     Declarative.we_proceed_by_cases_on t t1
  | GrafiteAst.We_proceed_by_induction_on (_, t, t1) ->
     Declarative.we_proceed_by_induction_on t t1
  | GrafiteAst.Byinduction (_, t, id) -> Declarative.byinduction t id
  | GrafiteAst.Thesisbecomes (_, t) -> Declarative.thesisbecomes t
  | GrafiteAst.ExistsElim (_, just, id1, t1, id2, t2) ->
     Declarative.existselim ~dbd:(LibraryDb.instance())
      ~universe:status.GrafiteTypes.universe just id1 t1 id2 t2
  | GrafiteAst.Case (_,id,params) -> Declarative.case id params
  | GrafiteAst.AndElim(_,just,id1,t1,id2,t2) ->
     Declarative.andelim ~dbd:(LibraryDb.instance ())
      ~universe:status.GrafiteTypes.universe just id1 t1 id2 t2
  | GrafiteAst.RewritingStep (_,termine,t1,t2,cont) ->
     Declarative.rewritingstep ~dbd:(LibraryDb.instance ())
      ~universe:status.GrafiteTypes.universe termine t1 t2 cont

let classify_tactic tactic = 
  match tactic with
  (* tactics that can't close the goal (return a goal we want to "select") *)
  | GrafiteAst.Rewrite _ 
  | GrafiteAst.Split _ 
  | GrafiteAst.Replace _ 
  | GrafiteAst.Reduce _
  | GrafiteAst.IdTac _ 
  | GrafiteAst.Generalize _ 
  | GrafiteAst.Elim _ 
  | GrafiteAst.Cut _
  | GrafiteAst.Decompose _ -> true
  (* tactics like apply *)
  | _ -> false
  
let reorder_metasenv start refine tactic goals current_goal always_opens_a_goal=
(*   let print_m name metasenv =
    prerr_endline (">>>>> " ^ name);
    prerr_endline (CicMetaSubst.ppmetasenv [] metasenv)
  in *)
  (* phase one calculates:
   *   new_goals_from_refine:  goals added by refine
   *   head_goal:              the first goal opened by ythe tactic 
   *   other_goals:            other goals opened by the tactic
   *)
  let new_goals_from_refine = PEH.compare_metasenvs start refine in
  let new_goals_from_tactic = PEH.compare_metasenvs refine tactic in
  let head_goal, other_goals, goals = 
    match goals with
    | [] -> None,[],goals
    | hd::tl -> 
        (* assert (List.mem hd new_goals_from_tactic);
         * invalidato dalla goal_tac
         * *)
        Some hd, List.filter ((<>) hd) new_goals_from_tactic, List.filter ((<>)
        hd) goals
  in
  let produced_goals = 
    match head_goal with
    | None -> new_goals_from_refine @ other_goals
    | Some x -> x :: new_goals_from_refine @ other_goals
  in
  (* extract the metas generated by refine and tactic *)
  let metas_for_tactic_head = 
    match head_goal with
    | None -> []
    | Some head_goal -> List.filter (fun (n,_,_) -> n = head_goal) tactic in
  let metas_for_tactic_goals = 
    List.map 
      (fun x -> List.find (fun (metano,_,_) -> metano = x) tactic)
    goals 
  in
  let metas_for_refine_goals = 
    List.filter (fun (n,_,_) -> List.mem n new_goals_from_refine) tactic in
  let produced_metas, goals = 
    let produced_metas =
      if always_opens_a_goal then
        metas_for_tactic_head @ metas_for_refine_goals @ 
          metas_for_tactic_goals
      else begin
(*         print_m "metas_for_refine_goals" metas_for_refine_goals;
        print_m "metas_for_tactic_head" metas_for_tactic_head;
        print_m "metas_for_tactic_goals" metas_for_tactic_goals; *)
        metas_for_refine_goals @ metas_for_tactic_head @ 
          metas_for_tactic_goals
      end
    in
    let goals = List.map (fun (metano, _, _) -> metano)  produced_metas in
    produced_metas, goals
  in
  (* residual metas, preserving the original order *)
  let before, after = 
    let rec split e =
      function 
      | [] -> [],[]
      | (metano, _, _) :: tl when metano = e -> 
          [], List.map (fun (x,_,_) -> x) tl
      | (metano, _, _) :: tl -> let b, a = split e tl in metano :: b, a
    in
    let find n metasenv =
      try
        Some (List.find (fun (metano, _, _) -> metano = n) metasenv)
      with Not_found -> None
    in
    let extract l =
      List.fold_right 
        (fun n acc -> 
          match find n tactic with
          | Some x -> x::acc
          | None -> acc
        ) l [] in
    let before_l, after_l = split current_goal start in
    let before_l = 
      List.filter (fun x -> not (List.mem x produced_goals)) before_l in
    let after_l = 
      List.filter (fun x -> not (List.mem x produced_goals)) after_l in
    let before = extract before_l in
    let after = extract after_l in
      before, after
  in
(* |+   DEBUG CODE  +|
  print_m "BEGIN" start;
  prerr_endline ("goal was: " ^ string_of_int current_goal);
  prerr_endline ("and metas from refine are:");
  List.iter 
    (fun t -> prerr_string (" " ^ string_of_int t)) 
  new_goals_from_refine;
  prerr_endline "";
  print_m "before" before;
  print_m "metas_for_tactic_head" metas_for_tactic_head;
  print_m "metas_for_refine_goals" metas_for_refine_goals;
  print_m "metas_for_tactic_goals" metas_for_tactic_goals;
  print_m "produced_metas" produced_metas;
  print_m "after" after; 
|+   FINE DEBUG CODE +| *)
  before @ produced_metas @ after, goals 
  
let apply_tactic ~disambiguate_tactic (text,prefix_len,tactic) (status, goal) =
 let starting_metasenv = GrafiteTypes.get_proof_metasenv status in
 let before = List.map (fun g, _, _ -> g) starting_metasenv in
 let status, tactic = disambiguate_tactic status goal (text,prefix_len,tactic) in
 let metasenv_after_refinement =  GrafiteTypes.get_proof_metasenv status in
 let proof = GrafiteTypes.get_current_proof status in
 let proof_status = proof, goal in
 let always_opens_a_goal = classify_tactic tactic in
 let tactic = tactic_of_ast status tactic in
 let (proof, opened) = ProofEngineTypes.apply_tactic tactic proof_status in
 let after = ProofEngineTypes.goals_of_proof proof in
 let opened_goals, closed_goals = Tacticals.goals_diff ~before ~after ~opened in
 let proof, opened_goals = 
  let uri, metasenv_after_tactic, _subst, t, ty, attrs = proof in
  let reordered_metasenv, opened_goals = 
    reorder_metasenv
     starting_metasenv
     metasenv_after_refinement metasenv_after_tactic
     opened goal always_opens_a_goal
  in
  let proof' = uri, reordered_metasenv, _subst, t, ty, attrs in
  proof', opened_goals
 in
 let incomplete_proof =
   match status.GrafiteTypes.proof_status with
   | GrafiteTypes.Incomplete_proof p -> p
   | _ -> assert false
 in
 { status with GrafiteTypes.proof_status =
    GrafiteTypes.Incomplete_proof
     { incomplete_proof with GrafiteTypes.proof = proof } },
 opened_goals, closed_goals

let apply_atomic_tactical ~disambiguate_tactic ~patch (text,prefix_len,tactic) (status, goal) =
 let starting_metasenv = GrafiteTypes.get_proof_metasenv status in
 let before = List.map (fun g, _, _ -> g) starting_metasenv in
 let status, tactic = disambiguate_tactic status goal (text,prefix_len,tactic) in
 let metasenv_after_refinement =  GrafiteTypes.get_proof_metasenv status in
 let proof = GrafiteTypes.get_current_proof status in
 let proof_status = proof, goal in
 let always_opens_a_goal = classify_tactic tactic in
 let tactic = tactic_of_ast status tactic in
 let tactic = patch tactic in
 let (proof, opened) = ProofEngineTypes.apply_tactic tactic proof_status in
 let after = ProofEngineTypes.goals_of_proof proof in
 let opened_goals, closed_goals = Tacticals.goals_diff ~before ~after ~opened in
 let proof, opened_goals = 
  let uri, metasenv_after_tactic, _subst, t, ty, attrs = proof in
  let reordered_metasenv, opened_goals = 
    reorder_metasenv
     starting_metasenv
     metasenv_after_refinement metasenv_after_tactic
     opened goal always_opens_a_goal
  in
  let proof' = uri, reordered_metasenv, _subst, t, ty, attrs in
  proof', opened_goals
 in
 let incomplete_proof =
   match status.GrafiteTypes.proof_status with
   | GrafiteTypes.Incomplete_proof p -> p
   | _ -> assert false
 in
 { status with GrafiteTypes.proof_status =
    GrafiteTypes.Incomplete_proof
     { incomplete_proof with GrafiteTypes.proof = proof } },
 opened_goals, closed_goals
type eval_ast =
 {ea_go:
  'term 'lazy_term 'reduction 'obj 'ident.
  disambiguate_tactic:
   (GrafiteTypes.status ->
    ProofEngineTypes.goal ->
    (('term, 'lazy_term, 'reduction, 'ident) GrafiteAst.tactic)
    disambiguator_input ->
    GrafiteTypes.status *
   (Cic.term, Cic.lazy_term, Cic.lazy_term GrafiteAst.reduction, string) GrafiteAst.tactic) ->

  disambiguate_command:
   (GrafiteTypes.status ->
    (('term,'obj) GrafiteAst.command) disambiguator_input ->
    GrafiteTypes.status * (Cic.term,Cic.obj) GrafiteAst.command) ->

  disambiguate_macro:
   (GrafiteTypes.status ->
    ('term GrafiteAst.macro) disambiguator_input ->
    Cic.context -> GrafiteTypes.status * Cic.term GrafiteAst.macro) ->

  ?do_heavy_checks:bool ->
  GrafiteTypes.status ->
  (('term, 'lazy_term, 'reduction, 'obj, 'ident) GrafiteAst.statement)
  disambiguator_input ->
  GrafiteTypes.status * UriManager.uri list
 }

type 'a eval_command =
 {ec_go: 'term 'obj.
  disambiguate_command:
   (GrafiteTypes.status -> (('term,'obj) GrafiteAst.command) disambiguator_input ->
    GrafiteTypes.status * (Cic.term,Cic.obj) GrafiteAst.command) -> 
  options -> GrafiteTypes.status -> 
    (('term,'obj) GrafiteAst.command) disambiguator_input ->
   GrafiteTypes.status * UriManager.uri list
 }

type 'a eval_executable =
 {ee_go: 'term 'lazy_term 'reduction 'obj 'ident.
  disambiguate_tactic:
   (GrafiteTypes.status ->
    ProofEngineTypes.goal ->
    (('term, 'lazy_term, 'reduction, 'ident) GrafiteAst.tactic)
    disambiguator_input ->
    GrafiteTypes.status *
   (Cic.term, Cic.lazy_term, Cic.lazy_term GrafiteAst.reduction, string) GrafiteAst.tactic) ->

  disambiguate_command:
   (GrafiteTypes.status ->
    (('term,'obj) GrafiteAst.command) disambiguator_input ->
    GrafiteTypes.status * (Cic.term,Cic.obj) GrafiteAst.command) ->

  disambiguate_macro:
   (GrafiteTypes.status ->
    ('term GrafiteAst.macro) disambiguator_input ->
    Cic.context -> GrafiteTypes.status * Cic.term GrafiteAst.macro) ->

  options ->
  GrafiteTypes.status ->
  (('term, 'lazy_term, 'reduction, 'obj, 'ident) GrafiteAst.code) disambiguator_input ->
  GrafiteTypes.status * UriManager.uri list
 }

type 'a eval_from_moo =
 { efm_go: GrafiteTypes.status -> string -> GrafiteTypes.status }
      
let coercion_moo_statement_of (uri,arity, saturations,_) =
  GrafiteAst.Coercion
   (HExtlib.dummy_floc, CicUtil.term_of_uri uri, false, arity, saturations)

let refinement_toolkit = {
  RefinementTool.type_of_aux' = 
    (fun ?localization_tbl e c t u ->
      let saved = !CicRefine.insert_coercions in 
      CicRefine.insert_coercions:= false;
      let rc = 
        try 
          let t, ty, metasenv, ugraph = 
            CicRefine.type_of_aux' ?localization_tbl e c t u in
          RefinementTool.Success (t, ty, metasenv, ugraph)
        with
        | CicRefine.RefineFailure s
        | CicRefine.Uncertain s 
        | CicRefine.AssertFailure s -> RefinementTool.Exception s
      in
      CicRefine.insert_coercions := saved;
      rc);
  RefinementTool.ppsubst = CicMetaSubst.ppsubst;
  RefinementTool.apply_subst = CicMetaSubst.apply_subst; 
  RefinementTool.ppmetasenv = CicMetaSubst.ppmetasenv; 
  RefinementTool.pack_coercion_obj = CicRefine.pack_coercion_obj;
 }
  
let eval_coercion status ~add_composites uri arity saturations =
 let uri = 
   try CicUtil.uri_of_term uri 
   with Invalid_argument _ -> 
     raise (Invalid_argument "coercion can only be constants/constructors")
 in
 let status,compounds =
  GrafiteSync.add_coercion ~add_composites refinement_toolkit status uri arity
   saturations (GrafiteTypes.get_baseuri status)
 in
 let moo_content = 
   List.map coercion_moo_statement_of ((uri,arity,saturations,0)::compounds)
 in
 let status = GrafiteTypes.add_moo_content moo_content status in
  {status with GrafiteTypes.proof_status = GrafiteTypes.No_proof},
   List.map (fun u,_,_,_ -> u) compounds

module MatitaStatus =
 struct
  type input_status = GrafiteTypes.status * ProofEngineTypes.goal

  type output_status =
    GrafiteTypes.status * ProofEngineTypes.goal list * ProofEngineTypes.goal list

  type tactic = input_status -> output_status

  let mk_tactic tac = tac
  let apply_tactic tac = tac
  let goals (_, opened, closed) = opened, closed
  let get_stack (status, _) = GrafiteTypes.get_stack status
  
  let set_stack stack (status, opened, closed) = 
    GrafiteTypes.set_stack stack status, opened, closed

  let inject (status, _) = (status, [], [])
  let focus goal (status, _, _) = (status, goal)
 end

module MatitaTacticals = Continuationals.Make(MatitaStatus)

let tactic_of_ast' tac =
 MatitaTacticals.Tactical (MatitaTacticals.Tactic (MatitaStatus.mk_tactic tac))

let punctuation_tactical_of_ast (text,prefix_len,punct) =
 match punct with
  | GrafiteAst.Dot _loc -> MatitaTacticals.Dot
  | GrafiteAst.Semicolon _loc -> MatitaTacticals.Semicolon
  | GrafiteAst.Branch _loc -> MatitaTacticals.Branch
  | GrafiteAst.Shift _loc -> MatitaTacticals.Shift
  | GrafiteAst.Pos (_loc, i) -> MatitaTacticals.Pos i
  | GrafiteAst.Merge _loc -> MatitaTacticals.Merge
  | GrafiteAst.Wildcard _loc -> MatitaTacticals.Wildcard

let non_punctuation_tactical_of_ast (text,prefix_len,punct) =
 match punct with
  | GrafiteAst.Focus (_loc,goals) -> MatitaTacticals.Focus goals
  | GrafiteAst.Unfocus _loc -> MatitaTacticals.Unfocus
  | GrafiteAst.Skip _loc -> MatitaTacticals.Tactical MatitaTacticals.Skip

let eval_tactical status tac =
  let status, _, _ = MatitaTacticals.eval tac (status, ~-1) in
  let status =  (* is proof completed? *)
    match status.GrafiteTypes.proof_status with
    | GrafiteTypes.Incomplete_proof
       { GrafiteTypes.stack = stack; proof = proof }
      when Continuationals.Stack.is_empty stack ->
        { status with GrafiteTypes.proof_status = GrafiteTypes.Proof proof }
    | _ -> status
  in
  status

let eval_comment status c = status

(* since the record syntax allows to declare coercions, we have to put this
 * information inside the moo *)
let add_coercions_of_record_to_moo obj lemmas status =
  let attributes = CicUtil.attributes_of_obj obj in
  let is_record = function `Class (`Record att) -> Some att | _-> None in
  match HExtlib.list_findopt is_record attributes with
  | None -> status,[]
  | Some fields -> 
      let is_a_coercion uri =
        try
          let obj,_ = 
            CicEnvironment.get_cooked_obj  CicUniv.oblivion_ugraph uri in
          let attrs = CicUtil.attributes_of_obj obj in
          try 
            match List.find 
             (function `Class (`Coercion _) -> true | _-> false) attrs
            with `Class (`Coercion n) -> true,n | _ -> assert false
          with Not_found -> false,0            
        with Not_found -> assert false
      in
      let buri = GrafiteTypes.get_baseuri status in
      (* looking at the fields we can know the 'wanted' coercions, but not the 
       * actually generated ones. So, only the intersection between the wanted
       * and the actual should be in the moo as coercion, while everithing in
       * lemmas should go as aliases *)
      let wanted_coercions = 
        HExtlib.filter_map 
          (function 
            | (name,true,arity) -> 
               Some 
                 (arity, UriManager.uri_of_string (buri ^ "/" ^ name ^ ".con" ))
            | _ -> None) 
          fields
      in
      (*prerr_endline "wanted coercions:";
      List.iter 
        (fun u -> prerr_endline (UriManager.string_of_uri u)) 
        wanted_coercions; *)
      let coercions, moo_content = 
        List.split
          (HExtlib.filter_map 
            (fun uri ->
              let is_a_wanted_coercion,arity_wanted = 
                try
                  let arity,_ = 
                    List.find (fun (n,u) -> UriManager.eq u uri) 
                      wanted_coercions
                  in
                  true, arity
                with Not_found -> false, 0
              in
              let is_a_coercion, arity_coercion = is_a_coercion uri in
              if is_a_coercion then
                Some (uri, coercion_moo_statement_of (uri,arity_coercion,0,0))
              else if is_a_wanted_coercion then
                Some (uri, coercion_moo_statement_of (uri,arity_wanted,0,0))
              else
                None)
            lemmas)
      in
      (*prerr_endline "actual coercions:";
      List.iter 
        (fun u -> prerr_endline (UriManager.string_of_uri u)) 
        coercions; 
      prerr_endline "lemmas was:";
      List.iter 
        (fun u -> prerr_endline (UriManager.string_of_uri u)) 
        lemmas; *)
      let status = GrafiteTypes.add_moo_content moo_content status in 
      {status with 
        GrafiteTypes.coercions = coercions @ status.GrafiteTypes.coercions}, 
      lemmas

let add_obj uri obj status =
 let status,lemmas = GrafiteSync.add_obj refinement_toolkit uri obj status in
 status, lemmas 
      
let rec eval_command = {ec_go = fun ~disambiguate_command opts status
(text,prefix_len,cmd) ->
 let status,cmd = disambiguate_command status (text,prefix_len,cmd) in
 let status,uris =
  match cmd with
  | GrafiteAst.Index (loc,None,uri) -> 
	assert false (* TODO: for user input *)
  | GrafiteAst.Index (loc,Some key,uri) -> 
      let universe = Universe.index 
	status.GrafiteTypes.universe key (CicUtil.term_of_uri uri) in
      let status = {status with GrafiteTypes.universe = universe} in
(* debug
      let msg =
       let candidates = Universe.get_candidates status.GrafiteTypes.universe key in
       ("candidates for " ^ (CicPp.ppterm key) ^ " = " ^ 
	  (String.concat "\n" (List.map CicPp.ppterm candidates))) 
     in
     prerr_endline msg;
*)
      let status = GrafiteTypes.add_moo_content [cmd] status in
      status,[] 
  | GrafiteAst.Coercion (loc, uri, add_composites, arity, saturations) ->
     eval_coercion status ~add_composites uri arity saturations 
  | GrafiteAst.Default (loc, what, uris) as cmd ->
     LibraryObjects.set_default what uris;
     GrafiteTypes.add_moo_content [cmd] status,[]
  | GrafiteAst.Drop loc -> raise Drop
  | GrafiteAst.Include (loc, baseuri) ->
     let moopath_rw, moopath_r = 
       LibraryMisc.obj_file_of_baseuri 
         ~must_exist:false ~baseuri ~writable:true,
       LibraryMisc.obj_file_of_baseuri 
         ~must_exist:false ~baseuri ~writable:false
     in
     let moopath = 
       if Sys.file_exists moopath_r then moopath_r else
         if Sys.file_exists moopath_rw then moopath_rw else
           raise (IncludedFileNotCompiled (moopath_rw,baseuri))
     in
     let status = eval_from_moo.efm_go status moopath in
(* debug
     let lt_uri = UriManager.uri_of_string "cic:/matita/nat/orders/lt.con" in
     let nat_uri = UriManager.uri_of_string "cic:/matita/nat/nat/nat.ind" in
     let nat = Cic.MutInd(nat_uri,0,[]) in
     let zero = Cic.MutConstruct(nat_uri,0,1,[]) in
     let succ = Cic.MutConstruct(nat_uri,0,2,[]) in
     let fake= Cic.Meta(-1,[]) in
     let term= Cic.Appl [Cic.Const (lt_uri,[]);zero;Cic.Appl[succ;zero]] in     let msg =
       let candidates = Universe.get_candidates status.GrafiteTypes.universe term in
       ("candidates for " ^ (CicPp.ppterm term) ^ " = " ^ 
	  (String.concat "\n" (List.map CicPp.ppterm candidates))) 
     in
     prerr_endline msg;
*)
     status,[]
  | GrafiteAst.Print (_,"proofterm") ->
      let _,_,_,p,_, _ = GrafiteTypes.get_current_proof status in
      prerr_endline (Auto.pp_proofterm p);
      status,[]
  | GrafiteAst.Print (_,_) -> status,[]
  | GrafiteAst.Qed loc ->
      let uri, metasenv, _subst, bo, ty, attrs =
        match status.GrafiteTypes.proof_status with
        | GrafiteTypes.Proof (Some uri, metasenv, subst, body, ty, attrs) ->
            uri, metasenv, subst, body, ty, attrs
        | GrafiteTypes.Proof (None, metasenv, subst, body, ty, attrs) -> 
            raise (GrafiteTypes.Command_error 
              ("Someone allows to start a theorem without giving the "^
               "name/uri. This should be fixed!"))
        | _->
          raise
           (GrafiteTypes.Command_error "You can't Qed an incomplete theorem")
      in
      if metasenv <> [] then 
        raise
         (GrafiteTypes.Command_error
           "Proof not completed! metasenv is not empty!");
      let name = UriManager.name_of_uri uri in
      let obj = Cic.Constant (name,Some bo,ty,[],attrs) in
      let status, lemmas = add_obj uri obj status in
       {status with 
	  GrafiteTypes.proof_status = GrafiteTypes.No_proof},
        (*CSC: I throw away the arities *)
        uri::lemmas
  | GrafiteAst.Relation (loc, id, a, aeq, refl, sym, trans) -> 
     Setoids.add_relation id a aeq refl sym trans;
     status, [] (*CSC: TO BE FIXED *)
  | GrafiteAst.Set (loc, name, value) -> status, []
(*       GrafiteTypes.set_option status name value,[] *)
  | GrafiteAst.Obj (loc,obj) ->
     let ext,name =
      match obj with
         Cic.Constant (name,_,_,_,_)
       | Cic.CurrentProof (name,_,_,_,_,_) -> ".con",name
       | Cic.InductiveDefinition (types,_,_,_) ->
          ".ind",
          (match types with (name,_,_,_)::_ -> name | _ -> assert false)
       | _ -> assert false in
     let buri = GrafiteTypes.get_baseuri status in 
     let uri = UriManager.uri_of_string (buri ^ "/" ^ name ^ ext) in
     let obj = CicRefine.pack_coercion_obj obj in
     let metasenv = GrafiteTypes.get_proof_metasenv status in
     match obj with
     | Cic.CurrentProof (_,metasenv',bo,ty,_, attrs) ->
         let name = UriManager.name_of_uri uri in
         if not(CicPp.check name ty) then
           HLog.warn ("Bad name: " ^ name);
         if opts.do_heavy_checks then
           begin
             let dbd = LibraryDb.instance () in
             let similar = Whelp.match_term ~dbd ty in
             let similar_len = List.length similar in
             if similar_len> 30 then
               (HLog.message
                 ("Duplicate check will compare your theorem with " ^ 
                   string_of_int similar_len ^ 
                   " theorems, this may take a while."));
             let convertible =
               List.filter (
                 fun u ->
                   let t = CicUtil.term_of_uri u in
                   let ty',g = 
                     CicTypeChecker.type_of_aux' 
                       metasenv' [] t CicUniv.oblivion_ugraph
                   in
                   fst(CicReduction.are_convertible [] ty' ty g)) 
               similar 
             in
             (match convertible with
             | [] -> ()
             | x::_ -> 
                 HLog.warn  
                 ("Theorem already proved: " ^ UriManager.string_of_uri x ^ 
                  "\nPlease use a variant."));
           end;
         let _subst = [] in
         let initial_proof = (Some uri, metasenv', _subst, bo, ty, attrs) in
         let initial_stack = Continuationals.Stack.of_metasenv metasenv' in
         { status with GrafiteTypes.proof_status =
            GrafiteTypes.Incomplete_proof
             { GrafiteTypes.proof = initial_proof; stack = initial_stack } },
          []
     | _ ->
         if metasenv <> [] then
          raise (GrafiteTypes.Command_error (
            "metasenv not empty while giving a definition with body: " ^
            CicMetaSubst.ppmetasenv [] metasenv));
         let status, lemmas = add_obj uri obj status in 
         let status,new_lemmas =
          add_coercions_of_record_to_moo obj lemmas status
         in
          {status with GrafiteTypes.proof_status = GrafiteTypes.No_proof},
           uri::new_lemmas@lemmas
 in
  match status.GrafiteTypes.proof_status with
     GrafiteTypes.Intermediate _ ->
      {status with GrafiteTypes.proof_status = GrafiteTypes.No_proof},uris
   | _ -> status,uris

} and eval_executable = {ee_go = fun ~disambiguate_tactic ~disambiguate_command
~disambiguate_macro opts status (text,prefix_len,ex) ->
  match ex with
  | GrafiteAst.Tactic (_, Some tac, punct) ->
     let tac = apply_tactic ~disambiguate_tactic (text,prefix_len,tac) in
     let status = eval_tactical status (tactic_of_ast' tac) in
      eval_tactical status
       (punctuation_tactical_of_ast (text,prefix_len,punct)),[]
  | GrafiteAst.Tactic (_, None, punct) ->
      eval_tactical status
       (punctuation_tactical_of_ast (text,prefix_len,punct)),[]
  | GrafiteAst.NonPunctuationTactical (_, tac, punct) ->
     let status = 
      eval_tactical status
       (non_punctuation_tactical_of_ast (text,prefix_len,tac))
     in
      eval_tactical status
       (punctuation_tactical_of_ast (text,prefix_len,punct)),[]
  | GrafiteAst.Command (_, cmd) ->
      eval_command.ec_go ~disambiguate_command opts status (text,prefix_len,cmd)
  | GrafiteAst.Macro (loc, macro) ->
     raise (Macro (loc,disambiguate_macro status (text,prefix_len,macro)))

} and eval_from_moo = {efm_go = fun status fname ->
  let ast_of_cmd cmd =
    ("",0,GrafiteAst.Executable (HExtlib.dummy_floc,
      GrafiteAst.Command (HExtlib.dummy_floc,
        cmd)))
  in
  let moo = GrafiteMarshal.load_moo fname in
  List.fold_left 
    (fun status ast -> 
      let ast = ast_of_cmd ast in
      let status,lemmas =
       eval_ast.ea_go
         ~disambiguate_tactic:(fun status _ (_,_,tactic) -> status,tactic)
         ~disambiguate_command:(fun status (_,_,cmd) -> status,cmd)
         ~disambiguate_macro:(fun _ _ -> assert false)
         status ast
      in
       assert (lemmas=[]);
       status)
    status moo
} and eval_ast = {ea_go = fun ~disambiguate_tactic ~disambiguate_command
~disambiguate_macro ?(do_heavy_checks=false) status
(text,prefix_len,st)
->
  let opts = { do_heavy_checks = do_heavy_checks ; } in
  match st with
  | GrafiteAst.Executable (_,ex) ->
     eval_executable.ee_go ~disambiguate_tactic ~disambiguate_command
      ~disambiguate_macro opts status (text,prefix_len,ex)
  | GrafiteAst.Comment (_,c) -> eval_comment status (text,prefix_len,c),[]
}

let eval_ast = eval_ast.ea_go
