/******************************** LICENSE ********************************


 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)
 
 Licensed under the Apache License, Version 2.0 (the "License"); 
 you may not use this file except in compliance with the License. 
 You may obtain a copy of the License at 
 
 	http://www.apache.org/licenses/LICENSE-2.0
 
 Unless required by applicable law or agreed to in writing, software 
 distributed under the License is distributed on an "AS IS" BASIS, 
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 See the License for the specific language governing permissions and 
 limitations under the License.


 ******************************** LICENSE ********************************/

/*! \file BinaryDriver.cc
    \brief Implementation of BinaryDriver.
    \author Graphics Section, ECMWF

    Started: Mon Jan  4 20:28:15 2010

*/

#include <BinaryDriver.h>
#include <Polyline.h>
#include <Text.h>
#include <Image.h>


using namespace magics;


/*!
  \brief Constructor
*/
BinaryDriver::BinaryDriver()
{
	Log::debug() << "BinaryDriver::BinaryDriver needs implementing." <<endl;
}

/*!
  \brief Destructor
*/
BinaryDriver::~BinaryDriver() 
{
}

/*!
  \brief Opening the driver
*/
void BinaryDriver::open()
{
        const int width = maground(getWidth());
        dimensionX_ = width;
        const float ratio = getYDeviceLength() / getXDeviceLength();
        dimensionY_ = maground(ratio*width);

        setCMscale(float(width)/getXDeviceLength());

	out_.open(getFileName("mag").c_str(),ios::out|ios::binary);
	if( !out_ ) {
		Log::error() << "BinaryDriver: Error opening output stream." << endl;
	}
	const int version = BINARY_VERSION;
	const char mag[7] = "MAGICS";
	out_.write((char *)(&mag), 6);
	out_.write((char *)(&version), sizeof(int));
}

/*!
  \brief Closing the driver
*/
void BinaryDriver::close()
{
	endPage();
	currentPage_ = 0;
}

/*!
  \brief starting a new page

  This method has to take care that previous pages are closed and that
  for formats with multiple output files a new file is set up.
*/
MAGICS_NO_EXPORT void BinaryDriver::startPage() const
{
	if(currentPage_ > 0) endPage();
	
	currentPage_++;
}

/*!
  \brief ending a page
 
  This method has to take care that for formats with multiple output 
  files are closed.
*/
MAGICS_NO_EXPORT void BinaryDriver::endPage() const
{
	out_.close();
}

/*!
  \brief project to a new Layout

  This method will update the offset and scale according to the new Layout given.

  \sa Layout
*/
MAGICS_NO_EXPORT void BinaryDriver::project(const magics::Layout& layout) const
{
        char c = 'P';
        out_.write(&c, 1);
 /*       out_.write((char *)(&coordRatioX_), sizeof(float));
        out_.write((char *)(&coordRatioY_), sizeof(float));
        out_.write((char *)(&offsetX_), sizeof(float));
        out_.write((char *)(&offsetY_), sizeof(float));

        // push current state
        dimensionStack_.push(dimensionX_);
        dimensionStack_.push(dimensionY_);
        offsetsX_.push(offsetX_);
        offsetsY_.push(offsetY_);
        scalesX_.push(coordRatioX_);
        scalesY_.push(coordRatioY_);

        offsetX_    += layout.x()     * 0.01 * dimensionX_;
        offsetY_    -= layout.y()     * 0.01 * dimensionY_;
        dimensionX_ =  layout.width() * 0.01 * dimensionX_;
        dimensionY_ =  layout.height()* 0.01 * dimensionY_;

        const float sumX = layout.maxX() - layout.minX();
        const float sumY = layout.maxY() - layout.minY();

        if( sumX!=0 && sumY!=0 )
        {
                coordRatioX_ = dimensionX_/sumX;
                coordRatioY_ = -dimensionY_/sumY;
        }

        offsetX_ = projectX( -layout.minX());
        offsetY_ = projectY( -layout.minY());
*/
	const double x = layout.x();
	const double y = layout.y();
	const double w = layout.width();
	const double h = layout.height();
	const double minX = layout.minX();
	const double minY = layout.minY();
	const double maxX = layout.maxX();
	const double maxY = layout.maxY();
	
        out_.write((char *)(&x), sizeof(double));
        out_.write((char *)(&y), sizeof(double));
        out_.write((char *)(&w), sizeof(double));
        out_.write((char *)(&h), sizeof(double));
	out_.write((char *)(&minX), sizeof(double));
	out_.write((char *)(&minY), sizeof(double));
	out_.write((char *)(&maxX), sizeof(double));
        out_.write((char *)(&maxY), sizeof(double));
}

/*!
  \brief reproject out of the last Layout

  This method will update the offset and scale to the state they were before the
  last Layout was received.

*/
MAGICS_NO_EXPORT void BinaryDriver::unproject() const
{
	char c = 'U';
        out_.write(&c, 1);
  /*      dimensionY_ = dimensionStack_.top();dimensionStack_.pop();
        dimensionX_ = dimensionStack_.top();dimensionStack_.pop();
        offsetX_ = offsetsX_.top();offsetsX_.pop();
        offsetY_ = offsetsY_.top();offsetsY_.pop();
        coordRatioX_  = scalesX_.top(); scalesX_.pop();
        coordRatioY_  = scalesY_.top(); scalesY_.pop();
        */
}


/*!
  \brief sets a new colour

  This colour stays the default drawing colour until the painting in the 
  current box is finished.

  \sa Colour
*/
MAGICS_NO_EXPORT void BinaryDriver::setNewColour(const Colour &colour) const
{
        const float r=colour.red();
        const float g=colour.green();
        const float b=colour.blue();
        char c = 'C';
        out_.write(&c, 1);
        out_.write((char *)(&r), sizeof(float));
        out_.write((char *)(&g), sizeof(float));
        out_.write((char *)(&b), sizeof(float));
}

/*!
  \brief sets a new line width

  This line width stays the default width until the painting in the 
  current box is finished.

  \sa setLineParameters()
*/
MAGICS_NO_EXPORT void BinaryDriver::setNewLineWidth(const float width) const
{
        char c = 'W';
        out_.write(&c, 1);
        out_.write((char *)(&width), sizeof(float));
}

/*!
  \brief sets new properties of how lines are drawn

  These properties stay the default until the painting in the 
  current box is finished.

  \sa LineStyle

  \param linestyle Object describing the line style
  \param w width of the line

*/
MAGICS_NO_EXPORT int BinaryDriver::setLineParameters(const LineStyle linestyle, const float w) const
{
        char c = 'L';
        out_.write(&c, 1);
        out_.write((char *)(&linestyle), sizeof(LineStyle));
        out_.write((char *)(&w), sizeof(float));
	return 0;
}

#include <Arrow.h>
#include <Flag.h>

void BinaryDriver::renderWindArrow(const Arrow &arrow) const
{
  /*
    arrow.getScale()
    arrow.getThickness()
    const LineStyle style = arrow.getStyle();
    const ArrowPosition pos = arrow.getArrowPosition();
  */
	char c = 'A';
	out_.write(&c, 1);
        const int no = arrow.size();
	out_.write((char *)(&no), sizeof(int));
        const double sc  = arrow.getScale();
	out_.write((char *)(&sc), sizeof(double));
        const int index  = arrow.getHeadIndex();
	out_.write((char *)(&index), sizeof(int));
	const LineStyle ls = arrow.getStyle();
	out_.write((char *)(&ls), sizeof(LineStyle));
	ArrowPosition ap = arrow.getArrowPosition();
	out_.write((char *)(&ap), sizeof(ArrowPosition));
	const int hi = arrow.getHeadIndex();
	out_.write((char *)(&hi), sizeof(int));
	const double hr = arrow.getHeadRatio();
	out_.write((char *)(&hr), sizeof(double));

	Colour colour = arrow.getColour();
	const float r=colour.red();
        const float g=colour.green();
        const float b=colour.blue();
        out_.write((char *)(&r), sizeof(float));
        out_.write((char *)(&g), sizeof(float));
        out_.write((char *)(&b), sizeof(float));

        Arrow::const_iterator arr = arrow.begin();

        for(int pts=0;pts<no;pts++)
        {
          const double x = arr->x_;
	  out_.write((char *)(&x), sizeof(double));
          const double y = arr->y_;
	  out_.write((char *)(&y), sizeof(double));
          const PaperPoint p = arr->point_;
	  out_.write((char *)(&p), sizeof(PaperPoint));
	  ++arr;
        }
}

void BinaryDriver::renderWindFlag(const Flag &flag) const
{

}


/*!
  \brief renders polylines

  This method renders a polyline given as two float arrays. The two 
  arrays given as X and Y values have to be at least the length of
  <i>n</i>. All values beyond <i>n</i> will be ignored. The style is
  determined by what is described in the current LineStyle.

  \sa setLineParameters()
  \param n number of points
  \param x array of x values
  \param y array of y values
*/
MAGICS_NO_EXPORT void BinaryDriver::renderPolyline(const int n, float *x, float *y) const
{
        char c = 'H';
	out_.write(&c, 1);
	out_.write((char *)(&n), sizeof(int));
	out_.write((char *)(x),  sizeof(float)*n);
	out_.write((char *)(y),  sizeof(float)*n);
}

/*!
  \brief renders a single line

  This method renders a polyline with two points.The style is
  determined by what is described in the current LineStyle.

  \sa setLineParameters()
  \param n number of points
  \param x array of x values
  \param y array of y values
*/
MAGICS_NO_EXPORT void BinaryDriver::renderPolyline2(const int n, float* x, float* y) const
{
        char c = 'B';
        out_.write(&c, 1);
        out_.write((char *)(&n), sizeof(int));
        out_.write((char *)(x),  sizeof(float)*n);
        out_.write((char *)(y),  sizeof(float)*n);
}

/*!
  \brief renders a filled polygon

  This method renders a filled polygon. The style is
  determined by what is described in the current LineStyle.

  \sa setLineParameters()
  \param n number of points
  \param x array of x values
  \param y array of y values
*/
MAGICS_NO_EXPORT void BinaryDriver::renderSimplePolygon(const int n, float* x, float* y) const
{
        char c = 'S';
        out_.write(&c, 1);
        out_.write((char *)(&n), sizeof(int));
        out_.write((char *)(x),  sizeof(float)*n);
        out_.write((char *)(y),  sizeof(float)*n);
}

/*!
  \brief renders text strings

  This method renders given text strings.

  \sa Text
  \param text object containing the strings and their description
*/
MAGICS_NO_EXPORT void BinaryDriver::renderText(const Text& text) const
{
        char c = 'T';
        out_.write(&c, 1);

        if(text.empty()) return;
        const vector<NiceText>& niceT = text.getNiceText();
        if(niceT.empty()) return;

        vector<NiceText>::const_iterator niceText = text.textBegin();
//        vector<NiceText>::const_iterator niceTextEnd = text.textEnd();
//        for(;niceText<niceTextEnd;niceText++)
        {
            const float x = text[0].x();
            const float y = text[0].y();
            out_.write((char *)(&x),sizeof(float));
            out_.write((char *)(&y),sizeof(float));
            const float an = text.getAngle();
            out_.write((char *)(&an),sizeof(float));
            MagFont magfont = (*niceText).font();
            const float s = magfont.size();
            out_.write((char *)(&s),sizeof(float));
            bool bl = text.getBlanking();
            out_.write((char *)(&bl),sizeof(bool));

            const enum Justification horizontal = text.getJustification();
            const enum VerticalAlign vertical   = text.getVerticalAlign();
            out_.write((char *)(&horizontal),sizeof(enum Justification));
            out_.write((char *)(&vertical),  sizeof(enum VerticalAlign));

            const string t=(*niceText).text();
            const int len = t.length();
            out_.write((char *)(&len),sizeof(int));

            char pp[len];
            strcpy(pp, t.c_str());

            out_.write(pp,sizeof(char)*len);
        }
}

/*!
  \brief drawing a circle

  This method renders given text strings.

  The meaning of the last parameter <i>s</i> is as follows:
     - 0-8 determines how many quarters of the circle are filled. Starting from the top clock-wise.
     - 9 fills the whole circle but leaves a vertical bar empty in the middle of the circle.

  \param x X Position
  \param y Y Position
  \param r Radius of circle
  \param s Style which determines how the circle is shaded
*/
MAGICS_NO_EXPORT void BinaryDriver::circle(const float x, const float y, const float r, const int s) const
{
	Log::debug() << "BinaryDriver::circle needs implementing." <<endl;
}

/*!
  \brief render pixmaps

  This method renders pixmaps. These are used for cell shading and raster input (GIFs and PNGs).

  \sa renderCellArray()

  \param x0 x of lower corner
  \param y0 y of lower corner
  \param x1 x of higher corner
  \param y1 y of higher corner
  \param w width of pixmap
  \param h height of pixmap
  \param pixmap contents
  \param landscape says if contents is landscape

*/
MAGICS_NO_EXPORT bool BinaryDriver::renderPixmap(float x0,float y0,float x1,float y1,
                                            int w,int h,unsigned char* pixmap,int landscape, bool) const
{
	Log::debug() << "BinaryDriver::renderPixmap needs implementing." <<endl;
	return true;
}

/*!
  \brief render cell arrays

  This method renders cell arrays, also called images in Magics language. These are 
  mainly used for satellite data.

  \sa renderPixmap()

  \param image Object containing an image
*/
MAGICS_NO_EXPORT bool BinaryDriver::renderCellArray(const Image& image) const
{
	Log::debug() << "BinaryDriver::renderCellArray needs implementing." <<endl;
	return true;
}



/*!
  \brief prints debug output

  When Magics++ is compiled in debug mode these extra strings are printed.

  \note This can increase file and log file sizes if you run Magics++ in debug mode!

  \param s string to be printed
*/
MAGICS_NO_EXPORT void BinaryDriver::debugOutput(const std::string &s) const
{
	Log::debug() << s << endl;
}

/*!
  \brief class information are given to the output-stream
*/
void BinaryDriver::print(ostream& out)  const
{
	out << "BinaryDriver[";
	out << "]";
}

static SimpleObjectMaker<BinaryDriver, BaseDriver> Binary_driver("Binary");

