 
#include "TePDIEuclidianBlendStrategy.hpp"
#include "TePDIBlending.hpp"
#include "TePDIUtils.hpp"
#include "TePDIPIManager.hpp"
#include <TeAgnostic.h>

#include <TeUtils.h>
#include <TeGeometryAlgorithms.h>
#include <TeDefines.h>
#include <TeOverlayUtils.h>
#include <TeOverlay.h>

#include <float.h>


TePDIEuclidianBlendStrategy::TePDIEuclidianBlendStrategy()
{
};      

TePDIEuclidianBlendStrategy::~TePDIEuclidianBlendStrategy()
{
};


bool TePDIEuclidianBlendStrategy::CheckParameters( 
  const TePDIParameters& parameters ) const
{
  /* Checking input_raster1 */
  
  TePDITypes::TePDIRasterPtrType input_raster1;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "input_raster1", 
    input_raster1 ),
    "Missing parameter: input_raster1" );
  TEAGN_TRUE_OR_RETURN( input_raster1.isActive(),
    "Invalid parameter: input_raster1 inactive" );
  TEAGN_TRUE_OR_RETURN( input_raster1->params().status_ != 
    TeRasterParams::TeNotReady, "Invalid parameter: input_raster1 not ready" );    
    
  /* Checking input_raster2 */
  
  TePDITypes::TePDIRasterPtrType input_raster2;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "input_raster2", 
    input_raster2 ),
    "Missing parameter: input_raster2" );
  TEAGN_TRUE_OR_RETURN( input_raster2.isActive(),
    "Invalid parameter: input_raster2 inactive" );
  TEAGN_TRUE_OR_RETURN( input_raster2->params().status_ != 
    TeRasterParams::TeNotReady, "Invalid parameter: input_raster2 not ready" );    
    
  /* Checking output_raster */
  
  TePDITypes::TePDIRasterPtrType output_raster;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "output_raster", 
    output_raster ),
    "Missing parameter: output_raster" );
  TEAGN_TRUE_OR_RETURN( output_raster.isActive(),
    "Invalid parameter: output_raster inactive" );
  TEAGN_TRUE_OR_RETURN( output_raster->params().status_ != 
    TeRasterParams::TeNotReady, "Invalid parameter: output_raster not ready" );    
    
  /* channels1 parameter checking */

  std::vector< int > channels1;
  TEAGN_TRUE_OR_RETURN( 
    parameters.GetParameter( "channels1", channels1 ), 
    "Missing parameter: channels1" );
  for( unsigned int channels1_index = 0 ; 
    channels1_index < channels1.size() ; 
    ++channels1_index ) {
    
    TEAGN_TRUE_OR_RETURN( ( channels1[ channels1_index ] >= 0 ) &&
      ( channels1[ channels1_index ] < input_raster1->nBands() ),
      "Invalid parameter: channels1" );
  }
  
  /* channels2 parameter checking */

  std::vector< int > channels2;
  TEAGN_TRUE_OR_RETURN( 
    parameters.GetParameter( "channels2", channels2 ), 
    "Missing parameter: channels2" );
  TEAGN_TRUE_OR_RETURN( ( channels2.size() == channels1.size() ),
    "Size mismatch between channels1 and channels2" );
  for( unsigned int channels2_index = 0 ; 
    channels2_index < channels2.size() ; 
    ++channels2_index ) {
    
    TEAGN_TRUE_OR_RETURN( ( channels2[ channels2_index ] >= 0 ) &&
      ( channels2[ channels2_index ] < input_raster2->nBands() ),
      "Invalid parameter: channels2" );
  }      
    
  /* Checking raster polygons */
  
  TePDITypes::TePDIPolygonPtrType raster1_pol_ptr;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "raster1_pol_ptr", 
    raster1_pol_ptr ), "Missing parameter : raster1_pol_ptr" );
  TEAGN_TRUE_OR_RETURN( raster1_pol_ptr.isActive(),
    "Invalid parameter : raster1_pol_ptr" )
  TEAGN_TRUE_OR_RETURN(
    ( ! input_raster1->begin( *raster1_pol_ptr, TeBoxPixelIn, 
    0 ).end() ), "Invalid parameter : raster1_pol_ptr" )
  TEAGN_TRUE_OR_RETURN( ( raster1_pol_ptr->size() == 1 ),
    "Invalid parameter : raster1_pol_ptr" )
    
  TePDITypes::TePDIPolygonPtrType raster2_pol_ptr;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "raster2_pol_ptr", 
    raster2_pol_ptr ), "Missing parameter : raster2_pol_ptr" );
  TEAGN_TRUE_OR_RETURN( raster2_pol_ptr.isActive(),
    "Invalid parameter : raster2_pol_ptr" )
  TEAGN_TRUE_OR_RETURN(
    ( ! input_raster2->begin( *raster2_pol_ptr, TeBoxPixelIn, 
    0 ).end() ), "Invalid parameter : raster2_pol_ptr" )
  TEAGN_TRUE_OR_RETURN( ( raster2_pol_ptr->size() == 1 ),
    "Invalid parameter : raster2_pol_ptr" )
    
  /* Checking raster2 polygon offsets */
  
  double raster2_pol_offset_x = 0;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "raster2_pol_offset_x", 
    raster2_pol_offset_x ), "Missing parameter : raster2_pol_offset_x" );  

  double raster2_pol_offset_y = 0;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "raster2_pol_offset_y", 
    raster2_pol_offset_y ), "Missing parameter : raster2_pol_offset_y" );     
    
  return true;
}


bool TePDIEuclidianBlendStrategy::Implementation( 
  const TePDIParameters& params )
{
  /* Extracting parameters */

  TePDITypes::TePDIRasterPtrType input_raster1;
  params.GetParameter( "input_raster1", input_raster1 );
  TeRaster& input_raster1_ref = *( input_raster1.nakedPointer() );

  TePDITypes::TePDIRasterPtrType input_raster2;
  params.GetParameter( "input_raster2", input_raster2 );
  TeRaster& input_raster2_ref = *( input_raster2.nakedPointer() );

  TePDITypes::TePDIRasterPtrType output_raster;
  params.GetParameter( "output_raster", output_raster );
  TeRaster& output_raster_ref = *( output_raster.nakedPointer() );
  
  std::vector< int > channels1;
  params.GetParameter( "channels1", channels1 );

  std::vector< int > channels2;
  params.GetParameter( "channels2", channels2 ); 
  
  TePDITypes::TePDIPolygonPtrType raster1_pol_ptr;
  params.GetParameter( "raster1_pol_ptr", raster1_pol_ptr );
  TePolygon& raster1_polygon = ( *raster1_pol_ptr );

  TePDITypes::TePDIPolygonPtrType raster2_pol_ptr;
  params.GetParameter( "raster2_pol_ptr", raster2_pol_ptr );
  TePolygon& raster2_polygon = ( *raster2_pol_ptr );
  
  double raster2_pol_offset_x = 0;
  params.GetParameter( "raster2_pol_offset_x", raster2_pol_offset_x );

  double raster2_pol_offset_y = 0;
  params.GetParameter( "raster2_pol_offset_y", raster2_pol_offset_y );

  
  /* Dumyy value definition */
  
  bool output_raster_uses_dummy = output_raster->params().useDummy_;
  double output_raster_dummy_value = 0;
  
  if( output_raster_uses_dummy ) {
    output_raster_dummy_value = output_raster->params().dummy_[ 0 ];
  }
  
  if( params.CheckParameter< double >( "dummy_value" ) ) {
    
    params.GetParameter( "dummy_value", output_raster_dummy_value );
    
    output_raster_uses_dummy = true;
  }  
  
  /* Blending lines draw */
  
  bool draw_blend_lines = false;
  double blend_lines_value = 0;
  
  if( params.CheckParameter< double >( "draw_blend_lines" ) ) {
    
    params.GetParameter( "draw_blend_lines", blend_lines_value );
    
    draw_blend_lines = true;
  }  
  
  /* Calculating intersecion polygons data */
  
  TePolygon new_raster1_polygon;
  TePolygon new_raster2_polygon;
  TePolygon int_pol_refout;
  TePolygon int_pol_ref2;
  double raster1_rel_index_offset_x = 0;
  double raster1_rel_index_offset_y = 0;
  double raster2_rel_index_offset_x = 0;
  double raster2_rel_index_offset_y = 0;
  short pols_relation = 0;

  TEAGN_TRUE_OR_RETURN( TePDIBlending::extractPolygons( params, 
    new_raster1_polygon, new_raster2_polygon, int_pol_refout, int_pol_ref2,
    pols_relation,
    raster1_rel_index_offset_x, raster1_rel_index_offset_y,
    raster2_rel_index_offset_x, raster2_rel_index_offset_y ),
    "Error extracting intersection polygons" );
  
  /* Global vars */
  
  TeRaster::iteratorPoly intersection_refout_it =
    output_raster->begin( int_pol_refout, TeBoxPixelIn, 0 );
  TEAGN_TRUE_OR_RETURN( ( ! intersection_refout_it.end() ), 
    "Unable to create an iterator over raster2 area" );     
  
  TePDIPIManager progress( "Blending intersection...", 
    intersection_refout_it.nLinesInPoly() * channels1.size(), 
    progress_interface_enabled_ );   
    
  /* Blending intersection */
  { 
    unsigned int current_line = 0;
    unsigned int current_col = 0;
    unsigned int last_line = 0;
    
    double value1 = 0;      
    double value2 = 0;      
    double value3 = 0;
    
    bool got_raster1_element = false;
    bool got_raster2_element = false;     
    
    int raster1_offset_x = TeRound( raster1_rel_index_offset_x );
    int raster1_offset_y = TeRound( raster1_rel_index_offset_y );    
    int raster2_offset_x = TeRound( raster2_rel_index_offset_x );
    int raster2_offset_y = TeRound( raster2_rel_index_offset_y );    
    
    if( ( pols_relation == TeCONTAINS ) || ( pols_relation == TeCOVERS ) ||
        ( pols_relation == TeWITHIN ) || ( pols_relation == TeCOVEREDBY ) ) {
        
      /* This is the case where one raster is inside onother */

      for( unsigned int channels_index = 0 ; 
        channels_index < channels1.size() ;
        ++channels_index ) {
    
        intersection_refout_it =
          output_raster->begin( int_pol_refout, TeBoxPixelIn, 0 );
        TEAGN_TRUE_OR_RETURN( ( ! intersection_refout_it.end() ), 
          "Unable to create an iterator over raster2 area" );    
      
        const unsigned int channel1 = channels1[ channels_index ];
        const unsigned int channel2 = channels2[ channels_index ];              
      
        while( ! intersection_refout_it.end() ) {
          current_line = intersection_refout_it.currentLine();
          current_col = intersection_refout_it.currentColumn();    
          
          if( input_raster1_ref.getElement( current_col + raster1_offset_x, 
              current_line + raster1_offset_y, value1, channel1 ) ) {
                  
            if( output_raster_uses_dummy &&
              ( value1 == output_raster_dummy_value ) ) {
            
              got_raster1_element = false;
            } else {
              got_raster1_element = true;
            }
          } else {
            got_raster1_element = false;
          }
                
          if( input_raster2_ref.getElement( current_col + raster2_offset_x, 
              current_line + raster2_offset_y, value2, channel2 ) ) {
                  
            if( output_raster_uses_dummy &&
              ( value2 == output_raster_dummy_value ) ) {
                
              got_raster2_element = false;
            } else {
              got_raster2_element = true;
            }
          } else {
            got_raster2_element = false;
          }          
        
          if( got_raster1_element && got_raster2_element ) {
            value3 = ( value1 + value2 ) / 2;
            
            if( ( ! output_raster_uses_dummy ) || 
              ( value3 != output_raster_dummy_value ) ) {
                  
              TEAGN_TRUE_OR_RETURN( 
                output_raster_ref.setElement( current_col, current_line, value3, 
                channels_index ), "Unable to write to output raster" );
            }            
          } else if( got_raster1_element ) {
            if( ( ! output_raster_uses_dummy ) || 
              ( value1 != output_raster_dummy_value ) ) {
                  
              TEAGN_TRUE_OR_RETURN( 
                output_raster_ref.setElement( current_col, current_line, value1, 
                channels_index ), "Unable to write to output raster" );
            }          
          } else {
            if( ( ! output_raster_uses_dummy ) || 
              ( value2 != output_raster_dummy_value ) ) {
                
              TEAGN_TRUE_OR_RETURN( 
                output_raster_ref.setElement( current_col, current_line, value2, 
                channels_index ), "Unable to write to output raster" );
            }          
          }
          
          if( current_line != last_line ) {
            last_line = current_line;
            progress.Increment();
          }    
        
          ++intersection_refout_it;
        }
      }
    } else {
    /* Raster1 positioning */
    
      double raster1_abs_index_offset_x = 0;// output raster */
      double raster1_abs_index_offset_y = 0;
      {
        TeBox pol1_box_indexed;
        TePDIUtils::MapCoords2RasterIndexes( raster1_polygon.box(),
          input_raster1, pol1_box_indexed );
              
        double r1_min_x = MIN( pol1_box_indexed.x1(),
          pol1_box_indexed.x2() );
        double r1_min_y = MIN( pol1_box_indexed.y1(),
          pol1_box_indexed.y2() );      
    
        TeCoord2D r1_UL_ref1_indexed( r1_min_x, r1_min_y );  
        TeCoord2D r1_UL_ref1( input_raster1->index2Coord( r1_UL_ref1_indexed ) );  
        TeCoord2D r1_UL_refout_indexed( output_raster->coord2Index( r1_UL_ref1 ) );
            
        raster1_abs_index_offset_x = r1_UL_refout_indexed.x();  
        raster1_abs_index_offset_y = r1_UL_refout_indexed.y();
      }      

      /* Calculating the raster 2 polygon with output raster reference */
    
      TePolygon r2_pol_refout;
      {
        TePolygon r2_pol_ref2_indexed;
        TePDIUtils::MapCoords2RasterIndexes( raster2_polygon,
          input_raster2, r2_pol_ref2_indexed );
          
        double r2_min_x = MIN( r2_pol_ref2_indexed.box().x1(),
          r2_pol_ref2_indexed.box().x2() );
        double r2_min_y = MIN( r2_pol_ref2_indexed.box().y1(),
          r2_pol_ref2_indexed.box().y2() );
    
        TePolygon r2_pol_refout_indexed = r2_pol_ref2_indexed;
            
        for( unsigned int lr_index = 0 ; 
          lr_index < r2_pol_refout_indexed.size() ; ++lr_index ) {
          for( unsigned int c2d_index = 0 ; 
            c2d_index < r2_pol_refout_indexed[ lr_index ].size() ; 
            ++c2d_index ) {
                
            r2_pol_refout_indexed[ lr_index ][ c2d_index ].x( 
              r2_pol_refout_indexed[ lr_index ][ c2d_index ].x() -
              r2_min_x + raster1_abs_index_offset_x + raster2_pol_offset_x );
            r2_pol_refout_indexed[ lr_index ][ c2d_index ].y( 
              r2_pol_refout_indexed[ lr_index ][ c2d_index ].y() -
              r2_min_y + raster1_abs_index_offset_y + raster2_pol_offset_y );          
          }
        }
        
        TePDIUtils::MapRasterIndexes2Coords( r2_pol_refout_indexed,
          output_raster, r2_pol_refout );
      }    
    
      /* Calculating the two blending lines - output raster indexed reference*/

      bool x_axis_parallels = false;
      bool y_axis_parallels = false;
      double line1_a = 0;
      double line1_b = 0;
      double line1_c = 0;
      double line1_x = 0;
      double line1_y = 0;
      double line2_a = 0;
      double line2_b = 0;
      double line2_c = 0;
      double line2_x = 0;
      double line2_y = 0;
      
      TePolygon raster1_polygon_indexed;
      TePDIUtils::MapCoords2RasterIndexes( raster1_polygon,
        output_raster, raster1_polygon_indexed );      
   
      TePolygon r2_pol_refout_indexed;
      TePDIUtils::MapCoords2RasterIndexes( r2_pol_refout,
        output_raster, r2_pol_refout_indexed );      
          
      TePolygon int_pol_refout_indexed;
      TePDIUtils::MapCoords2RasterIndexes( int_pol_refout,
        output_raster, int_pol_refout_indexed );        
          
      generateBlendingLines( raster1_polygon_indexed, r2_pol_refout_indexed, 
        int_pol_refout_indexed, 
        line1_a, line1_b, line1_c,
        line2_a, line2_b, line2_c );
        
      if( line1_a == 0.0 ) {
        x_axis_parallels = true;
        line1_y = ( -1 ) * ( line1_c / line1_b );
        line2_y = ( -1 ) * ( line2_c / line2_b );
      } else if( line1_b == 0.0 ) {
        y_axis_parallels = true;
        line1_x = ( -1 ) * ( line1_c / line1_a );
        line2_x = ( -1 ) * ( line2_c / line2_a );
      }
      
      /* Blending intersection */  
    
      double dist1 = 0;   
      double dist2 = 0;
      const double const_hip1 = sqrt( ( line1_a * line1_a ) + 
        ( line1_b * line1_b ) );
      const double const_hip2 = sqrt( ( line2_a * line2_a ) + 
        ( line2_b * line2_b ) );
        
      for( unsigned int channels_index = 0 ; 
        channels_index < channels1.size() ;
        ++channels_index ) {
    
        intersection_refout_it =
          output_raster->begin( int_pol_refout, TeBoxPixelIn, 0 );
        TEAGN_TRUE_OR_RETURN( ( ! intersection_refout_it.end() ), 
          "Unable to create an iterator over raster2 area" );    
      
        const unsigned int channel1 = channels1[ channels_index ];
        const unsigned int channel2 = channels2[ channels_index ];        
    
        while( ! intersection_refout_it.end() ) {
          current_line = intersection_refout_it.currentLine();
          current_col = intersection_refout_it.currentColumn();
          
          if( input_raster1_ref.getElement( current_col + raster1_offset_x, 
              current_line + raster1_offset_y, value1, channel1 ) ) {
                  
            if( output_raster_uses_dummy &&
              ( value1 == output_raster_dummy_value ) ) {
            
              got_raster1_element = false;
            } else {
              got_raster1_element = true;
            }
          } else {
            got_raster1_element = false;
          }
                
          if( input_raster2_ref.getElement( current_col + raster2_offset_x, 
              current_line + raster2_offset_y, value2, channel2 ) ) {
                  
            if( output_raster_uses_dummy &&
              ( value2 == output_raster_dummy_value ) ) {
                
              got_raster2_element = false;
            } else {
              got_raster2_element = true;
            }
          } else {
            got_raster2_element = false;
          }        
            
          if( got_raster1_element && got_raster2_element ) {
            if( x_axis_parallels ) {
              dist1 = ABS( line1_y - ( (double)current_line ) );
              dist2 = ABS( line2_y - ( (double)current_line ) );
            } else if( y_axis_parallels ) {
              dist1 = ABS( line1_x - ( (double)current_col ) );
              dist2 = ABS( line2_x - ( (double)current_col ) );
            } else {
              dist1 = ABS( ( ( line1_a * ( (double)current_line ) ) +
                ( line1_b * ( (double)current_col) ) + line1_c ) / 
                const_hip1 );
              dist2 = ABS( ( ( line2_a * ( (double)current_line ) ) +
                ( line2_b * ( (double)current_col) ) + line2_c ) / 
                const_hip2 );
            }
          
            if( dist1 == 0.0 ) {
              value3 = value1;
            } else if( dist2 == 0.0 ) {
              value3 = value2;
            } else {
              value3 = ( ( dist1 * value2 ) + ( dist2 * value1 ) ) /
                ( dist1 + dist2 );
            }
                
            TEAGN_TRUE_OR_RETURN( 
              output_raster_ref.setElement( current_col, 
              current_line,
              value3, channels_index ), "Unable to write to output raster" );                 
          } else if( got_raster1_element ) {
            TEAGN_TRUE_OR_RETURN( 
              output_raster_ref.setElement( current_col, 
              current_line,
              value1, channels_index ), "Unable to write to output raster" );
          } else if( got_raster2_element ) {
            TEAGN_TRUE_OR_RETURN( 
              output_raster_ref.setElement( current_col, 
              current_line,
              value2, channels_index ), "Unable to write to output raster" );
          }        
          
          ++intersection_refout_it;
          
          if( current_line != last_line ) {
            last_line = current_line;
            progress.Increment();
          }
        }   
        
        if( draw_blend_lines ) {
          drawLine( output_raster, channels_index, blend_lines_value,
            line1_a, line1_b, line1_c );
          drawLine( output_raster, channels_index, blend_lines_value,
            line2_a, line2_b, line2_c );
        }
      }
    } 
  }
  
  return true;
}


void TePDIEuclidianBlendStrategy::generateBlendingLines( 
  const TePolygon& polygon1,  
  const TePolygon& polygon2, const TePolygon& inter_pol, 
  double& line1_a, double& line1_b, double& line1_c,
  double& line2_a, double& line2_b, double& line2_c )
{
  const TeCoord2D centroid1 = TeFindCentroid( polygon1 );
  const TeCoord2D centroid2 = TeFindCentroid( polygon2 );
  const TeCoord2D inter_pol_centroid = TeFindCentroid( inter_pol );
  
  TePointSet pols_inter_points;
  {
    TeLineSet ls1;
    TePDIUtils::makeSegmentSet( polygon1, ls1 );
    
    TeLineSet ls2;
    TePDIUtils::makeSegmentSet( polygon2, ls2 );    
    
    TePDIUtils::TeSegSetIntersection( ls1, ls2, pols_inter_points );
  }
  
  if( centroid1.x() == centroid2.x() ) {
    /* Centroids are lined up ( y axis ) then
       blending lines are X parallels */
    
    if( centroid1.y() > centroid2.y() ) {
      line1_a = 0.0;
      line1_b = -1.0;
      line1_c = inter_pol.box().y2() ;
      
      line2_a = 0.0;
      line2_b = -1.0;
      line2_c = inter_pol.box().y1() ;
    } else {
      line1_a = 0.0;
      line1_b = -1.0;
      line1_c = inter_pol.box().y1() ;
      
      line2_a = 0.0;
      line2_b = -1.0;
      line2_c = inter_pol.box().y2() ;
    }
  } else if( centroid1.y() == centroid2.y() ) {
    /* Centroids are lined up ( x axis ) then
       blending lines are Y parallels */
      
    if( centroid1.x() > centroid2.x() ) {
      line1_a = -1.0;
      line1_b = 0.0;
      line1_c = inter_pol.box().x2() ;
      
      line2_a = -1.0;
      line2_b = 0.0;
      line2_c = inter_pol.box().x1() ;
    } else {
      line1_a = -1.0;
      line1_b = 0.0;
      line1_c = inter_pol.box().x1() ;
      
      line2_a = -1.0;
      line2_b = 0.0;
      line2_c = inter_pol.box().x2() ;
    } 
    
  } else if( pols_inter_points.size() == 2 ) {
    /* This is the case where the intersection of the two polygons
       occurs on just two points, these points will be used as
       guides for the blending lines */
  
    const TeCoord2D& inter_point1 = pols_inter_points[ 0 ].elem();
    const TeCoord2D& inter_point2 = pols_inter_points[ 1 ].elem();
  
    if( inter_point1.x() == inter_point2.x() ) {
      /* Intersection points are lined up ( y axis ) then
         blending lines are Y parallels */
         
      if( centroid1.x() > centroid2.x() ) {
        line1_a = -1.0;
        line1_b = 0.0;
        line1_c = inter_pol.box().x2() ;
        
        line2_a = -1.0;
        line2_b = 0.0;
        line2_c = inter_pol.box().x1() ;
      } else {
        line1_a = -1.0;
        line1_b = 0.0;
        line1_c = inter_pol.box().x1() ;
        
        line2_a = -1.0;
        line2_b = 0.0;
        line2_c = inter_pol.box().x2() ;
      }          
    } else if( inter_point1.y() == inter_point2.y() ) {
      /* Intersection points are lined up ( x axis ) then
         blending lines are X parallels */
        
      if( centroid1.y() > centroid2.y() ) {
        line1_a = 0.0;
        line1_b = -1.0;
        line1_c = inter_pol.box().y2() ;
        
        line2_a = 0.0;
        line2_b = -1.0;
        line2_c = inter_pol.box().y1() ;
      } else {
        line1_a = 0.0;
        line1_b = -1.0;
        line1_c = inter_pol.box().y1() ;
        
        line2_a = 0.0;
        line2_b = -1.0;
        line2_c = inter_pol.box().y2() ;
      }
    } else {
      /* The intersection points are NOT lined up */
    
      /* Blending lines angular coef */
    
      const double blend_lines_m = ( inter_point2.y() - inter_point1.y() ) /
        ( inter_point2.x() - inter_point1.x() ); 
        
      /* generating the equation for the parallel line passing 
         throught the intersection polygon centroid */
         
      double ip_cent_para_line_a = 0;
      double ip_cent_para_line_b = 0;
      double ip_cent_para_line_c = 0;
      {
        double ip_centroid_para_line_q = inter_pol_centroid.y() -
        ( blend_lines_m * inter_pol_centroid.x() ); 
        
        TeCoord2D aux_point( ( inter_pol_centroid.x() + 10 ), 
          ( blend_lines_m * ( inter_pol_centroid.x() + 10 ) ) + 
          ip_centroid_para_line_q );
          
        ip_cent_para_line_a = inter_pol_centroid.y() - aux_point.y();
        ip_cent_para_line_b = aux_point.x() - inter_pol_centroid.x();
        ip_cent_para_line_c = ( inter_pol_centroid.x() * aux_point.y() ) - 
          ( aux_point.x() * inter_pol_centroid.y() );          
      }        
    
      /* perpendicular line passing throught intersection polygon centroid 
         coeficients  */      
    
      const double ipc_perp_line_m = ( -1.0 ) / blend_lines_m;    
      
      double ipc_perp_line_q = inter_pol_centroid.y() -
        ( ipc_perp_line_m * inter_pol_centroid.x() );      
      
      /* Projecting each intersection polygon point over the perpendicular 
         line passing throught intersection polygon centroid */
      
      std::vector< TeCoord2D > proj_points;
      
      for( unsigned int lr_index = 0 ; lr_index < inter_pol.size() ; 
        ++lr_index  ) {
        for( unsigned int coord_index = 0 ; 
          coord_index < inter_pol[ lr_index ].size() ; ++coord_index  ) {
          
          /* projection line linear coef passing throught the current point */
        
          double proj_line_q = inter_pol[ lr_index ][ coord_index ].y() -
            ( blend_lines_m * inter_pol[ lr_index ][ coord_index ].x() );
            
          /* Calculating intersection between the projection line and
             the perpendicular line passing throught the intersection
             polygon centroid */
          
          TeCoord2D int_point;
          int_point.x( ( proj_line_q - ipc_perp_line_q ) / 
            ( ipc_perp_line_m - blend_lines_m  ) );
          int_point.y( ( ipc_perp_line_m * int_point.x() ) + ipc_perp_line_q );
          
          proj_points.push_back( int_point );
        }
      }
      
      TEAGN_TRUE_OR_THROW( ( proj_points.size() > 1 ), 
        "Invalid projected points" );      
        
      /* Finding the two pivots for the two blending lines */
    
      TeCoord2D pivot1; /* related to polygon 1 */
      TeCoord2D pivot2; /* related to polygon 2 */
      {
        /* Taking the two points  with the higher distance one to another */
        
        TeCoord2D proj_point1 = proj_points[ 0 ];
        TeCoord2D proj_point2 = proj_points[ 1 ];
        double higher_dist = TeDistance( proj_point1, proj_point2 );
        
        for( unsigned int proj_points_index = 2 ; 
          proj_points_index < proj_points.size() ; ++proj_points_index ) {
          
          double dist1 = TeDistance( proj_points[ proj_points_index ], 
            proj_point1 );
          double dist2 = TeDistance( proj_points[ proj_points_index ], 
            proj_point2 );
            
          if( dist1 > dist2 ) {
            if( dist1 > higher_dist ) {
              proj_point2 = proj_points[ proj_points_index ];
              higher_dist = dist1;
            }
          } else {
            if( dist2 > higher_dist ) {
              proj_point1 = proj_points[ proj_points_index ];
              higher_dist = TeDistance( proj_points[ proj_points_index ], 
                proj_point2 );
            }
          }
        }
        
        /* Association of the projected points with the correct polygons */     
      
        {
          double rel_pos_proj_point1 = ( ip_cent_para_line_a * proj_point1.x() ) 
            + ( ip_cent_para_line_b * proj_point1.y() ) + ip_cent_para_line_c;
          double rel_pos_centroid1 = ( ip_cent_para_line_a * centroid1.x() ) 
            + ( ip_cent_para_line_b * centroid1.y() ) + ip_cent_para_line_c;
           
          if( ( rel_pos_proj_point1 > 0 ) && ( rel_pos_centroid1 > 0 ) ) {
            pivot1 = proj_point1;
            pivot2 = proj_point2;        
          } else {
            pivot2 = proj_point1;
            pivot1 = proj_point2;
          }
        }        
      }
      
      /* Calculating blending line 1 equation */
      
      {
        double line1_q = pivot1.y() -
          ( blend_lines_m * pivot1.x() );
          
        TeCoord2D aux_point( ( pivot1.x() + 10 ), 
          ( blend_lines_m * ( pivot1.x() + 10 ) ) + line1_q );
          
        line1_a = pivot1.y() - aux_point.y();
        line1_b = aux_point.x() - pivot1.x();
        line1_c = ( pivot1.x() * aux_point.y() ) - 
          ( aux_point.x() * pivot1.y() );
      }
  
      /* Calculating blending line 2 equation */
      
      {
        double line2_q = pivot2.y() -
          ( blend_lines_m * pivot2.x() );
          
        TeCoord2D aux_point( ( pivot2.x() + 10 ), 
          ( blend_lines_m * ( pivot2.x() + 10 ) ) + line2_q );
          
        line2_a = pivot2.y() - aux_point.y();
        line2_b = aux_point.x() - pivot2.x();
        line2_c = ( pivot2.x() * aux_point.y() ) - 
          ( aux_point.x() * pivot2.y() );
      }        
    }
  } else {
    /* This is the case where the polygons intersection gives us more than 
       two points */
       
    /* line passing throught intersection polygon centroid coeficients -
       this line is parallel with the line passing throught centroids */
    
    const double ipc_line_m = ( centroid2.y() - centroid1.y() ) /
      ( centroid2.x() - centroid1.x() );
    const double ipc_line_q = inter_pol_centroid.y() - 
      ( ipc_line_m * inter_pol_centroid.x() );
      
    /* perpendicular line angular coeficient */
    
    const double perp_line_m = ( -1.0 ) / ipc_line_m;
    
    /* Projecting each intersection polygon point over the parallel line */
    
    std::vector< TeCoord2D > proj_points;
    
    for( unsigned int lr_index = 0 ; lr_index < inter_pol.size() ; 
      ++lr_index  ) {
      for( unsigned int coord_index = 0 ; 
        coord_index < inter_pol[ lr_index ].size() ; ++coord_index  ) {
        
        /* perpendicular line linear coef passing throught the current point */
      
        double perp_line_q = inter_pol[ lr_index ][ coord_index ].y() -
          ( perp_line_m * inter_pol[ lr_index ][ coord_index ].x() );
          
        /* Calculating the lines intersection */
        
        TeCoord2D int_point;
        int_point.x( ( perp_line_q - ipc_line_q ) / 
          ( ipc_line_m - perp_line_m  ) );
        int_point.y( ( ipc_line_m * int_point.x() ) + ipc_line_q );
        
        proj_points.push_back( int_point );
      }
    }
    
    TEAGN_TRUE_OR_THROW( ( proj_points.size() > 1 ), 
      "Invalid projected points" );
      
    /* Finding the two pivots for the two blending lines */
    
    TeCoord2D pivot1; /* related to polygon 1 */
    TeCoord2D pivot2; /* related to polygon 2 */
    {
      /* Taking the two points  with the higher distance one to another */
      
      TeCoord2D proj_point1 = proj_points[ 0 ];
      TeCoord2D proj_point2 = proj_points[ 1 ];
      double higher_dist = TeDistance( proj_point1, proj_point2 );
      
      for( unsigned int proj_points_index = 2 ; 
        proj_points_index < proj_points.size() ; ++proj_points_index ) {
        
        double dist1 = TeDistance( proj_points[ proj_points_index ], 
          proj_point1 );
        double dist2 = TeDistance( proj_points[ proj_points_index ], 
          proj_point2 );
          
        if( dist1 > dist2 ) {
          if( dist1 > higher_dist ) {
            proj_point2 = proj_points[ proj_points_index ];
            higher_dist = dist1;
          }
        } else {
          if( dist2 > higher_dist ) {
            proj_point1 = proj_points[ proj_points_index ];
            higher_dist = TeDistance( proj_points[ proj_points_index ], 
              proj_point2 );
          }
        }
      }
      
      /* generating the equation for the perpendicular line passing 
         throught the intersection polygon centroid */
         
      double ip_cent_perp_line_a = 0;
      double ip_cent_perp_line_b = 0;
      double ip_cent_perp_line_c = 0;
      {
        double ip_centroid_perp_line_q = inter_pol_centroid.y() -
        ( perp_line_m * inter_pol_centroid.x() ); 
        
        TeCoord2D aux_point( ( inter_pol_centroid.x() + 10 ), 
          ( perp_line_m * ( inter_pol_centroid.x() + 10 ) ) + 
          ip_centroid_perp_line_q );
          
        ip_cent_perp_line_a = inter_pol_centroid.y() - aux_point.y();
        ip_cent_perp_line_b = aux_point.x() - inter_pol_centroid.x();
        ip_cent_perp_line_c = ( inter_pol_centroid.x() * aux_point.y() ) - 
        ( aux_point.x() * inter_pol_centroid.y() );          
      }
        
      /* Association of the projected points with the correct polygons */     
      
      {
        double rel_pos_proj_point1 = ( ip_cent_perp_line_a * proj_point1.x() ) 
          + ( ip_cent_perp_line_b * proj_point1.y() ) + ip_cent_perp_line_c;
        double rel_pos_centroid1 = ( ip_cent_perp_line_a * centroid1.x() ) 
          + ( ip_cent_perp_line_b * centroid1.y() ) + ip_cent_perp_line_c;
         
        if( ( rel_pos_proj_point1 > 0 ) && ( rel_pos_centroid1 > 0 ) ) {
          pivot1 = proj_point1;
          pivot2 = proj_point2;        
        } else {
          pivot2 = proj_point1;
          pivot1 = proj_point2;
        }
      }
    }
    
    /* Calculating blending line 1 equation */
    
    {
      double line1_q = pivot1.y() -
        ( perp_line_m * pivot1.x() );
        
      TeCoord2D aux_point( ( pivot1.x() + 10 ), 
        ( perp_line_m * ( pivot1.x() + 10 ) ) + line1_q );
        
      line1_a = pivot1.y() - aux_point.y();
      line1_b = aux_point.x() - pivot1.x();
      line1_c = ( pivot1.x() * aux_point.y() ) - 
        ( aux_point.x() * pivot1.y() );
    }

    /* Calculating blending line 2 equation */
    
    {
      double line2_q = pivot2.y() -
        ( perp_line_m * pivot2.x() );
        
      TeCoord2D aux_point( ( pivot2.x() + 10 ), 
        ( perp_line_m * ( pivot2.x() + 10 ) ) + line2_q );
        
      line2_a = pivot2.y() - aux_point.y();
      line2_b = aux_point.x() - pivot2.x();
      line2_c = ( pivot2.x() * aux_point.y() ) - 
        ( aux_point.x() * pivot2.y() );
    }
  }  
}


void TePDIEuclidianBlendStrategy::drawLine( 
  const TePDITypes::TePDIRasterPtrType& raster, unsigned int channel,
  double value, double line_a, double line_b, double line_c )
{
  double line_x = 0;
  double line_y = 0;
  
  if( line_a == 0 ) {
    /* horizontal parallels */
    
    line_y = ( -1.0 * line_c ) / line_b;
  
    for( line_x = 0 ; line_x < raster->params().ncols_ ; ++line_x ) {
      raster->setElement( (int)line_x, (int)line_y, value, channel );
    }             
  } else if( line_b == 0 ) {
    /* vertical parallels */
    
    line_x = ( -1.0 * line_c ) / line_a;
  
    for( line_y = 0 ; line_y < raster->params().nlines_ ; ++line_y ) {
      
      raster->setElement( (int)line_x, (int)line_y, value, channel );
    }            
  } else {
    for( line_x = 0 ; line_x < raster->params().ncols_ ; ++line_x ) {

      line_y = ( ( -1.0 * line_c ) + ( -1.0 * line_a * line_x ) ) / line_b;
      
      raster->setElement( (int)line_x, (int)line_y, value, channel );
    }           
  }
}
