/* SimpleAction.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl.actions;

import java.awt.event.ActionEvent;
import java.util.ResourceBundle;
import javax.swing.AbstractAction;
import javax.swing.ButtonModel;
import javax.swing.Icon;

/**
 * Abstract action (controller) based on an internationalized description.
 * As with {@link AbstractAction} you need only  define the
 * <code>actionPerformed</code> method when you extend this class.<p>
 * Also defines several new action properties which allow a more
 * detailed configuration
 * of buttons and menu items created from this action to be configured.
 *
 * @see Description
 * @see ToolBarButton
 * @see ToolBarToggleButton
 * @see MenuButton
 * @see MenuRadioButton
 * @see MenuCheckBoxButton
 */
public abstract class SimpleAction extends AbstractAction {

    /**
     * The action key used for storing a medium size <code>Icon</code>
     * for use with tool bar buttons. Icons registered
     * with this key will be used for tool bar buttons
     * of type {@link ToolBarToggleButton} or {@link ToolBarButton}, but not
     * for standard Swing buttons.
     */
    public static final String MEDIUM_ICON = "MediumIcon";
    /**
     * The action key used for storing a medium size 'disabled'
     * <code>Icon</code> for use with tool bar buttons. Icons registered
     * with this key will be used for tool bar buttons
     * of type {@link ToolBarToggleButton} or {@link ToolBarButton}, but not
     * for standard Swing buttons.
     */
    public static final String MEDIUM_DISABLED_ICON = "MediumDisabledIcon";
    /**
     * The action key used for storing a small 'disabled'
     * <code>Icon</code> for use in menus.<p>
     * <b>Note:</b> Swing menu items do not honour this key, use
     * one of the {@code Menu...Button} classes instead.
     * @since 1.1
     */
    public static final String SMALL_DISABLED_ICON = "SmallDisabledIcon";
    /**
     * The action key used for storing a shared toggle button model. This allows
     * selection state to be synchronised between different toggle buttons and 
     * radio or checkbox menu items.<p>
     * <b>Note:</b> Swing buttons do not honour this key, use
     * one of the {@code ToolBar...Button} or {@code Menu...Button} classes instead.
     * @since 1.1
     * @see javax.swing.JToggleButton.ToggleButtonModel
     */
    public static final String TOGGLE_BUTTON_MODEL = "ToggleButtonModel";

    /**
     * Construct an action with an internationalized description
     * string and given icons.
     *
     * @param bundle Resource bundle which contains the description for
     * this action.
     * @param key Key used for the description of this action in the bundle.
     * @param icon Small icon to be used for menu and standard button decoration (or null)
     * @param mediumIcon Medium size icon to be used for tool bar buttons (or null)
     * @param mediumDisabledIcon Medium size icon to be used for a disabled tool bar buttons (or null)
     */
    protected SimpleAction(ResourceBundle bundle, String key, Icon icon, Icon mediumIcon, Icon mediumDisabledIcon) {
        this(bundle, key, icon);
        if (mediumIcon != null) {
            putValue(MEDIUM_ICON, mediumIcon);
        }
        if (mediumDisabledIcon != null) {
            putValue(MEDIUM_DISABLED_ICON, mediumDisabledIcon);
        }
    }

    /**
     * Construct an action with an internationalized description
     * string and given icon.
     *
     * @param bundle Resource bundle which contains the description for
     * this action.
     * @param key Key used for the description of this action in the bundle.
     * @param icon Small icon to be used for menu and standard button decoration (or null).
     */
    protected SimpleAction(ResourceBundle bundle, String key, Icon icon) {
        super(null, icon);
        new Description(bundle.getString(key)).init(this);
    }
    
    /**
     * Make a simple action which is not really different from an {@code AbstractAction}.<p>
     * Clients should consider using {@code AbstractAction} directly or else
     * use one of the other constructors of this class.
     */
    protected SimpleAction () {
        // primarily intended for use with UndoAction and RedoAction
    }

    // avoids buggy PMD warning
    public abstract void actionPerformed(ActionEvent e);

    /**
     * Set the medium size icon for this action. This icon will be used
     * for all tool bar buttons
     * of type {@link ToolBarToggleButton} or {@link ToolBarButton} which are
     * created with this action.
     * @since 1.1
     */
    public void setMediumIcon(Icon mediumIcon) {
        putValue(MEDIUM_ICON, mediumIcon);
    }

    /**
     * Set the medium size 'disabled' icon for this action. This icon will be used
     * for all tool bar buttons
     * of type {@link ToolBarToggleButton} or {@link ToolBarButton} which are
     * created with this action.
     * @since 1.1
     */
    public void setMediumDisabledIcon(Icon mediumDisabledIcon) {
        putValue(MEDIUM_DISABLED_ICON, mediumDisabledIcon);
    }

    /**
     * Set the small 'disabled' icon for this action. This icon will be used
     * for all menu items
     * of type {@code Menu...Button} which are
     * created with this action.
     * @since 1.1
     */
    public void setSmallDisabledIcon(Icon smallDisabledIcon) {
        putValue(SMALL_DISABLED_ICON, smallDisabledIcon);
    }

    /**
     * Set the shared button model for toggle buttons created with this
     * action.<p>
     * <b>Note:</b> Swing buttons do not honour this property, use
     * one of the {@code ToolBar...Button} or {@code Menu...Button} classes instead.
     * @since 1.1
     * @see javax.swing.JToggleButton.ToggleButtonModel
     */
    public void setSharedToggleButtonModel(ButtonModel buttonModel) {
        putValue(TOGGLE_BUTTON_MODEL, buttonModel);
    }
}
