/*
 * libSpiff - XSPF playlist handling library
 *
 * Copyright (C) 2007, Sebastian Pipping / Xiph.Org Foundation
 * All rights reserved.
 *
 * Redistribution  and use in source and binary forms, with or without
 * modification,  are permitted provided that the following conditions
 * are met:
 *
 *     * Redistributions   of  source  code  must  retain  the   above
 *       copyright  notice, this list of conditions and the  following
 *       disclaimer.
 *
 *     * Redistributions  in  binary  form must  reproduce  the  above
 *       copyright  notice, this list of conditions and the  following
 *       disclaimer   in  the  documentation  and/or  other  materials
 *       provided with the distribution.
 *
 *     * Neither  the name of the Xiph.Org Foundation nor the names of
 *       its  contributors may be used to endorse or promote  products
 *       derived  from  this software without specific  prior  written
 *       permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS  IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT  NOT
 * LIMITED  TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS
 * FOR  A  PARTICULAR  PURPOSE ARE DISCLAIMED. IN NO EVENT  SHALL  THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL,    SPECIAL,   EXEMPLARY,   OR   CONSEQUENTIAL   DAMAGES
 * (INCLUDING,  BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES;  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT  LIABILITY,  OR  TORT (INCLUDING  NEGLIGENCE  OR  OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Sebastian Pipping, sping@xiph.org
 */

/**
 * @file SpiffDataWriter.cpp
 * Implementation of SpiffDataWriter.
 */

#include <spiff/SpiffData.h>
#include <spiff/SpiffDataWriter.h>
#include <spiff/SpiffXmlFormatter.h>
#include <spiff/SpiffExtension.h>
#include <spiff/SpiffExtensionWriter.h>
using namespace std;

namespace Spiff {



/// @cond DOXYGEN_NON_API

/**
 * D object for SpiffDataWriter.
 */
class SpiffDataWriterPrivate {

	friend class SpiffDataWriter;

	SpiffData * data; ///< Data object to write

	/**
	 * Creates a new D object.
	 *
	 * @param data	Data object to write
	 */
	SpiffDataWriterPrivate(SpiffData * data) : data(data) {

	}

	/**
	 * Destroys this D object.
	 */
	~SpiffDataWriterPrivate() {

	}

};

/// @endcond



SpiffDataWriter::SpiffDataWriter(SpiffData * data)
		: d(new SpiffDataWriterPrivate(data)),
		output(NULL) {

}



SpiffDataWriter::SpiffDataWriter(const SpiffDataWriter & source)
		: d(new SpiffDataWriterPrivate(*(source.d))),
		output(source.output) {

}



SpiffDataWriter & SpiffDataWriter::operator=(const SpiffDataWriter & source) {
	if (this != &source) {
		*(this->d) = *(source.d);
		this->output = source.output;
	}
	return *this;
}



SpiffDataWriter::~SpiffDataWriter() {
	delete this->d;
}



void SpiffDataWriter::setData(SpiffData * data) {
	this->d->data = data;
}





void SpiffDataWriter::writePrimitive(const XML_Char * name, const XML_Char * body) {
	const XML_Char * atts[1] = {NULL};
	this->output->writeHomeStart(name, atts);
	this->output->writeBody(body);
	this->output->writeHomeEnd(name);
}



void SpiffDataWriter::writePrimitive(const XML_Char * name, int body) {
	const XML_Char * atts[1] = {NULL};
	this->output->writeHomeStart(name, atts);
	this->output->writeBody(body);
	this->output->writeHomeEnd(name);
}



void SpiffDataWriter::writeImage() {
	const XML_Char * const image = this->d->data->getImage();
	if (image != NULL) {
		writePrimitive(_PT("image"), image);
	}
}



void SpiffDataWriter::writeInfo() {
	const XML_Char * const info = this->d->data->getInfo();
	if (info != NULL) {
		writePrimitive(_PT("info"), info);
	}
}



void SpiffDataWriter::writeAnnotation() {
	const XML_Char * const annotation = this->d->data->getAnnotation();
	if (annotation != NULL) {
		writePrimitive(_PT("annotation"), annotation);
	}
}



void SpiffDataWriter::writeCreator() {
	const XML_Char * const creator = this->d->data->getCreator();
	if (creator != NULL) {
		writePrimitive(_PT("creator"), creator);
	}
}



void SpiffDataWriter::writeTitle() {
	const XML_Char * const title = this->d->data->getTitle();
	if (title != NULL) {
		writePrimitive(_PT("title"), title);
	}
}



void SpiffDataWriter::writeLinks() {
	int index = 0;
	const pair<const XML_Char *, const XML_Char *> * entry;
	for (;;) {
		entry = this->d->data->getLink(index++);
		if (entry == NULL) {
			return;
		}

		const XML_Char * atts[3] = {_PT("rel"), entry->first, NULL};
		this->output->writeHomeStart(_PT("link"), atts);
		this->output->writeBody(entry->second);
		this->output->writeHomeEnd(_PT("link"));

		delete entry; // since the pair was created for us
	}
}



void SpiffDataWriter::writeMetas() {
	int index = 0;
	const pair<const XML_Char *, const XML_Char *> * entry;
	for (;;) {
		entry = this->d->data->getMeta(index++);
		if (entry == NULL) {
			return;
		}

		const XML_Char * atts[3] = {_PT("rel"), entry->first, NULL};
		this->output->writeHomeStart(_PT("meta"), atts);
		this->output->writeBody(entry->second);
		this->output->writeHomeEnd(_PT("meta"));

		delete entry; // since the pair was created for us
	}
}



void SpiffDataWriter::writeExtensions() {
	int index = 0;
	const SpiffExtension * entry;
	for (;;) {
		entry = this->d->data->getExtension(index++);
		if (entry == NULL) {
			return;
		}

		SpiffExtensionWriter * const writer
				= entry->newWriter(this->output);
		if (writer != NULL) {
			writer->write();
			delete writer;
		}
	}
}



}
