/*
*   pam_abl - a PAM module and program for automatic blacklisting of hosts and users
*
*   Copyright (C) 2005 Andy Armstrong andy@hexten.net
*   Copyright (C) 2009 Chris Tasma pam-abl@deksai.com
*
*   This program is free software: you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation, either version 2 of the License, or
*   (at your option) any later version.
*
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "pam_abl.h"


static int record_host(const abl_args *args, abl_info *info, time_t tm) {
    if (NULL != args->host_db) {
        const void *rhost;
        int err;

        if (err = pam_get_item(args->pamh, PAM_RHOST, &rhost), PAM_SUCCESS != err) {
            log_pam_error(args, err, "getting PAM_RHOST");
            return err;
        }
        if (NULL != rhost) {
            info->subject = HOST;
            info->host = rhost;
            return record(args, info, tm, args->host_purge);
        } else {
            log_debug(args, "PAM_RHOST is NULL");
            return 0;
        }
    } else {
        return 0;
    }
}

static int record_user(const abl_args *args, abl_info *info, time_t tm) {
    const void *user;
    int err;
    if (err = pam_get_item(args->pamh, PAM_USER, &user), PAM_SUCCESS != err) {
        log_pam_error(args, err, "getting PAM_USER");
        return err;
    }
    if (NULL != user) {
        info->subject = USER;
        info->user = user;
        return record(args, info, tm, args->user_purge);
    } else {
        log_debug(args, "PAM_USER is NULL");
        return 0;
    }
    return 1;
}

static int record_attempt(const abl_context *context) {
    int err = 0;
    time_t tm = time(NULL);

    log_debug(context->args, "Recording failed attempt");

    if (err = record_host(context->args, context->info, tm), 0 != err) {
        return err;
    }

    if (err = record_user(context->args, context->info, tm), 0 != err) {
        return err;
    }
    return err;
}

static int check_attempt(const abl_args *args, abl_info *info) {
    int err;
    time_t tm = time(NULL);
    if (info->user != NULL && info->service != NULL && info->host != NULL) {
        info->subject = HOST;
        err = check_host(args, info, tm);
        if (err) log_warning(args,"Failed to check host.");
        if (info->state == BLOCKED ) {
            /*No point in continuing since they've failed.*/
            return 0;
        }

        info->subject = USER;
        err = check_user(args, info, tm);
        if (err) log_warning(args,"Failed to check user.");
        
    }
    return 0;
}


static void cleanup(pam_handle_t *pamh, void *data, int err) {
    DB* db;
    if (NULL != data) {
        abl_context *context = data;
        log_debug(context->args, "In cleanup, err is %08x", err); 

        if (err && (err & PAM_DATA_REPLACE) == 0) {
            log_debug(context->args, "record returned %d",record_attempt(context));
        }
        config_free(context->args);
        free(context->args);

        db = context->info->utdb;
        db->close(db,0);
        db = context->info->usdb;
        db->close(db,0);
        db = context->info->htdb;
        db->close(db,0);
        db = context->info->hsdb;
        db->close(db,0);
        free(context->info);
        free(context);
    }
}
/* Authentication management functions */

PAM_EXTERN int pam_sm_authenticate(pam_handle_t *pamh, int flags, int argc, const char **argv) {
    abl_args *args;
    abl_info *info;
    abl_context *context;
    DB *utdb;
    DB *usdb;
    DB *htdb;
    DB *hsdb;
    int err = PAM_SUCCESS;

    /*log_debug(args, "pam_sm_authenticate(), flags=%08x", flags);*/


    if (args = malloc(sizeof(abl_args)), NULL == args) {
        return PAM_BUF_ERR;
    }
    if (info = malloc(sizeof(abl_info)), NULL == info) {
        return PAM_BUF_ERR;
    }
    if (context = malloc(sizeof(abl_context)), NULL == context) {
        return PAM_BUF_ERR;
    }

    memset(info,0,sizeof(abl_info));
    memset(context,0,sizeof(abl_context));


    if (err = config_parse_args(pamh, argc, argv, args), PAM_SUCCESS == err) {

        /* We now keep the database open from the beginning to avoid the cost
         * of opening them repeatedly. */

        err = dbopen(args, args->user_db, DB_TIME, &utdb);
        if (err) goto psa_fail;
        info->utdb = utdb;
        err = dbopen(args, args->user_db, DB_STATE, &usdb);
        if (err) goto psa_fail;
        info->usdb = usdb;
        err = dbopen(args, args->host_db, DB_TIME, &htdb);
        if (err) goto psa_fail;
        info->htdb = htdb;
        err = dbopen(args, args->host_db, DB_STATE, &hsdb);
        if (err) goto psa_fail;
        info->hsdb = hsdb;

        context->args = args;
        context->info = info;
        
        if (err = pam_set_data(pamh, DATA_NAME, context, cleanup), PAM_SUCCESS != err) {
            log_pam_error(args,err,"setting PAM data");
            goto psa_fail;
        }
        if (err = pam_get_item(args->pamh, PAM_USER, (const void **) &info->user), PAM_SUCCESS != err) {
            log_pam_error(args, err, "getting PAM_USER");
            goto psa_fail;
        }
        if (err = pam_get_item(args->pamh, PAM_SERVICE, (const void **) &info->service), PAM_SUCCESS != err) {
            log_pam_error(args, err, "getting PAM_SERVICE");
            goto psa_fail;
        }
        if (err = pam_get_item(args->pamh, PAM_RHOST, (const void **) &info->host), PAM_SUCCESS != err) {
            log_pam_error(args, err, "getting PAM_RHOST");
            goto psa_fail;
        }

        check_attempt(args, info);
        if (info->state == BLOCKED) {
            log_info("Blocking access from %s to service %s, user %s", info->host, info->service, info->user);
            return PAM_AUTH_ERR;
        }
        else {
            return PAM_SUCCESS;
        }
    } 

psa_fail:
    config_free(args);
    utdb->close(utdb,0);
    usdb->close(usdb,0);
    htdb->close(htdb,0);
    hsdb->close(hsdb,0);
    free(args);
    free(info);
    free(context);
    return err;
}

PAM_EXTERN int pam_sm_setcred(pam_handle_t *pamh, int flags, int argc, const char **argv) {
    return pam_set_data(pamh, DATA_NAME, NULL, cleanup);
}

/* Init structure for static modules */
#ifdef PAM_STATIC
struct pam_module _pam_abl_modstruct = {
    MODULE_NAME,
    pam_sm_authenticate,
    pam_sm_setcred,
    NULL,
    NULL,
    NULL,
    NULL
};
#endif

