/*******************************************************************************************************************************************
 ccomponent.h
 
 CClass
 |--CSerialized
    |--CMetaModule
       |--CObject
   	  |--CComponent

 gtkol application components hierarchy definition. CComponent is the spine of gtkol. The graphical specific calls are not performed on
 this level, only the application hierarchy representation is coded here, so many of the defined functions should be overwritten to handle
 gtk gui requests.
 When performing in threaded applications, the CComponent class automatically detects the glib initialization if any and locks the spine
 on instances deletion and instanciation only if so. Due to performance reasons, there is no lock on the gtk/gdk part nor while requesting
 the spine's hierarchy. 
*******************************************************************************************************************************************/

#ifndef __CCOMPONENT_H__
#define __CCOMPONENT_H__

#include "cobject.h"

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

// check thread implementation availability
#ifndef G_THREADS_ENABLED
#error "glib does not support threads, please check it first."
#endif

// what's defined here
class CComponent;

// we'll use the libgenerics namespace services
using namespace NServices;

//-----------------------------------------------------------------------------------------------------------------------------------------
// special types declaration
//-----------------------------------------------------------------------------------------------------------------------------------------
typedef TBuffer <CComponent 	*> CComponents;		// gtkol components list
typedef TNode	<CComponent  	 > CComponentNode;	// gtkol component hierarchical node, internally used
typedef TBuffer <CComponentNode *> CComponentNodes;	// gtkol component hierarchical nodes list, internally used

//------------------------------------------------------------------------------------------------------------------------------------------
// component xml serialization constants :
//------------------------------------------------------------------------------------------------------------------------------------------
// <ccomponent [owner-id="UInt32"]>
//  [ <children>
//       ...
//    </children> ]
// </ccomponent>
//------------------------------------------------------------------------------------------------------------------------------------------
static CString XML_COMPONENT_ELEMENT		("ccomponent");
static CString XML_COMPONENT_CHILDREN_ELEMENT	("children");
static CString XML_COMPONENT_ATTR_OWNER_ID	("owner-id");

//-----------------------------------------------------------------------------------------------------------------------------------------
// CComponent class
//-----------------------------------------------------------------------------------------------------------------------------------------
class CComponent : public CObject
{
	// instanciation section
	public :

		CComponent			(CComponent *inOwner, const CObjectListener *inListener=NULL);
		virtual ~CComponent		() =0;

	// component identification section
	public :

		// component's id, component of specified id
		UInt32				GetId			() const;
		static CComponent *		GetComponent		(const UInt32 inId);

	// components list and descendant hierarchy access section
	public :

		// application components list of potential requested and rejected type specifications (i.e. the desired components 
		// must be derived from "inRequested" type and must not be derived from "inRejected" type)
		static CComponents 		GetComponents		(const CMetaClass *inRequested=NULL, 
									 const CMetaClass *inRejected =NULL);

		// direct children of potential requested and rejected type specifications (i.e. the desired components must be
		// derived from "inRequested" type and must not be derived from "inRejected" type)
		CComponents 			GetChildren		(const CMetaClass *inRequested=NULL, 
									 const CMetaClass *inRejected =NULL) const;

		// all subcomponents of potential requested and rejected type specifications (i.e. the desired components must be
		// derived from "inRequested" type and must not be derived from "inRejected" type), this instance included, prefixed order
		CComponents			GetSubComponents	(const CMetaClass *inRequested=NULL, 
									 const CMetaClass *inRejected =NULL) const;

		// all components of this instance hierarchy level of potential requested and rejected type specifications (i.e. the 
		// desired components of the same hierarchy level sould be derived from "inRequested" type and must not be derived from
		// "inRejected" type), this instance included
		CComponents			GetSiblings		(const CMetaClass *inRequested=NULL, 
									 const CMetaClass *inRejected =NULL) const;

		// check if the specified component instance is a subcomponent of this one
		Bool				HasSubComponent		(const CComponent *inComponent) const;

		// check if the specified component instance is a sibling of this one
		Bool				HasSibling		(const CComponent *inComponent) const;

		// in owner direct child component index; index incrementation only on specified "inRequested" type if any, -1 if not found
		static SInt16			GetInOwnerIndex		(const CComponent *inOwner, const CComponent *inChild,
									 const CMetaClass *inRequested=NULL, 
									 const CMetaClass *inRejected =NULL);

		// in owner direct or indirect child component index; index incrementation only on specified "inRequested" type if any, -1
		// if not found, 0 indexed
		static SInt16			GetInOwnerOccurence	(const CComponent *inOwner, const CComponent *inChild,
									 const CMetaClass *inRequested=NULL, 
									 const CMetaClass *inRejected =NULL);

	// owner section
	public :

		// check if the specified component could be assigned as an owner of this instance
		virtual Bool			CheckSetOwner		(const CComponent *inCandidate) const;

		// owner's affectation should be overwritten by the derived definitions in order to handle specific graphics modifications;
		// the inIndex parameter specifies the child index the owner component should insert to, -1 to append at the end of the
		// current handled children list; overloaded definitions may introduce some specific rules
		virtual Bool			SetOwner		(CComponent *inOwner, const SInt16 inIndex=-1);

		// indirect search of requested type's owner or direct owner access if no special type requested
		CComponent *			GetOwner		(const CMetaClass *inRequested=NULL) const;

		// component expected owner types and expected direct children types, defaults to __metaclasses(CComponent) and should be
		// overwritten as needed to reflect the hierachical types rules (CComponent::CheckSetOwner and the derived notions of d&d 
		// are partly based on the results of both of these functions)
		virtual CMetaClasses		OwnerMustBe		() const;
		virtual CMetaClasses		ChildMustBe		() const;

	// serialization redefiniton
	public :

		// gtkol component xml serialization, manages the whole gtkol application's hierarchy serialization
		virtual void			Serialize		(CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE;
	
		// gtkol does not support archive serialization, this function is defined only to generate an exception if called
		virtual void			Serialize		(CArchive &ioArchive) THROWABLE;

	// glib thread support layer
	public :

		// gtkol is thread aware when compiled on the glib's thread support, the following functions are automatically internaly
		// used to lock the gtkol spine in multiple threaded environments while instanciating or deleting instances, so no need to 
		// explicitly call them unless you specifically need it; the gtk/gdk lock is not handled by gtkol and should be explicitly 
		// called while performing gui requests
		static void			Lock			();
		static Bool			TryLock			();
		static void			Unlock			();
		static void			LockFull		(const UInt32 inDepth);
		static UInt32			UnlockFull		();

	// private attributes
	private :

		// static application hierarchy
		static CComponentNode		m_AbsoluteNode;

		// component associated node
		CComponentNode *		m_ComponentNode;

		// component's id, automatic incrementation
		const  UInt32			m_Id;
		static UInt32			m_IdInc;
		static TBuffer <UInt32>		m_IdLost;

		// mutex handling
		static GStaticRecMutex		m_GStaticRecMutex;

		// particular friend for component absolute node value affectation (the gtkol application's instance is the absolute one)
		friend class			CApplication;

		// CComponent metaclass association
		SECTION_GENERIC_METACLASS;
};

// metaclass and class tag declarations
DECLARE_GENERIC_METACLASS ('cpnt', CComponent, CObject);

#endif
