/**
 * @file libgalago-gtk/galago-gdk-pixbuf.c GdkPixbuf utility functions
 *
 * @Copyright (C) 2004-2006 Christian Hammond.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago-gtk/galago-gdk-pixbuf.h>

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "prefix.h"

static char *
get_pixmap_path(const char *name)
{
	char *path;

#ifdef DATADIR
	path = g_build_filename(DATADIR, "pixmaps", "galago", "gtk", name, NULL);

	if (g_file_test(path, G_FILE_TEST_EXISTS))
		return path;

	g_free(path);
#endif /* DATADIR */

	path = g_build_filename(PIXMAPDIR, name, NULL);

	if (g_file_test(path, G_FILE_TEST_EXISTS))
		return path;

	g_free(path);

	return NULL;
}

static GdkPixbuf *
_galago_gdk_pixbuf_scale(GdkPixbuf *pixbuf, int width, int height)
{
	GdkPixbuf *scale;

	if (width == -1 && height == -1)
		return pixbuf;

	if (width == -1)
	{
		double ratio = (double)gdk_pixbuf_get_height(pixbuf) / (double)height;
		width = gdk_pixbuf_get_width(pixbuf) / ratio;
	}
	else if (height == -1)
	{
		double ratio = (double)gdk_pixbuf_get_width(pixbuf) / (double)width;
		height = gdk_pixbuf_get_height(pixbuf) / ratio;
	}

	scale = gdk_pixbuf_scale_simple(pixbuf, width, height,
									GDK_INTERP_BILINEAR);

	g_object_unref(pixbuf);

	return scale;
}

GdkPixbuf *
galago_gdk_pixbuf_new_generic(int width, int height)
{
	char *path;
	GdkPixbuf *pixbuf;

	if ((path = get_pixmap_path("generic-presence.png")) == NULL)
		return NULL;

	if (width == -1 || height == -1)
	{
		pixbuf = _galago_gdk_pixbuf_scale(gdk_pixbuf_new_from_file(path, NULL),
										  width, height);
	}
	else
		pixbuf = gdk_pixbuf_new_from_file_at_size(path, width, height, NULL);

	g_free(path);

	return pixbuf;
}

static GdkPixbuf *
_galago_gdk_pixbuf_new_from_image_data(unsigned char *data, size_t len,
									   int width, int height)
{
	GdkPixbufLoader *loader;
	GdkPixbuf *pixbuf;

	loader = gdk_pixbuf_loader_new();
	gdk_pixbuf_loader_write(loader, data, len, NULL);

	pixbuf = gdk_pixbuf_loader_get_pixbuf(loader);

	if (pixbuf != NULL)
		g_object_ref(G_OBJECT(pixbuf));

	gdk_pixbuf_loader_close(loader, NULL);
	g_object_unref(G_OBJECT(loader));

	return _galago_gdk_pixbuf_scale(pixbuf, width, height);
}

GdkPixbuf *
galago_gdk_pixbuf_new_from_image(const GalagoImage *image, int width,
								 int height)
{
	unsigned char *data;
	size_t len;

	g_return_val_if_fail(image != NULL, NULL);
	g_return_val_if_fail(GALAGO_IS_IMAGE(image), NULL);

	galago_image_get_data(GALAGO_IMAGE(image), &data, &len);

	return _galago_gdk_pixbuf_new_from_image_data(data, len, width, height);
}

GdkPixbuf *
galago_gdk_pixbuf_new_from_image_with_size(const GalagoImage *image,
										   GtkIconSize size)
{
	int width, height;

	g_return_val_if_fail(image != NULL, NULL);
	g_return_val_if_fail(GALAGO_IS_IMAGE(image), NULL);
	g_return_val_if_fail(size  != GTK_ICON_SIZE_INVALID, NULL);

	if (!gtk_icon_size_lookup(size, &width, &height))
	{
		width  = -1;
		height = -1;
	}

	return galago_gdk_pixbuf_new_from_image(image, width, height);
}

GdkPixbuf *
galago_gdk_pixbuf_new_from_service(const GalagoService *service,
								   int width, int height)
{
	char *filename, *path;
	GdkPixbuf *pixbuf;

	g_return_val_if_fail(service != NULL, NULL);

	filename = g_strdup_printf("%s.png", galago_service_get_id(service));
	path = get_pixmap_path(filename);
	g_free(filename);

	if (path == NULL)
		return NULL;

	if (width == -1 || height == -1)
	{
		pixbuf = gdk_pixbuf_new_from_file(path, NULL);

		if (width != -1 || height != -1)
		{
			GdkPixbuf *scale;

			if (width == -1)
				width = gdk_pixbuf_get_width(pixbuf);

			if (height == -1)
				height = gdk_pixbuf_get_height(pixbuf);

			scale = gdk_pixbuf_scale_simple(pixbuf, width, height,
											GDK_INTERP_BILINEAR);

			g_object_unref(pixbuf);

			pixbuf = scale;
		}
	}
	else
		pixbuf = gdk_pixbuf_new_from_file_at_size(path, width, height, NULL);

	g_free(path);

	return pixbuf;
}

GdkPixbuf *
galago_gdk_pixbuf_new_from_service_with_size(const GalagoService *service,
											 GtkIconSize size)
{
	int width, height;

	g_return_val_if_fail(service != NULL,                  NULL);
	g_return_val_if_fail(size    != GTK_ICON_SIZE_INVALID, NULL);

	if (!gtk_icon_size_lookup(size, &width, &height))
	{
		width  = -1;
		height = -1;
	}

	return galago_gdk_pixbuf_new_from_service(service, width, height);
}

static GdkPixbuf *
_new_pixbuf_from_account(const GalagoAccount *account,
						 int width, int height,
						 GalagoGtkIconPrecedence precedence)
{
	GdkPixbuf *pixbuf;
	GalagoService *service;
	GalagoPresence *presence;

	presence = galago_account_get_presence(account, TRUE);

	if (presence == NULL)
	{
		service = galago_account_get_service(account);

		pixbuf = galago_gdk_pixbuf_new_from_service(service, width, height);

		if (pixbuf != NULL && !galago_account_is_connected(account))
			gdk_pixbuf_saturate_and_pixelate(pixbuf, pixbuf, 0.0, FALSE);
	}
	else
	{
		pixbuf = galago_gdk_pixbuf_new_from_presence(presence, width, height,
													 precedence);
	}

	return pixbuf;
}

GdkPixbuf *
galago_gdk_pixbuf_new_from_account(const GalagoAccount *account,
								   int width, int height)
{
	g_return_val_if_fail(account != NULL, NULL);

	return _new_pixbuf_from_account(account, width, height,
									GALAGO_GTK_ICON_PROTOCOL);
}

GdkPixbuf *
galago_gdk_pixbuf_new_from_account_with_size(const GalagoAccount *account,
											 GtkIconSize size)
{
	int width, height;

	g_return_val_if_fail(account != NULL,                  NULL);
	g_return_val_if_fail(size    != GTK_ICON_SIZE_INVALID, NULL);

	if (!gtk_icon_size_lookup(size, &width, &height))
	{
		width  = -1;
		height = -1;
	}

	return galago_gdk_pixbuf_new_from_account(account, width, height);
}

GdkPixbuf *
galago_gdk_pixbuf_new_from_presence(const GalagoPresence *presence,
									int width, int height,
									GalagoGtkIconPrecedence precedence)
{
	GalagoAccount *account;
	GalagoService *service;
	GalagoPerson *person;
	GalagoImage *image;
	GList *statuses = NULL, *l;
	GdkPixbuf *pixbuf = NULL;
	int num_emblems, i;
	size_t pixbuf_width, pixbuf_height;

	g_return_val_if_fail(presence   != NULL,                 NULL);
	g_return_val_if_fail(precedence != GALAGO_GTK_ICON_NONE, NULL);

	account = galago_presence_get_account(presence);
	service = galago_account_get_service(account);
	person  = galago_account_get_person(account);

	if (precedence == GALAGO_GTK_ICON_PHOTO &&
		(image = galago_person_get_photo(person, TRUE)) != NULL)
	{
		pixbuf = galago_gdk_pixbuf_new_from_image(image, width, height);
	}
	else if (precedence >= GALAGO_GTK_ICON_AVATAR &&
			 (image = galago_account_get_avatar(account, TRUE)) != NULL)
	{
		pixbuf = galago_gdk_pixbuf_new_from_image(image, width, height);
	}
	else if (precedence == GALAGO_GTK_ICON_GENERIC ||
			 (precedence >= GALAGO_GTK_ICON_PROTOCOL &&
			  (pixbuf = galago_gdk_pixbuf_new_from_service(service, width,
														   height)) == NULL))
	{
		pixbuf = galago_gdk_pixbuf_new_generic(width, height);
	}

	if (pixbuf == NULL)
		return NULL;

	statuses = galago_presence_get_statuses(presence);

	pixbuf_width  = gdk_pixbuf_get_width(pixbuf);
	pixbuf_height = gdk_pixbuf_get_height(pixbuf);

	if (pixbuf_width <= 24 || pixbuf_height <= 24)
		num_emblems = 1;
	else
		num_emblems = MIN(2, g_list_length(statuses));

	for (i = 0, l = statuses;
		 i < num_emblems && l != NULL;
		 i++, l = l->next)
	{
		GalagoStatus *status = (GalagoStatus *)l->data;
		GdkPixbuf *emblem = NULL;
		char *emblem_file;
		char *path;

		emblem_file = g_strdup_printf("%s.png", galago_status_get_id(status));
		path = get_pixmap_path(emblem_file);
		g_free(emblem_file);

		if (path != NULL)
		{
			emblem = gdk_pixbuf_new_from_file(path, NULL);

			g_free(path);
		}

		if (emblem != NULL)
		{
			int emblem_width, emblem_height;
			int x, y;
			double scale_x, scale_y;

			if (pixbuf_width <= 24 || pixbuf_height <= 24)
			{
				emblem_width  = 10;
				emblem_height = 10;
				scale_x = 0.6;
				scale_y = 0.6;
			}
			else
			{
				emblem_width  = gdk_pixbuf_get_width(emblem);
				emblem_height = gdk_pixbuf_get_height(emblem);
				scale_x = 1.0;
				scale_y = 1.0;
			}

			x = pixbuf_width  - ((i + 1) * emblem_width) - (i * 5);
			y = pixbuf_height - emblem_height;

			gdk_pixbuf_composite(emblem, pixbuf,
								 x, y,
								 emblem_width, emblem_height,
								 x, y,
								 scale_x, scale_y,
								 GDK_INTERP_BILINEAR,
								 255);

			g_object_unref(emblem);
		}
	}

	if (galago_presence_has_status_type(presence, GALAGO_STATUS_OFFLINE))
		gdk_pixbuf_saturate_and_pixelate(pixbuf, pixbuf, 0.0, FALSE);
	else if (galago_presence_is_idle(presence))
		gdk_pixbuf_saturate_and_pixelate(pixbuf, pixbuf, 0.25, FALSE);

	return pixbuf;
}

GdkPixbuf *
galago_gdk_pixbuf_new_from_presence_with_size(
	const GalagoPresence *presence, GtkIconSize size,
	GalagoGtkIconPrecedence precedence)
{
	int width, height;

	g_return_val_if_fail(presence   != NULL,                  NULL);
	g_return_val_if_fail(size       != GTK_ICON_SIZE_INVALID, NULL);
	g_return_val_if_fail(precedence != GALAGO_GTK_ICON_NONE,  NULL);

	if (!gtk_icon_size_lookup(size, &width, &height))
	{
		width  = -1;
		height = -1;
	}

	return galago_gdk_pixbuf_new_from_presence(presence, width, height,
											   precedence);
}

GdkPixbuf *
galago_gdk_pixbuf_new_from_person(const GalagoPerson *person,
								  int width, int height,
								  GalagoGtkIconPrecedence precedence)
{
	GalagoAccount *account;

	g_return_val_if_fail(person     != NULL,                  NULL);
	g_return_val_if_fail(precedence != GALAGO_GTK_ICON_NONE,  NULL);

	account = galago_person_get_priority_account(person);

	if (account == NULL)
		return NULL;

	return _new_pixbuf_from_account(account, width, height, precedence);
}

GdkPixbuf *
galago_gdk_pixbuf_new_from_person_with_size(const GalagoPerson *person,
											GtkIconSize size,
											GalagoGtkIconPrecedence precedence)
{
	int width, height;

	g_return_val_if_fail(person     != NULL,                  NULL);
	g_return_val_if_fail(size       != GTK_ICON_SIZE_INVALID, NULL);
	g_return_val_if_fail(precedence != GALAGO_GTK_ICON_NONE,  NULL);

	if (!gtk_icon_size_lookup(size, &width, &height))
	{
		width  = -1;
		height = -1;
	}

	return galago_gdk_pixbuf_new_from_person(person, width, height,
											 precedence);
}
