/**
 * =========================================
 * LibFormula : a free Java formula library
 * =========================================
 *
 * Project Info:  http://reporting.pentaho.org/libformula/
 *
 * (C) Copyright 2006-2007, by Pentaho Corporation and Contributors.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * [Java is a trademark or registered trademark of Sun Microsystems, Inc.
 * in the United States and other countries.]
 *
 *
 * ------------
 * $Id: ConcatOperator.java 2887 2007-06-06 17:07:52Z taqua $
 * ------------
 * (C) Copyright 2006-2007, by Pentaho Corporation.
 */
package org.jfree.formula.operators;

import org.jfree.formula.EvaluationException;
import org.jfree.formula.FormulaContext;
import org.jfree.formula.LibFormulaErrorValue;
import org.jfree.formula.lvalues.TypeValuePair;
import org.jfree.formula.typing.TypeRegistry;
import org.jfree.formula.typing.coretypes.TextType;

/**
 * Concats two strings operator.
 *
 * @author Thomas Morgner
 */
public class ConcatOperator implements InfixOperator
{
  public ConcatOperator()
  {
  }

  public TypeValuePair evaluate(final FormulaContext context,
                                final TypeValuePair value1,
                                final TypeValuePair value2)
      throws EvaluationException
  {
    final TypeRegistry typeRegistry = context.getTypeRegistry();

    // Error or empty string, that's the question ..
    final Object raw1 = value1.getValue();
    final Object raw2 = value2.getValue();
    if (raw1 == null || raw2 == null)
    {
      throw new EvaluationException(LibFormulaErrorValue.ERROR_NA_VALUE);
    }

    final String text1 = typeRegistry.convertToText(value1.getType(), raw1);
    final String text2 = typeRegistry.convertToText(value2.getType(), raw2);
    if (text1 == null && text2 == null)
    {
      throw new EvaluationException
          (LibFormulaErrorValue.ERROR_INVALID_ARGUMENT_VALUE);
    }
    if (text1 == null)
    {
      return new TypeValuePair(TextType.TYPE, text2);
    }
    if (text2 == null)
    {
      return new TypeValuePair(TextType.TYPE, text1);
    }

    return new TypeValuePair(TextType.TYPE, text1 + text2);
  }

  public int getLevel()
  {
    return 300;
  }


  public String toString()
  {
    return "&";
  }

  public boolean isLeftOperation()
  {
    return true;
  }

  /**
   * Defines, whether the operation is associative. For associative operations,
   * the evaluation order does not matter, if the operation appears more than
   * once in an expression, and therefore we can optimize them a lot better than
   * non-associative operations (ie. merge constant parts and precompute them
   * once).
   *
   * @return true, if the operation is associative, false otherwise
   */
  public boolean isAssociative()
  {
    return false;
  }

}
