/*
  CLAW - a C++ Library Absolutely Wonderful

  CLAW is a free library without any particular aim but being useful to 
  anyone.

  Copyright (C) 2005-2008 Julien Jorge

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: julien_jorge@yahoo.fr
*/
/**
 * \file arguments.cpp
 * \brief Implementation of the  claw::arguments class.
 * \author Julien Jorge
 */
#include <sstream>
#include <claw/arguments.hpp>
#include <claw/assert.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
claw::arguments::arguments()
  : m_program_name("<unknow>")
{

} // arguments::arguments()

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param argc Number of arguments.
 * \param argv Arguments.
 *
 * You should construct an instance with the parameters given to your function
 * main(). The constructor will remove all supported arguments from argv.
 */
claw::arguments::arguments( int& argc, char** &argv )
{
  parse(argc, argv);
} // arguments::arguments()

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param argc Number of arguments.
 * \param argv Arguments.
 * \param allowed The set of allowed arguments.
 *
 * You should construct an instance with the parameters given to your function
 * main(). The constructor will remove all supported arguments from argv.
 */
claw::arguments::arguments(int& argc, char** &argv,
                           const claw::math::ordered_set<std::string>& allowed )

{
  parse(argc, argv, allowed);
} // arguments::arguments()

/*----------------------------------------------------------------------------*/
/**
 * \brief Parse arguments.
 * \param argc Number of arguments.
 * \param argv Arguments.
 *
 * All supported arguments will be removed from argv.
 */
void claw::arguments::parse( int& argc, char** &argv )
{
  bool stop = false;
  int base = 0;

  if (m_program_name == "")
    {
      m_program_name = argv[0];
      argv[0] = NULL;
      base = 1;
    }

  for (int argi=base; (argi!=argc) && !stop; ++argi)
    {
      std::string arg(argv[argi]);

      if ( (arg[0] == '-') && (arg.length() > 1) )
        {
          if (arg == "--")
            stop = true;
          else
            {
              add_argument( arg );
              argv[argi] = NULL;
            }
        }
    }

  remove_null_arguments( argc, argv );
} // arguments::parse()

/*----------------------------------------------------------------------------*/
/**
 * \brief Parse arguments.
 * \param argc Number of arguments.
 * \param argv Arguments.
 * \param allowed The set of allowed arguments.
 *
 * All supported arguments will be removed from argv.
 */
void claw::arguments::parse
( int& argc, char** &argv,
  const claw::math::ordered_set<std::string>& allowed )
{
  bool stop = false;
  int base = 0;

  if (m_program_name == "")
    {
      m_program_name = argv[0];
      argv[0] = NULL;
      base = 1;
    }

  for (int argi=base; (argi!=argc) && !stop; ++argi)
    {
      std::string arg(argv[argi]);

      if ( (arg[0] == '-') && (arg.length() > 1) )
        {
          if (arg == "--")
            stop = true;
          else
            {
              std::string name, value;
              split_argument( arg, name, value );

              if ( allowed.find( name ) != allowed.end() )
                {
                  add_argument( arg );
                  argv[argi] = NULL;
                }
            }
        }
    }

  remove_null_arguments( argc, argv );
} // arguments::parse()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if a value is associated to an argument.
 * \param arg_name The name of the argument to test.
 */
bool claw::arguments::has_value( const std::string& arg_name ) const
{
  return m_pairs.find( arg_name ) != m_pairs.end();
} // arguments::has_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the name of the program.
 */
const std::string& claw::arguments::get_program_name() const
{
  return m_program_name;
} // arguments::get_program_name()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the boolean state of an argument.
 * \param arg_name The name of the argument to get.
 */
bool claw::arguments::get_bool( const std::string& arg_name ) const
{
  return m_flags.find( arg_name ) != m_flags.end();
} // arguments::get_bool()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the integer value of an argument.
 * \param arg_name The name of the argument to get.
 * \pre has_value(arg_name)
 */
int claw::arguments::get_integer( const std::string& arg_name ) const
{
  CLAW_ASSERT( has_value(arg_name),
               "arguments::get_integer(): argument is not set." );

  std::istringstream iss( m_pairs.find( arg_name )->second );
  int val;
  iss >> val;

  return val;
} // arguments::get_integer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the real value of an argument.
 * \param arg_name The name of the argument to get.
 * \pre has_value(arg_name)
 */
double claw::arguments::get_real( const std::string& arg_name ) const
{
  CLAW_ASSERT( has_value(arg_name),
               "arguments::get_real(): argument is not set." );

  std::istringstream iss( m_pairs.find( arg_name )->second );
  double val;
  iss >> val;

  return val;
} // arguments::get_real()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the string value of an argument.
 * \param arg_name The name of the argument to get.
 * \pre has_value(arg_name)
 */
const std::string&
claw::arguments::get_string( const std::string& arg_name ) const
{
  CLAW_ASSERT( has_value(arg_name),
               "arguments::get_string(): argument is not set." );

  return m_pairs.find( arg_name )->second;
} // arguments::get_string()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add an argument in our list.
 *
 * You can use this method to set default values to the parameters of your
 * program, before calling parse_arguments.
 *
 * \param arg The argument to add.
 * \pre (arg != "--") && (arg[0] == '-')
 */
void claw::arguments::add_argument( const std::string& arg )
{
  CLAW_ASSERT( arg != "--", "arguments::add_argument(): arg can't be '--'" );
  CLAW_ASSERT( arg[0] == '-',
               "arguments::add_argument(): arg must begin by '-'" );
  
  std::string name, value;
  split_argument(arg, name, value);

  if ( value == "" )
    m_flags.insert( arg );
  else
    m_pairs[ name ] = value;
} // arguments::add_argument()

/*----------------------------------------------------------------------------*/
/**
 * \brief Split an argument to get its name and its value.
 * \param arg The argument to split.
 * \param name (out) The name of the argument.
 * \param value (out) The value of the argument.
 * \pre (arg != "--") && (arg[0] == '-')
 */
void claw::arguments::split_argument( const std::string& arg, std::string& name,
                                      std::string& value ) const
{
  CLAW_ASSERT( arg != "--", "arguments::split_argument(): arg can't be '--'" );
  CLAW_ASSERT( arg[0] == '-',
               "arguments::split_argument(): arg must begin by '-'" );
  
  std::string::size_type pos = arg.find("=");

  if ( pos == std::string::npos )
    {
      name = arg;
      value = "";
    }
  else
    {
      name = arg.substr(0, pos);
      value = arg.substr(pos+1, arg.length() - pos - 1);
    }
} // arguments::split_argument()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove all NULL arguments from argv and update argc.
 * \param argc The number of arguments.
 * \param argv The arguments.
 */
void claw::arguments::remove_null_arguments( int& argc, char** &argv ) const
{
  unsigned int c=0; // number of non-NULL arguments
  
  for (int i=0; i!=argc; ++i)
    if ( argv[i] != NULL )
      ++c;
    else
      {
        bool ok = false;
        int j=i;
        
        while ( (j!=argc) && !ok )
          if ( argv[j] == NULL )
            ++j;
          else
            ok = true;

        if (ok)
          {
            argv[i] = argv[j];
            argv[j] = NULL;
            ++c;
          }
      }

  if ( c > 0 )
    if ( (std::string(argv[c-1]) == "--") )
      --c;
    
  argc=c;
} // arguments::remove_null_arguments()
