/** vim:sw=4:sts=4
 * Library to use the Gtk2 widget library from Lua 5.1
 * Copyright (C) 2007 Wolfgang Oertl
 *
 * Handle type conversion from Lua to and from C (or Gtk).
 *
 * Exported symbols:
 *   luagtk_type_name
 *   luagtk_empty_table
 *   ffi_type_lua2ffi
 *   ffi_type_ffi2lua
 *   ffi_type_lua2struct
 *   ffi_type_struct2lua
 *   ffi_type_map
 *   ffi_type_names
 */
#include "luagtk.h"
#include "luagtk_ffi.h"

#include <lauxlib.h>
#include <string.h>	    // strcmp
#include <stdlib.h>	    // strtol

// The order in these ENUMs must match the function pointer arrays at the
// end of this file.
enum { LUA2FFI_BOOL=1, LUA2FFI_LONG, LUA2FFI_ENUM, LUA2FFI_LONGLONG,
    LUA2FFI_DOUBLE, LUA2FFI_FLOAT, LUA2FFI_UCHAR,
    LUA2FFI_CHAR_PTR, LUA2FFI_PTR,
    LUA2FFI_STRUCT_PTR, LUA2FFI_VARARG,
    LUA2FFI_INT_PTR, LUA2FFI_UNSIGNED_INT_PTR,
    LUA2FFI_LONG_UNSIGNED_INT_PTR, LUA2FFI_VOID_PTR,
    LUA2FFI_FUNC_PTR, LUA2FFI_STRUCT_PTR_PTR, LUA2FFI_CHAR_PTR_PTR,
    LUA2FFI_ENUM_PTR, LUA2FFI_ENUM_PTR_PTR };
// max. items: 31

enum { FFI2LUA_VOID=1, FFI2LUA_LONG, FFI2LUA_BOOL, FFI2LUA_DOUBLE,
    FFI2LUA_ENUM, FFI2LUA_UCHAR, FFI2LUA_CHAR_PTR, FFI2LUA_STRUCT_PTR, 
    FFI2LUA_LONG_PTR, FFI2LUA_VOID_PTR, FFI2LUA_INT_PTR,
    FFI2LUA_UNSIGNED_INT_PTR, FFI2LUA_LONG_UNSIGNED_INT_PTR,
    FFI2LUA_STRUCT_PTR_PTR, FFI2LUA_CHAR_PTR_PTR, FFI2LUA_ENUM_PTR,
    FFI2LUA_ENUM_PTR_PTR };
// max. items: 31

enum { LUA2STRUCT_LONG=1, LUA2STRUCT_ENUM, LUA2STRUCT_FUNC_PTR,
    LUA2STRUCT_DOUBLE };
// max items: 15

enum { STRUCT2LUA_LONG=1, STRUCT2LUA_STRUCT_PTR, STRUCT2LUA_STRUCT,
    STRUCT2LUA_CHAR_PTR, STRUCT2LUA_PTR, STRUCT2LUA_ENUM, STRUCT2LUA_DOUBLE,
    STRUCT2LUA_VOID_PTR };
// max items: 15


/**
 * Given a pointer, a bit offset and a bit length, retrieve the value.
 * This is used for non 8 bit things, like single bits.  The destination
 * is a long.
 *
 * Tested on 32 and 64 bit architectures.
 */
#define BITS_PER_INT (sizeof(unsigned long int)*8)
static void get_bits_unaligned(lua_State *L, const unsigned char *ptr,
    int bitofs, int bitlen, char *dest)
{
    unsigned long int val;

    if (bitlen && bitlen <= BITS_PER_INT) {
	ptr += bitofs >> 3;
	bitofs = bitofs & 7;
	val = (* (unsigned long*) ptr) >> bitofs;
	if (bitlen < BITS_PER_INT)
	    val &= (1L << bitlen) - 1;
	* (unsigned long*) dest = val;
	return;
    }


    luaL_error(L, "%s access to attribute of size %d not supported",
	msgprefix, bitlen);
}
#undef BITS_PER_INT

/**
 * Retrieve an arbitrarily long memory block, which must be byte aligned
 * and the length must be a multiple of 8 bits.  Note that if "bitlen" is
 * less than the variable "dest" points to, the additional bytes won't be
 * initialized.  Therefore, set them to zero before calling this function.
 *
 * If "bitlen" is not a multiple of 8, then get_bits_unaligned will be
 * called, which assumes that "dest" is an unsigned long integer.
 */
static inline void get_bits_long(lua_State *L, const unsigned char *ptr,
    int bitofs, int bitlen, char *dest)
{
    if (((bitofs | bitlen) & 7) == 0)
	memcpy(dest, ptr + (bitofs >> 3), bitlen >> 3);
    else
	get_bits_unaligned(L, ptr, bitofs, bitlen, dest);
}



/**
 * Write a numerical field within a structure.
 *
 * Note. Writing to fields spanning more than one "unsigned long int" is not
 * supported - but this shouldn't happen anyway.
 *
 * @param ptr Pointer to the start of the structure
 * @param bitofs Offset of the field within the structure
 * @param bitlen Length of the field
 * @param val value to write into the field.
 */
static void set_bits(const unsigned char *ptr, int bitofs, int bitlen,
    unsigned long int val)
{
    unsigned long int v, mask;

    /* do byte aligned accesses */
    ptr += bitofs / 8;
    bitofs = bitofs % 8;

    if (bitlen == 0 || bitlen+bitofs > sizeof(v)*8) {
	printf("%s write to attribute of size %d not supported\n",
	    msgprefix, bitlen);
	return;
    }

    mask = (bitlen < sizeof(mask)*8) ? ((1L << bitlen) - 1) : -1L;
    mask <<= bitofs;

    // fetch the old value, replace bits with new value, write back.
    v = * (unsigned long int*) ptr;
    v &= ~mask;
    v |= (val << bitofs) & mask;
    * (unsigned long int*) ptr = v;
}

static inline void set_bits_long(lua_State *L, unsigned char *dest, int bitofs,
    int bitlen, const char *src)
{
    if (((bitofs | bitlen) & 7) == 0)
	memcpy(dest + (bitofs >> 3), src, (bitlen >> 3));
    else
	luaL_error(L, "%s unaligned access in set_bits_long", msgprefix);
}



// ----- LUA2FFI FUNCTIONS -----
// these functions retrieve a value from Lua and store it in a ffi
// value

/**
 * These functions retrieve a value from the Lua stack and store it into
 * the provided target.  They are used to convert the arguments to
 * C functions called from Lua, but also to convert the return values
 * of callbacks.
 */

static int lua2ffi_bool(struct argconv_t *ar)
{
    luaL_checktype(ar->L, ar->index, LUA_TBOOLEAN);
    ar->arg->l = (long) lua_toboolean(ar->L, ar->index);
    return 1;
}

// might be a number, or an ENUM or FLAGS.
static int lua2ffi_long(struct argconv_t *ar)
{
    if (ar->lua_type == LUA_TUSERDATA) {
	struct luagtk_enum_t *e = (struct luagtk_enum_t*)
	    luaL_checkudata(ar->L, ar->index, ENUM_META);
	ar->arg->l = e->value;
	return 1;
    }

    // If it's not userdata, must be something else useable as integer,
    // like number, or a string containing a number.
    ar->arg->l = (long) luaL_checkinteger(ar->L, ar->index);
    return 1;
}

static int lua2ffi_uchar(struct argconv_t *ar)
{
    lua_State *L = ar->L;

    switch (ar->lua_type) {
	case LUA_TNUMBER:
	    ar->arg->uc = (unsigned char) lua_tonumber(L, ar->index);
	    break;
	case LUA_TBOOLEAN:
	    ar->arg->uc = (unsigned char) lua_toboolean(L, ar->index);
	    break;
	case LUA_TSTRING:
	    ar->arg->uc = (unsigned char) lua_tostring(L, ar->index)[0];
	    break;
	default:
	    luaL_error(L, "%s can't convert Lua type %s to char",
		msgprefix, lua_typename(L, ar->lua_type));
    }
    return 1;
}

/**
 * ENUMs can be represented by simple numbers, or by a userdata
 * that additionally contains the ENUM type, which can be used for
 * type checking.
 */
static int lua2ffi_enum(struct argconv_t *ar)
{
    switch (ar->lua_type) {
	case LUA_TNUMBER:
	    ar->arg->l = (long) lua_tonumber(ar->L, ar->index);

	    // for zero it is probably OK; like for gtk_table_attach, when
	    // xoptions should be zero.
	    if (ar->arg->l != 0)
		call_info_msg(ar->ci, LUAGTK_WARNING,
		    "Arg %d enum (type %s) given as number\n", ar->func_arg_nr,
		    TYPE_NAME(ar->type));
	    return 1;
	
	case LUA_TUSERDATA:;
	    struct luagtk_enum_t *e = (struct luagtk_enum_t*) luaL_checkudata(ar->L,
		ar->index, ENUM_META);
	    if (ar->type_idx != e->type_idx) {
		call_info_msg(ar->ci, LUAGTK_WARNING,
		    "Arg %d enum expects type %s, given %s\n", ar->func_arg_nr,
		    TYPE_NAME(ar->type),
		    TYPE_NAME(type_list + e->type_idx));
	    }
	    ar->arg->l = e->value;
	    return 1;
    }

    return 0;
}

static int lua2ffi_longlong(struct argconv_t *ar)
{
    ar->arg->ll = (long long) luaL_checknumber(ar->L, ar->index);
    return 1;
}

static int lua2ffi_double(struct argconv_t *ar)
{
    ar->arg->d = (double) luaL_checknumber(ar->L, ar->index);
    return 1;
}

static int lua2ffi_float(struct argconv_t *ar)
{
    ar->arg->f = (float) luaL_checknumber(ar->L, ar->index);
    return 1;
}

// XXX Might need to g_strdup() when returning a string to Gtk from a closure
// invocation.  The Lua string will be garbage collected eventually, but Gtk
// might continue to reference the string!
static int lua2ffi_char_ptr(struct argconv_t *ar)
{
    ar->arg->p = ar->lua_type == LUA_TNIL ? NULL
	: (void*) luaL_checkstring(ar->L, ar->index);
    return 1;
}

/**
 * Generic pointer - only NIL can be transported
 * XXX This function should eventually be replaced by specialized functions
 * for each pointer type, like "short*" etc.
 *
 * If required, various other Lua types could be converted: LUA_TUSERDATA,
 * LUA_TLIGHTUSERDATA, LUA_TSTRING, or even LUA_TFUNCTION?
 */
static int lua2ffi_ptr(struct argconv_t *ar)
{
    if (ar->lua_type == LUA_TNIL) {
	ar->arg->p = NULL;
	return 1;
    }
    call_info_msg(ar->ci, LUAGTK_WARNING,
	"Arg #%d type %s not supported, replaced by NULL\n",
	ar->func_arg_nr, FTYPE_NAME(&ffi_type_map[ar->ffi_type_nr]));
    ar->arg->p = NULL;
    return 0;
}


/**
 * Handle pointers to structures/unions/widgets/GValues.
 */
static int lua2ffi_struct_ptr(struct argconv_t *ar)
{
    lua_State *L = ar->L;
    const struct type_info *ti = ar->type;

    // The GValue is a special type of structure; it can hold almost any type
    // of value, therefore even numbers or strings can be converted to it.

    if (!strcmp(TYPE_NAME(ti), "GValue")) {
	GValue *gvalue = (GValue*) call_info_alloc_item(ar->ci, sizeof *gvalue);
	ar->arg->p = luagtk_set_gvalue(L, gvalue, ar->index, ar->lua_type);
	return 1;
    }

    switch (ar->lua_type) {

	// NIL is always ok
	case LUA_TNIL:
	    ar->arg->p = NULL;
	    return 1;

	// might be a widget or a structure.
	case LUA_TUSERDATA:

	    if (!lua_getmetatable(L, ar->index)) {
		printf("%s widget has no meta table.\n", msgprefix);
		return 0;
	    }

	    // The userdata's metatable should be of a widget, in this case
	    // it has the field _classname.  Other userdatas are not useful,
	    // like ENUMs.

	    lua_pushliteral(L, "_classname");
	    lua_rawget(L, -2);

	    if (lua_isnil(L, -1)) {
		luaL_error(L, "%s arg #%d expects a %s, got some userdata\n",
		    ar->ci->fi->name, ar->func_arg_nr, TYPE_NAME(ar->type));
	    }

	    // The given object might be derived from the required type, which
	    // would be OK.  Check this.
	    const char *is_name = lua_tostring(L, -1);
	    GType is_type = g_type_from_name(is_name);

	    const char *req_name = TYPE_NAME(ar->type);
	    GType req_type = g_type_from_name(req_name);

	    if (is_type != req_type && !g_type_is_a(is_type, req_type)) {
		luaL_error(L, "%s arg #%d expects a %s, got %s\n",
		    ar->ci->fi->name, ar->func_arg_nr, req_name, is_name);
	    }
	    lua_pop(L, 2);

	    // all right.
	    struct widget *w = (struct widget*) lua_topointer(L, ar->index);
	    ar->arg->p = w->p;
	    return 1;

	// other Lua types can't possibly be a structure pointer.
	default:
	    luaL_error(L, "%s incompatible argument #%d for %s (given %s, "
		"required %s)\n",
		msgprefix, ar->func_arg_nr, ar->ci->fi->name,
		lua_typename(L, ar->lua_type), TYPE_NAME(ar->type));
    }

    return 0;
}


/**
 * A func* should be passed to a Gtk function.  A closure is created for the
 * given Lua function.  Note that the memory allocated for it is leaked; it
 * can't be freed automatically because the callback might be called at a
 * later time, even after the function now called returns; think of
 * g_tree_new_full, for example.  OTOH, after g_tree_foreach the wrapper
 * could be freed.  XXX So, what should be done?
 *
 * For callbacks, the func* argument is often followed by a void* argument
 * which is passed to the callback.  This is already handled properly by
 * the void* wrapper, see voidptr.c.
 */
static int lua2ffi_func_ptr(struct argconv_t *ar)
{
    const unsigned char *sig;

    luaL_checktype(ar->L, ar->index, LUA_TFUNCTION);
    sig = GET_PROTOTYPE(ar->type);
    ar->arg->p = luagtk_make_closure(ar->L, ar->index, sig);
    return 1;
}


/**
 * A "struct**" is most likely an output parameter, but may be out/in.
 * Allocate memory for a pointer, and set it to the given value, which may
 * be nil.  When collecting results, the output value will be used.
 */
static int lua2ffi_struct_ptr_ptr(struct argconv_t *ar)
{
    void *ptr = NULL;

    switch (lua_type(ar->L, ar->index)) {
	case LUA_TSTRING:
	    ptr = (void*) lua_tostring(ar->L, ar->index);
	    break;
	
	case LUA_TUSERDATA:
	case LUA_TLIGHTUSERDATA:
	    ptr = (void*) lua_topointer(ar->L, ar->index);
	    break;
	
	case LUA_TNIL:
	    break;
    }

    void **p = (void**) call_info_alloc_item(ar->ci, sizeof(*p));
    *p = ptr;
    ar->arg->p = (void*) p;
    // printf("struct** in: %p/%p\n", p, ptr);

    return 1;
}


/*-
 * A char** argument should point to a memory location where a new char* can
 * be stored (output parameter).
 *
 * Exactly the same thing as a struct**.
 */
#define lua2ffi_char_ptr_ptr lua2ffi_struct_ptr_ptr

/**
 * A table is given as an argument for a vararg.  An optional "type" field
 * tells what type the items should be; default is strings.
 *
 * Problem: somtimes the resulting array must be ended with a NULL pointer;
 * when you specify nil as the last item in a table it is simply discarded.
 * Use gtk.NIL instead of nil in this situation.
 */
static void _lua2ffi_vararg_table(struct argconv_t *ar)
{
    lua_State *L = ar->L;
    int items = lua_objlen(L, ar->index), i;
    int arg_nr = ar->func_arg_nr, type;
    struct call_info *ci = ar->ci;
    const char **a;

    /* determine the item type. */
    lua_pushliteral(L, "type");
    lua_rawget(L, ar->index);
    if (!lua_isnil(L, -1)) {
	const char *type_str = lua_tostring(L, -1);
	printf("vararg type string is %s\n", type_str);
	// XXX so what...  the rest assumes string.
    }
    lua_pop(L, 1);

    /* allocate output array */
    ci->argtypes[arg_nr] = &ffi_type_pointer;
    a = (const char**) call_info_alloc_item(ci,
	sizeof(char*) * items);
    ci->ffi_args[arg_nr].p = (void*) a;

    /* copy the values */
    for (i=0; i<items; i++) {
	lua_rawgeti(L, ar->index, i + 1);
	type = lua_type(L, -1);
	switch (type) {
	    case LUA_TSTRING:
	    a[i] = lua_tostring(L, -1);
	    break;

	    case LUA_TLIGHTUSERDATA:
	    a[i] = lua_topointer(L, -1);    // works for gtk.NIL
	    break;

	    case LUA_TNUMBER:
	    break;

	}

	lua_pop(L, 1);
    }
}

/**
 * Handle a vararg argument.  This must be the last argument for the function,
 * and collects all remaining given parameters (zero or more).
 *
 * This function must handle all Lua types and find an appropriate FFI type
 * for each of them.  The ci->argtypes must be set, too, for each argument.
 */
static int lua2ffi_vararg(struct argconv_t *ar)
{
    int stack_top = lua_gettop(ar->L), type, arg_nr = ar->func_arg_nr;
    struct call_info *ci = ar->ci;
    lua_State *L = ar->L;

    for (; ar->index <= stack_top; ar->index ++, arg_nr ++) {
	type = lua_type(L, ar->index);
	ci->argvalues[arg_nr] = &ci->ffi_args[arg_nr].l;

	switch (type) {

	    case LUA_TBOOLEAN:
		ci->argtypes[arg_nr] = &ffi_type_uint;
		ci->ffi_args[arg_nr].l = (long) lua_toboolean(L, ar->index);
		break;

	    case LUA_TNUMBER:
		ci->argtypes[arg_nr] = &ffi_type_uint;
		ci->ffi_args[arg_nr].l = (long) lua_tonumber(L, ar->index);
		break;

	    case LUA_TSTRING:
		ci->argtypes[arg_nr] = &ffi_type_pointer;
		ci->ffi_args[arg_nr].p = (void*) lua_tostring(L, ar->index);
		break;

	    case LUA_TNIL:
		ci->argtypes[arg_nr] = &ffi_type_pointer;
		ci->ffi_args[arg_nr].p = NULL;
		break;

	    case LUA_TLIGHTUSERDATA:
		ci->argtypes[arg_nr] = &ffi_type_pointer;
		ci->ffi_args[arg_nr].p = (void*) lua_topointer(L, ar->index);
		break;

	    // can be: enum, flags, object
	    case LUA_TUSERDATA:
		ci->argtypes[arg_nr] = luagtk_userdata_to_ffi(L, ar->index,
		    &ci->ffi_args[arg_nr], 0) == 0 ? &ffi_type_pointer
		    : &ffi_type_uint;
		break;

	    // Convert a table into a array of char*.
	    case LUA_TTABLE:
		ar->func_arg_nr = arg_nr;
		_lua2ffi_vararg_table(ar);
		break;

	    default:
		call_info_msg(ci, LUAGTK_WARNING,
		    "Arg %d: Unhandled vararg type %s\n", arg_nr+1,
		    lua_typename(L, type));
	}
    }

    ar->func_arg_nr = arg_nr - 1;
    return 1;
}


/**
 * Store an integer of a given ffi_type at *p.
 * XXX might be an ENUM etc.
 *
 * @param L  Lua State
 * @param p  Pointer to the destination; can be 4 or 8 bytes long
 * @param type_idx  LUA2FFI_*INT_PTR to specify the type at *p
 * @param index  Where the number is on the Lua stack
 */
static void _store_int(lua_State *L, void *p, int type_idx, int index)
{
    switch (type_idx) {
	case LUA2FFI_INT_PTR:
	* (int*) p = lua_tonumber(L, index);
	break;

	case LUA2FFI_UNSIGNED_INT_PTR:
	* (unsigned int*) p = lua_tonumber(L, index);
	break;

	case LUA2FFI_LONG_UNSIGNED_INT_PTR:
	* (unsigned long int*) p = lua_tonumber(L, index);
	break;

	default:
	luaL_error(L, "%s internal error; unhandled ffi_type %d in "
	    "_store_int", msgprefix, type_idx);
    }
}

/**
 * Pointer to an integer - can be input or output.
 * This works for int, unsigned int, long int, unsigned long int.
 *
 * Input, e.g. for gdk_pango_layout_get_clip_region.  Use as such when an array
 * (of numbers) is given.
 * Output in other cases.  Initialize with whatever the user passed as
 * parameter.
 */
static int lua2ffi_int_ptr(struct argconv_t *ar)
{
    lua_State *L = ar->L;
    int index = ar->index;
    int bytes = ar->arg_type->bit_len >> 3;
    int type_idx = ar->arg_type->lua2ffi_idx;

    /* If no table is given, then this is an output value.  Allocate some
     * space for it, and set the pointer to it. */
    if (ar->lua_type != LUA_TTABLE) {
	ar->arg->p = call_info_alloc_item(ar->ci, bytes);
	_store_int(L, ar->arg->p, type_idx, index);
	ar->ci->arg_flags[ar->func_arg_nr] = 1;
	return 1;
    }

    /* otherwise, this is an input.  Allocate memory large enough for
     * all table items, and copy them. */
    int i, n;
    n = luaL_getn(L, index);
    char *a = (char*) call_info_alloc_item(ar->ci, bytes * n);
    ar->arg->p = (void*) a;

    for (i=0; i<n; i++) {
	lua_rawgeti(L, index, i+1);
	_store_int(L, (void*) a, type_idx, -1);
	lua_pop(L, 1);
	a += bytes;
    }
    return 1;
}

/**
 * Array of ENUMs - as input
 */
static int lua2ffi_enum_ptr(struct argconv_t *ar)
{
    lua_State *L = ar->L;
    int index = ar->index;

    switch (ar->lua_type) {
	case LUA_TNIL:
	    ar->arg->p = NULL;
	    return 1;
	
	// initialize with a number.  should be zero.
	case LUA_TNUMBER:
	    ar->arg->p = call_info_alloc_item(ar->ci, sizeof(int));
	    * (int*) ar->arg->p = lua_tonumber(L, index);
	    return 1;
	
	// array of enums
	case LUA_TTABLE:;
	    int i, n, *a, type_idx;
	    const struct type_info *ti;
	    n = luaL_getn(L, index);
	    a = (int*) call_info_alloc_item(ar->ci, sizeof(*a) * n);
	    ti = luagtk_type_modify(ar->type, -1);
	    type_idx = ti - type_list;

	    for (i=1; i<=n; i++) {
		lua_rawgeti(L, index, i);
		struct luagtk_enum_t *e = luagtk_enum_get(L, -1,
		    type_idx, 0);
		if (!e)
		    luaL_error(L, "%s table element %d of arg %d is not "
			"an enum of type %s", msgprefix, i, ar->func_arg_nr,
			TYPE_NAME(ti));
		a[i - 1] = e->value;
		lua_pop(L, 1);
	    }

	    ar->arg->p = (void*) a;
	    return 1;

    }

    luaL_argerror(L, ar->func_arg_nr, "must be NIL or table");
    return 0;
}

// enum** - can only be output, and must be given NIL
// the following argument must be int*, and is output too.
static int lua2ffi_enum_ptr_ptr(struct argconv_t *ar)
{
    lua_State *L = ar->L;

    luaL_checktype(L, ar->index, LUA_TTABLE);
    int **a = (int**) call_info_alloc_item(ar->ci, sizeof(*a));
    ar->arg->p = (void*) a;

    // XXX check that next arg is int* -- can't.

    return 1;
}


/** ------- FFI2LUA FUNCTIONS -----
 * These functions take a ffi value and push it onto the Lua stack.  They are
 * required to convert return values from library calls back to Lua values.
 * Note that some "output arguments", like pointers to integers, are handled,
 * too.
 *
 * All of these functions return the number of arguments used; usually one, but
 * sometimes two.
 */

static int ffi2lua_void(struct argconv_t *ar)
{
    return 1;
}

static int ffi2lua_long(struct argconv_t *ar)
{
    lua_pushnumber(ar->L, ar->arg->l);
    return 1;
}

static int ffi2lua_bool(struct argconv_t *ar)
{
    lua_pushboolean(ar->L, ar->arg->l);
    return 1;
}

static int ffi2lua_double(struct argconv_t *ar)
{
    lua_pushnumber(ar->L, ar->arg->d);
    return 1;
}

static int ffi2lua_enum(struct argconv_t *ar)
{
    return luagtk_enum_push(ar->L, ar->arg->l, ar->type_idx);
}

static int ffi2lua_uchar(struct argconv_t *ar)
{
    lua_pushlstring(ar->L, (char*) &ar->arg->uc, 1);
    return 1;
}


/**
 * Only an output value if it is the actual function return value.
 */
static int ffi2lua_char_ptr(struct argconv_t *ar)
{
    char *p = (char*) ar->arg->p;

    if (ar->func_arg_nr == 0) {
	if (p) {
	    lua_pushstring(ar->L, p);
	    // This flag 4 is set in one of the two char* type entries of the
	    // type list.  If set, this means that it is "const char*", and
	    // must not be freed.  See script/parse-xml.lua,
	    // src/char_ptr_handling.txt.
	    if (!(ar->arg_type->flags2 & FFI_CHAR_PTR_CONST))
		g_free(p);
	} else
	    lua_pushnil(ar->L);
    }

    return 1;
}

/* list of functions that return a widget where we need to increase the
 * refcount.
 */
static const char _need_inc_ref[] =
    "gtk_text_tag_table_lookup\0"   /* like a _get function */
    "gtk_text_buffer_create_tag\0"  /* owned by text tag table */
    "\0";


/**
 * Refcounting for GObject derived objects that are not Gtk objects is tricky.
 * These are returned by the creating function with a refcount of 1 but
 * not with a floating reference.  Therefore, g_object_ref_sink must not be
 * called.
 *
 * On the other hand, when an existing GObject derived object is returned by
 * a function, the refcount must be increased, because a new reference to it
 * is held and the refcount will be decreased upon GC of the Lua proxy object.
 *
 * It seems that functions that match the pattern gtk_*_get_ return existing
 * objects.  This is just a guess, I haven't verified them all, and maybe
 * there are others that also return existing objects but don't match this
 * naming pattern.  Currently 181 functions match this:
 * grep -c '^struct\*.* gtk_.*_get_' funclist
 *
 * Note that computing the flags in this way is only relevant when the Lua
 * proxy object doesn't exist yet, but that is determined later.  OTOH
 * luagtk_get_widget doesn't know about the name of the function that returns
 * the widget, so...
 */
static int _determine_flags(struct argconv_t *ar)
{
    // if called without a call info, then this is called for a
    // Gtk -> Lua callback.  All objects provided are, of course, not new.
    if (!ar->ci)
	return FLAG_NOT_NEW_OBJECT;

    const char *name = ar->ci->fi->name;

    if (!strncmp(name, "gtk_", 4) && strstr(name, "_get_")) {
	// printf("%p %s returns not new object\n", ar->arg->p, name);
	return FLAG_NOT_NEW_OBJECT;
    }

    /* check for functions that return non-new widgets */
    const char *p = _need_inc_ref;
    while (*p) {
	if (!strcmp(name, p)) {
	    return FLAG_NOT_NEW_OBJECT;
	}
	p += strlen(p) + 1;
    }

    return FLAG_NEW_OBJECT;
}


/**
 * When the requested type is derived from GObject, return 0, which lets
 * luagtk_get_widget figure out the actual type of the object.
 */
static int _guess_type_idx(int type_idx)
{
    static GType go_type = 0;

    if (G_UNLIKELY(go_type == 0))
	go_type = g_type_from_name("GObject");

    if (type_idx) {
	const char *s = TYPE_NAME(type_list + type_idx);
	GType my_type = g_type_from_name(s);
	if (my_type && g_type_is_a(my_type, go_type))
	    return 0;
    }

    return type_idx;
}

/**
 * Convert a structure pointer to a Lua value.
 */
static int ffi2lua_struct_ptr(struct argconv_t *ar)
{
    const char *s;

    // primary return value of the function?
    if (ar->func_arg_nr == 0) {
	luagtk_get_widget(ar->L, ar->arg->p,
	    _guess_type_idx(ar->type_idx), _determine_flags(ar));
	return 1;
    }

    // other struct pointer - only return if it is a GValue.
    s = TYPE_NAME(ar->type);
    if (!strcmp(s, "GValue")) {
	GValue *gvalue = (GValue*) ar->arg->p;
	if (!gvalue)
	    lua_pushnil(ar->L);
	else
	    luagtk_push_gvalue(ar->L, gvalue);
	return 1;
    }

    return 1;
}



// Get functions for output arguments; mark these specially?

// Sometimes arguments to a function can be used as OUTPUT values; push
// these to the Lua stack after they return.

static int ffi2lua_long_ptr(struct argconv_t *ar)
{
    lua_pushnumber(ar->L, * (long*) ar->arg->p);
    return 1;
}

// pointer to enum; might be output
static int ffi2lua_enum_ptr(struct argconv_t *ar)
{
    if (ar->arg->p) {
	int v = * (int*) ar->arg->p;
	luagtk_enum_push(ar->L, v, ar->type_idx);
    } else
	lua_pushnil(ar->L);
    return 1;
}

/**
 * Remove all entries of the given table.  Generally it would be easier to
 * just create a new table, but for tables given as arguments it is required
 * to write into them.
 */
void luagtk_empty_table(lua_State *L, int index)
{
    for (;;) {
	lua_pushnil(L);
	if (!lua_next(L, index))    // key, value (or nothing)
	    break;
	lua_pop(L, 1);		    // key
	lua_pushnil(L);		    // key, nil
	lua_rawset(L, index);	    // x
    }
}

/**
 * Handle enum** plus int* as output parameters.  This currently occurs only
 * in three API functions:
 *
 *  gdk_query_visual_types
 *  gtk_icon_set_get_sizes
 *  pango_tab_array_get_tabs -- requires an override, not handled here.
 *
 * The table given to the function call is filled with the result.
 */
static int ffi2lua_enum_ptr_ptr(struct argconv_t *ar)
{
    lua_State *L = ar->L;

    luaL_checktype(L, ar->index, LUA_TTABLE);
    luaL_checktype(L, ar->index + 1, LUA_TNUMBER);

    int *a = * (int**) ar->arg->p;
    int cnt = * (int*) (ar->arg+1)->p;
    int i;

    // Remove another level of indirection.
    ar->type = luagtk_type_modify(ar->type, -1);
    ar->type_idx = ar->type - type_list;

    luagtk_empty_table(L, ar->index);

    for (i=0; i<cnt; i++) {
	luagtk_enum_push(L, a[i], ar->type_idx);
	lua_rawseti(L, ar->index, i + 1);
    }

    // the output array must sometimes be freed, sometimes not.
    const char *fname = ar->ci->fi->name;
    if (!strcmp(fname, "gtk_icon_set_get_sizes"))
	g_free(a);

    // used up two items
    return 2;
}


// generic pointer - try to make a widget out of it.  It might also be a
// magic wrapper, though.
static int ffi2lua_void_ptr(struct argconv_t *ar)
{
    void *p = ar->arg->p;

    if (!p) {
	lua_pushnil(ar->L);
	return 1;
    }

    if (luagtk_is_vwrapper(p))
	return luagtk_push_value_wrapper(ar->L, p);

    // is this a new object, or not??  guess not.
    luagtk_get_widget(ar->L, ar->arg->p, 0, FLAG_NOT_NEW_OBJECT);
    if (!lua_isnil(ar->L, -1))
	return 1;

    // The return pointer wasn't nil, but get_widget couldn't make
    // anything of it?
    if (ar->arg->p && runtime_flags & RUNTIME_WARN_RETURN_VALUE) {
	call_info_msg(ar->ci, LUAGTK_WARNING,
	    "Return value of arg %d (void*) discarded.\n", ar->func_arg_nr);
    }

    return 1;
}


/**
 * A int* type parameter can be an output parameter.  If so, push the returned
 * value onto the stack.  Note: the arg_flags for this parameter is set by
 * lua2ffi_int_ptr if a single integer was passed.
 */
static int ffi2lua_int_ptr(struct argconv_t *ar)
{
    if (ar->func_arg_nr == 0)
	luaL_error(ar->L, "int* not supported as return value\n");
    // flag set if it is an output parameter.
    if (ar->ci->arg_flags[ar->func_arg_nr])
	lua_pushnumber(ar->L, * (int*) ar->arg->p);
    return 1;
}

static int ffi2lua_unsigned_int_ptr(struct argconv_t *ar)
{
    if (ar->func_arg_nr == 0)
	luaL_error(ar->L, "unsigned int* not supported as return value\n");
    if (ar->ci->arg_flags[ar->func_arg_nr])
	lua_pushnumber(ar->L, * (unsigned int*) ar->arg->p);
    return 1;
}

static int ffi2lua_long_unsigned_int_ptr(struct argconv_t *ar)
{
    if (ar->func_arg_nr == 0)
	luaL_error(ar->L, "long_unsigned int* not supported as return value\n");
    if (ar->ci->arg_flags[ar->func_arg_nr])
	lua_pushnumber(ar->L, * (long unsigned int*) ar->arg->p);
    return 1;
}

/**
 * A Gtk function filled in a SomeStruct **p pointer.  Assume that this is a
 * regular object.
 *
 * XXX Some functions return an existing object, others create a new one,
 * and even others allocate a memory block and return that.  It seems that
 * there's no way to automatically figure out the right way.
 *
 * Returns an existing object:
 *  GtkTreeModel** (gtk_tree_selection_get_selected)
 *
 * Returns a new object:
 *
 * Returns a newly allocate memory block that must be freed:
 *  GError**, GdkRectangle**, PangoAttrList**
 *
 * Returns an existing memory block that must not be freed:
 */
static int ffi2lua_struct_ptr_ptr(struct argconv_t *ar)
{
    const char *name = TYPE_NAME(ar->type);

    int flags = FLAG_NOT_NEW_OBJECT;

    if (!strcmp(name, "GError") || !strcmp(name, "GdkRectangle"))
	flags = FLAG_ALLOCATED | FLAG_NEW_OBJECT;

    else if (!strcmp(name, "PangoAttrList"))
	flags = FLAG_NEW_OBJECT;

    luagtk_get_widget(ar->L, * (void**) ar->arg->p, ar->type_idx, flags);
    return 1;
}


/**
 * A char** argument was filled with a pointer to a newly allocated string.
 * Copy that to the Lua stack, then free the original string.
 */
static int ffi2lua_char_ptr_ptr(struct argconv_t *ar)
{
    char *s = * (char**) ar->arg->p;

    if (s) {
	lua_pushstring(ar->L, s);
	printf("free char** retval %d of function %s?\n",
	    ar->func_arg_nr, ar->ci->fi->name);
	// XXX well... don't know if this is OK?  If a NULL value was passed
	// in, probably yes.  How to find that out?
	g_free(s);
    } else
	lua_pushnil(ar->L);

    return 1;
}


// ------ STRUCT2LUA ----------
// given a structure info and a pointer, extract the value from the structure
// and push it onto the Lua stack.

// for: bool, long, unsigned short int, ...
static int struct2lua_long(lua_State *L, const struct struct_elem *se,
    unsigned char *ptr)
{
    unsigned long v = 0;    // must initialize, because any number of bits
    // might be read and stored.
    get_bits_long(L, ptr, se->bit_offset, se->bit_length, (char*) &v);
    lua_pushnumber(L, v);
    return 1;
}

static int struct2lua_double(lua_State *L, const struct struct_elem *se,
    unsigned char *ptr)
{
    double v = 0;
    get_bits_long(L, ptr, se->bit_offset, se->bit_length, (char*) &v);
    lua_pushnumber(L, v);
    return 1;
}

// this is for a structure pointer
static int struct2lua_struct_ptr(lua_State *L, const struct struct_elem *se,
    unsigned char *ptr)
{
    void *p = 0;
    get_bits_long(L, ptr, se->bit_offset, se->bit_length, (char*) &p);
    luagtk_get_widget(L, p, _guess_type_idx(se->type_idx),
	FLAG_NOT_NEW_OBJECT);
    return 1;
}

// ptr must point directly to the structure.  It makes no sense to autodetect
// the type for derivations, because it can't be a derived (larger) struct.
static int struct2lua_struct(lua_State *L, const struct struct_elem *se,
    unsigned char *ptr)
{
    luagtk_get_widget(L, ptr + se->bit_offset/8, se->type_idx,
	FLAG_NOT_NEW_OBJECT);
    return 1;
}

// strings
static int struct2lua_char_ptr(lua_State *L, const struct struct_elem *se,
    unsigned char *ptr)
{
    gchar **addr = (gchar**) (ptr + se->bit_offset/8);
    if (*addr)
	lua_pushstring(L, *addr);
    else
	lua_pushnil(L);
    return 1;
}

// generic pointer
static int struct2lua_ptr(lua_State *L, const struct struct_elem *se,
    unsigned char *ptr)
{
    void **addr = (void*) (ptr + se->bit_offset/8);
    if (addr) {
	printf("Warning: access to AT_POINTER address %p\n", *addr);
	lua_pushlightuserdata(L, *addr);
    } else
	lua_pushnil(L);
    return 1;
}

// enum or flags
static int struct2lua_enum(lua_State *L, const struct struct_elem *se,
    unsigned char *ptr)
{
    unsigned long v = 0;
    get_bits_long(L, ptr, se->bit_offset, se->bit_length, (char*) &v);
    luagtk_enum_push(L, v, se->type_idx);
    return 1;
}

#define UNTYPED_META "untyped"
struct luagtk_untyped {
    void *p;
};


/**
 * A void* pointer was returned by a Gtk function; cast it to the desired
 * type.
 *
 * @param p  The void pointer
 * @param type  A string with a type name.
 * @return  The requested object.
 */
static int untyped_cast(lua_State *L)
{
    struct luagtk_untyped *u = luaL_checkudata(L, 1, UNTYPED_META);
    const char *name = luaL_checkstring(L, 2);

    // some built in (non widget) types
    if (!strcmp(name, "string")) {
	lua_pushstring(L, (char*) u->p);
	return 1;
    }

    // Look up the widget name - find a pointer to it.
    const struct type_info *ti = find_struct(name, 1);
    if (!ti)
	return luaL_error(L, "%s cast to unknown type %s", msgprefix, name);

    luagtk_get_widget(L, u->p, ti - type_list, 0);
    return 1;
}

static const luaL_reg untyped_methods[] = {
    { "cast", untyped_cast },
    { NULL, NULL }
};

/**
 * A void* pointer will be pushed onto the stack as a userdata with a metatable
 * that contains the function "cast".
 */
static int luagtk_pushuntyped(lua_State *L, void *p)
{
    struct luagtk_untyped *u = (struct luagtk_untyped*)
	lua_newuserdata(L, sizeof(*u));
    u->p = p;

    // add a metatable with some methods
    if (luaL_newmetatable(L, UNTYPED_META)) {
	luaL_register(L, NULL, untyped_methods);
	lua_pushliteral(L, "__index");
	lua_pushvalue(L, -2);
	lua_rawset(L, -3);
    }

    lua_setmetatable(L, -2);
    return 1;
}

// a void* should be converted.
static int struct2lua_void_ptr(lua_State *L, const struct struct_elem *se,
    unsigned char *ptr)
{
    void *p = NULL;

    get_bits_long(L, ptr, se->bit_offset, se->bit_length, (char*) &p);

    // NULL pointer
    if (!p) {
	lua_pushnil(L);
	return 1;
    }

    if (luagtk_is_vwrapper(p))
	return luagtk_vwrapper_get(L, p);

    return luagtk_pushuntyped(L, p);
}

// -------------------------------------------------------

static lua_Number luagtk_tonumber(lua_State *L, int index)
{
    switch (lua_type(L, index)) {
	case LUA_TNUMBER:
	    return lua_tonumber(L, index);
	
	case LUA_TBOOLEAN:
	    return lua_toboolean(L, index) ? 1 : 0;
	
	case LUA_TUSERDATA:;
	    struct luagtk_enum_t *e = LUAGTK_TO_ENUM(L, index);
	    return e->value;
    }

    return luaL_argerror(L, index, "can't convert to number");
}

/**
 * Write a number into a structure element
 */
static int lua2struct_long(lua_State *L, const struct type_info *si,
    const struct struct_elem *se, unsigned char *ptr, int index)
{
    unsigned long int v = luagtk_tonumber(L, index);
    set_bits(ptr, se->bit_offset, se->bit_length, v);
    return 1;
}

static int lua2struct_double(lua_State *L, const struct type_info *si,
    const struct struct_elem *se, unsigned char *ptr, int index)
{
    double v = lua_tonumber(L, index);
    set_bits_long(L, ptr, se->bit_offset, se->bit_length, (char*) &v);
    return 1;
}


static int lua2struct_enum(lua_State *L, const struct type_info *si,
    const struct struct_elem *se, unsigned char *ptr, int index)
{
    struct luagtk_enum_t *e = luagtk_enum_get(L, index, se->type_idx, 1);
    set_bits(ptr, se->bit_offset, se->bit_length, e->value);
    return 1;
}

// set a function pointer, most likely in an Iface structure.  At the given
// index, a lightuserdata (function pointer) must be found.
static int lua2struct_func_ptr(lua_State *L, const struct type_info *si,
    const struct struct_elem *se, unsigned char *ptr, int index)
{
    luaL_checktype(L, index, LUA_TLIGHTUSERDATA);
    set_bits(ptr, se->bit_offset, se->bit_length,
	(unsigned long int) lua_topointer(L, index));
    return 1;
}


/*-
 * In order to keep each ffi_type_map entry (there are about 50 of them)
 * as small as possible (not larger than a cache line), instead of function
 * pointers there are just indices to these small pointer tables.
 */
const lua2ffi_t ffi_type_lua2ffi[] = {
    NULL,
    &lua2ffi_bool,
    &lua2ffi_long,
    &lua2ffi_enum,
    &lua2ffi_longlong,
    &lua2ffi_double,
    &lua2ffi_float,
    &lua2ffi_uchar,
    &lua2ffi_char_ptr,
    &lua2ffi_ptr,
    &lua2ffi_struct_ptr,
    &lua2ffi_vararg,
    &lua2ffi_int_ptr,
    &lua2ffi_int_ptr,	    // LUA2FFI_UNSIGNED_INT_PTR
    &lua2ffi_int_ptr,	    // LUA2FFI_LONG_UNSIGNED_INT_PTR
    &lua2ffi_void_ptr,
    &lua2ffi_func_ptr,
    &lua2ffi_struct_ptr_ptr,
    &lua2ffi_char_ptr_ptr,
    &lua2ffi_enum_ptr,
    &lua2ffi_enum_ptr_ptr,
};

const ffi2lua_t ffi_type_ffi2lua[] = {
    NULL,
    &ffi2lua_void,
    &ffi2lua_long,
    &ffi2lua_bool,
    &ffi2lua_double,
    &ffi2lua_enum,
    &ffi2lua_uchar,
    &ffi2lua_char_ptr,
    &ffi2lua_struct_ptr,
    &ffi2lua_long_ptr,
    &ffi2lua_void_ptr,
    &ffi2lua_int_ptr,
    &ffi2lua_unsigned_int_ptr,
    &ffi2lua_long_unsigned_int_ptr,
    &ffi2lua_struct_ptr_ptr,
    &ffi2lua_char_ptr_ptr,
    &ffi2lua_enum_ptr,
    &ffi2lua_enum_ptr_ptr,
};

const lua2struct_t ffi_type_lua2struct[] = {
    NULL,
    &lua2struct_long,
    &lua2struct_enum,
    &lua2struct_func_ptr,
    &lua2struct_double,
};

const struct2lua_t ffi_type_struct2lua[] = {
    NULL,
    &struct2lua_long,
    &struct2lua_struct_ptr,
    &struct2lua_struct,
    &struct2lua_char_ptr,
    &struct2lua_ptr,
    &struct2lua_enum,
    &struct2lua_double,
    &struct2lua_void_ptr,
};


#include "gtkdata.types.c"


