package tests::TestDlfConverterTestCaseTest;

use strict;

use base qw/Lire::Test::TestCase/;

use Lire::DlfConverter;
use Lire::Test::DlfConverterTestCase;

sub new {
    my $self = shift->SUPER::new( @_ );

    return $self;
}

sub set_up {
    my $self = $_[0];
    $self->SUPER::set_up();

    $self->{'converter'} =
      new Class::Inner( 'parent' => 'Lire::DlfConverter',
                        methods =>
                        {
                         'new' => sub { bless {}, shift },
                         'name' => sub { 'myconverter' },
                         'schemas' => sub { 'test' },
                        } );

    $self->{'test'} =
      new Class::Inner( 'parent' => 'Lire::Test::DlfConverterTestCase',
                        'methods' =>
                        {
                         'new' => sub { bless {}, shift },
                         'create_converter' => sub { $self->{'converter'} }
                        } );
    return;
}

sub tear_down {
    my $self = $_[0];
    $self->SUPER::tear_down();

    return;
}

sub test_create_mock_process {
    my $self = $_[0];

    my $process = $self->{'test'}->create_mock_process();
    $self->assert_isa( 'Lire::Test::MockDlfConverterProcess', $process );
    $self->assert_str_equals( $self->{'converter'}, $process->{'_converter'} );
    $self->assert_num_equals( 0, $process->{'_line_count'} );
    $self->assert_num_equals( 0, $process->{'_dlf_count'} );
    $self->assert_num_equals( 0, $process->{'_error_count'} );
    $self->assert_num_equals( 0, $process->{'_saved_count'} );
    $self->assert_num_equals( 0, $process->{'_ignored_count'} );

    $self->assert_deep_equals( [], $process->{'_errors'} );
    $self->assert_deep_equals( [], $process->{'_saved'} );
    $self->assert_deep_equals( [], $process->{'_ignored'} );
    $self->assert_deep_equals( { 'test' => [] }, $process->{'_dlf'} );
}

sub test_mock_process_error {
    my $self = $_[0];

    my $process = $self->{'test'}->create_mock_process();
    $self->assert_isa( 'Lire::Test::MockDlfConverterProcess', $process );
    $process->error( 'Test error', 'A line' );
    $self->assert_deep_equals( [ [ 'Test error', 'A line' ] ],
                               $process->{'_errors'} );

    $process->error( 'Another error' );
    $self->assert_deep_equals( [ [ 'Test error', 'A line' ],
                                 ['Another error', undef ] ],
                               $process->errors() );
    $self->assert_num_equals( 2, $process->error_count() );
}

sub test_mock_process_save_log_line {
    my $self = $_[0];

    my $process = $self->{'test'}->create_mock_process();
    $self->assert_isa( 'Lire::Test::MockDlfConverterProcess', $process );
    $process->save_log_line(  'A line' );
    $self->assert_deep_equals( [  'A line' ],
                               $process->{'_saved'} );

    $process->save_log_line( 'Another line' );
    $self->assert_deep_equals( [ 'A line', 'Another line' ],
                               $process->saved() );
    $self->assert_num_equals( 2, $process->saved_count() );
}

sub test_mock_process_write_dlf {
    my $self = $_[0];

    my $process = $self->{'test'}->create_mock_process();
    $self->assert_isa( 'Lire::Test::MockDlfConverterProcess', $process );

    $self->assert_dies( qr/schema 'noschema' wasn't defined by 'myconverter' converter /,
                        sub { $process->write_dlf( 'noschema', {} ) } );

    my $dlf = { 'connection_id' => 'An id' };
    $process->write_dlf( 'test', $dlf );
    $self->assert_deep_equals( [ $dlf ], $process->{'_dlf'}{'test'} );
    $self->assert_str_not_equals( $dlf, $process->{'_dlf'}{'test'}[0] );

    $dlf->{'time_start'} = time;
    $process->write_dlf( 'test', $dlf );
    $self->assert_deep_equals( [ { 'connection_id' => 'An id' }, $dlf ],
                                 $process->dlf( 'test' )  );
    $self->assert_num_equals( 2, $process->dlf_count() );
}

sub test_mock_process_ignore_log_line {
    my $self = $_[0];

    my $process = $self->{'test'}->create_mock_process();
    $self->assert_isa( 'Lire::Test::MockDlfConverterProcess', $process );
    $process->ignore_log_line( 'A line', 'A reason' );
    $self->assert_deep_equals( [ [ 'A line', 'A reason' ] ],
                               $process->{'_ignored'} );

    $process->ignore_log_line( 'Another line' );
    $self->assert_deep_equals( [ [ 'A line', 'A reason' ],
                                 ['Another line', undef ] ],
                               $process->ignored() );
    $self->assert_num_equals( 2, $process->ignored_count() );
}

1;
