/* -*- mode: C++; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* linbox/blackbox/jit-matrix.h
 *
 * bds, jpm
 *
 * See COPYING for license information.
 */

#ifndef __JITMATRIX_H
#define __JITMATRIX_H

#include <linbox/blackbox/blackbox-interface.h>

namespace LinBox
{

	/** 

\brief   Example of a blackbox that is space efficient, 
                   though not time efficient.

\ingroup blackbox
	 Just In Time Matrix.
	 
	 The matrix itself is not stored in memory.  Rather, an EntryGenerator 
         function is called to provide the entries.  The entry generator is 
         called once for each entry during an apply or applyTranspose 
         operation.  

	 An toy example of its use is the Hilbert matrix, whose i,j entry is 
         generated by the formula 1/(i+j+2) in zero based indexing.  
		 The motivating examples were matrices also defined by formula, the Paley type matrices.  
		 \cite{MSW07}% ISSAC 07 paper
		 In that context block structured turned out to be essential and the 
         JIT_Matrix class is primarily intended for block structured matrices, 
         the JIT entries being matrix blocks.

@param The _Field only need provide the init() and axpyin() functions.

@param The JIT_EntryGenerator gen() is a function object defining the 
       matrix by  providing gen(e, i, j) which sets field element e to the i,j entry 
       of the matrix. Indexing is zero based.

	 */


	template <class _Field, class JIT_EntryGenerator>
	class JIT_Matrix
	{
	  public:

		typedef _Field Field;
		typedef typename Field::Element Element;
		typedef MatrixCategories::BlackboxTag MatrixCategory;

/**
 * m by n matrix is constructed.  
 * JIT(Field::Element& e, size_t i, size_t j) is a function object which 
 * assigns the i,j entry to e (and returns a reference to e) 
 * and must be valid for 0 <= i < m, 0 <= j < n.
 **/

		JIT_Matrix (_Field& F, const size_t m, const size_t n, 
                            const JIT_EntryGenerator& JIT)
		: _F(F), _m(m), _n(n), _gen(JIT){};

		template<class OutVector, class InVector>
		OutVector& apply (OutVector& y, const InVector& x) ;
		//OutVector& apply (OutVector& y, const InVector& x) const;


		template<class OutVector, class InVector>
		OutVector& applyTranspose (OutVector& y, const InVector& x);
		//OutVector& applyTranspose (OutVector& y, const InVector& x) const;
		size_t rowdim (void) const { return _m; } 
		size_t coldim (void) const { return _n; } 
		const Field& field() const { return _F; }

      protected:

		// Field for arithmetic
		Field _F;

		// Number of rows and columns of matrix.
		size_t _m;
		size_t _n;

		// STL vector of field elements used in applying matrix.
		JIT_EntryGenerator _gen;

	}; // class JIT_Matrix
   

	// Method implementations 
 
	template <class Field, class JIT_EntryGenerator>
	template <class OutVector, class InVector>
	inline OutVector& JIT_Matrix<Field, JIT_EntryGenerator>
		::apply (OutVector& y, const InVector& x) 
	{	Element entry;  _F.init(entry); 
		for (size_t i = 0; i < _m; ++i)
		{   _F.init(y[i], 0);
		    for (size_t j = 0; j < _n; ++j) 
			{ 
			    _gen(entry, i, j); 

			    _F.axpyin (y[i], entry, x[j]); 
			}
		}
		return y;
	} //apply 

 
	template <class Field, class JIT_EntryGenerator>
	template <class OutVector, class InVector>
	inline OutVector& JIT_Matrix<Field, JIT_EntryGenerator>
		::applyTranspose (OutVector& y, const InVector& x) 
	{	Element entry;  _F.init(entry);
		for (size_t i = 0; i < _m; ++i)
		{   _F.init(y[i], 0);
		    for (size_t j = 0; j < _n; ++j) 
			{ _F.axpyin ( y[i], x[j], _gen(entry, j, i) ); }
		}
		return y;
	} // applyTranspose



// Example: Generator to create psuedo-random entries
// !WARNING! repeated calls will give different values for the same entry
        
	template < class Field >
	class JIT_RandomEntryGenerator 
	{
		typename Field::RandIter _r;
                size_t _b;

          public:
		JIT_RandomEntryGenerator(Field& F, size_t b):_r(F), _b(b) {}

		typename Field::Element& operator()(typename Field::Element& e,
                   size_t k,  size_t l) 
		{
			return _r.random(e);
		}
	};
  

} // namespace LinBox

#endif // _JITMATRIX_H
