/***********************************************************************************

	Copyright (C) 2009 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#include <gtkmm/table.h>

#include "dialog_export.hpp"
#include "dialog_password.hpp"


using namespace LIFEO;

// DIALOGSAVEDIARY
DialogSaveDiary::DialogSaveDiary( )
	:	Gtk::FileChooserDialog(
		// TRANSLATORS: this is the title of filechooser dialog for saving a new diary
		_( "Where to Save The Diary?" ), Gtk::FILE_CHOOSER_ACTION_SAVE )
{
	add_button( Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL );
	add_button( Gtk::Stock::OK, Gtk::RESPONSE_ACCEPT );
}


// DIALOG PASSWORD PROMPT ======================================================
DialogPasswordPrompt::DialogPasswordPrompt( BaseObjectType* cobject,
											const Glib::RefPtr< Gtk::Builder > &refbuilder )
	:	DialogEvent( cobject, refbuilder )
{
	Lifeobase::builder->get_widget_derived( "entry_import_password", m_entry );
	set_response_sensitive( RESPONSE_GO, false );

	// SIGNALS
	m_entry->signal_changed().connect(
			sigc::mem_fun( this, &DialogPasswordPrompt::handle_entry_changed ) );
}

void
DialogPasswordPrompt::on_hide( void )
{
	m_entry->set_text( "" );
	m_entry->grab_focus();	// closing with cancel steals focus
	Gtk::Dialog::on_hide();
}

void
DialogPasswordPrompt::handle_entry_changed( void )
{
	set_response_sensitive( RESPONSE_GO, ! m_entry->get_text().empty() );
}

// DIALOG IMPORT ===============================================================
DialogImport::DialogImport( BaseObjectType* cobject,
							const Glib::RefPtr< Gtk::Builder > &refbuilder )
	:	DialogEvent( cobject, refbuilder ), m_dialog_password( NULL), m_diary( NULL )
{
	Lifeobase::builder->get_widget( "filebutton_import_diary", m_file_chooser_button );
	Lifeobase::builder->get_widget( "label_import_info", m_label_info );
	Lifeobase::builder->get_widget( "label_import_entries", m_label_entries );
	Lifeobase::builder->get_widget( "label_import_tags", m_label_tags );
	Lifeobase::builder->get_widget( "label_import_chapters", m_label_chapters );
	Lifeobase::builder->get_widget_derived( "entry_import_add_tag", m_tag_widget );

	Lifeobase::builder->get_widget( "checkbutton_import_entries", m_checkbutton_import_entries );
	Lifeobase::builder->get_widget( "checkbutton_import_tags", m_checkbutton_import_tags );
	Lifeobase::builder->get_widget( "checkbutton_import_chapters", m_checkbutton_import_chapters );

	// SIGNALS
	m_file_chooser_button->signal_file_set().connect(
			sigc::mem_fun( this, &DialogImport::handle_diary_changed ) );

	m_checkbutton_import_entries->signal_toggled().connect(
			sigc::mem_fun( this, &DialogImport::handle_parts_changed ) );
	m_checkbutton_import_tags->signal_toggled().connect(
			sigc::mem_fun( this, &DialogImport::handle_parts_changed ) );
	m_checkbutton_import_chapters->signal_toggled().connect(
			sigc::mem_fun( this, &DialogImport::handle_parts_changed ) );
}

inline bool
DialogImport::is_options_ready( void )
{
	return( m_option_import_entries || m_option_import_tags || m_option_import_chapters );
}

void
DialogImport::on_show( void )
{
	Gtk::Dialog::on_show();
	set_not_ready( _( "Select a diary file for importing" ) );

	m_checkbutton_import_entries->set_active( true );
	m_checkbutton_import_tags->set_active( true );
	m_checkbutton_import_chapters->set_active( false );	// TODO...

	m_tag_widget->populate();

	m_file_chooser_button->grab_focus();
}

void
DialogImport::handle_diary_changed( void )
{
	if( m_diary != NULL )
		delete m_diary;

	m_diary = new Diary( m_file_chooser_button->get_filename() );

	if( m_diary->read_header() == SUCCESS )
	{
		if( m_diary->is_encrypted() )
		{
			if( m_dialog_password == NULL )
				Lifeobase::builder->get_widget_derived(
						"dialog_import_password", m_dialog_password );
			m_dialog_password->set_transient_for( *this );
			if( m_dialog_password->run() == RESPONSE_GO )
				m_diary->set_passphrase( m_dialog_password->m_entry->get_text() );
			m_dialog_password->hide();
		}

		Result result = m_diary->read_body();
		if( result == SUCCESS || result == EMPTY_DATABASE )
		{
			set_ready();
		}
		else
		if( result == WRONG_PASSWORD )
		{
			set_not_ready( _( "Entered password is wrong; please try again" ) );
		}
		else
		{
			set_not_ready( _( "Selected file cannot be read" ) );
		}
	}
	else
	{
		set_not_ready( _( "Selected file does not seem to be a valid diary" ) );
	}
}

void
DialogImport::handle_parts_changed( void )
{
	m_option_import_entries = m_checkbutton_import_entries->get_active();
	m_option_import_tags = m_checkbutton_import_tags->get_active();
	m_option_import_chapters = m_checkbutton_import_chapters->get_active();

	set_response_sensitive( RESPONSE_GO, m_flag_diary_is_ready && is_options_ready() );
}

void
DialogImport::set_ready( void )
{
	set_response_sensitive( RESPONSE_GO, is_options_ready() );

	m_label_entries->set_text( Glib::ustring::compose( "%1", m_diary->get_size() ) );
	m_label_tags->set_text(
			Glib::ustring::compose( "%1", m_diary->get_tags()->size() ) );
	m_label_chapters->set_text(
			Glib::ustring::compose( "%1", m_diary->m_chapter_categories.size() ) );

	m_label_info->set_text( _( "Diary is ready for importing" ) );
	m_flag_diary_is_ready = true;
}

void
DialogImport::set_not_ready( const Glib::ustring &message )
{
	set_response_sensitive( RESPONSE_GO, false );

	m_label_entries->set_text( "-" );
	m_label_tags->set_text( "-" );
	m_label_chapters->set_text( "-" );

	if( m_diary )
	{
		delete m_diary;
		m_diary = NULL;
	}

	m_label_info->set_text( message );
	m_flag_diary_is_ready = false;
}

void
DialogImport::on_response( int response )
{
	if( response == RESPONSE_GO )
	{
		if( m_option_import_tags )
		{
			for( PoolTags::const_iterator iter = m_diary->m_tags.begin();
				 iter != m_diary->m_tags.end();
				 ++iter )
			{
				Diary::d->import_tag( ( *iter ).second );
			}
		}

		if( m_option_import_entries )
		{
			Diary::d->import_entryset( m_diary->m_entries,
									   m_option_import_tags,
									   m_tag_widget->get_text() );
		}
	}

	delete m_diary;
	m_diary = NULL;

	Gtk::Dialog::on_response( response );
}

// DIALOGEXPORT ================================================================
DialogExport::DialogExport( Diary *diary )
	:	DialogEvent( _( "Save a Copy of the Diary" ) ),
	m_checkbutton_overwrite( _( "Overwrite" ) ),
	m_ptr2widget_options( NULL ), m_ptr2diary( diary )
{
	set_resizable( false );

	add_saver( new SaverEncrypted( diary ) );
	add_saver( new SaverUnencrypted( diary ) );
	add_saver( new ExporterText( diary ) );

	m_table.set_spacings( 5 );
	m_table.set_border_width( 5 );

	Gtk::Label *label_file = Gtk::manage( new Gtk::Label( _( "File:" ) ) );
	// TRANSLATORS: type of file export operation will produce
	Gtk::Label *label_type = Gtk::manage( new Gtk::Label( _( "Type:" ) ) );

	m_entry_path.set_icon_from_icon_name( "folder" );
	m_entry_path.set_icon_tooltip_text( _( "Browse..." ) );

	m_table.attach( *label_file, 0, 1, 0, 1, Gtk::SHRINK, Gtk::SHRINK );
	m_table.attach( m_entry_path, 1, 2, 0, 1, Gtk::EXPAND | Gtk::FILL, Gtk::SHRINK );
	m_table.attach( m_checkbutton_overwrite, 1, 2, 1, 2, Gtk::FILL, Gtk::SHRINK );
	m_table.attach( *label_type, 0, 1, 2, 3, Gtk::SHRINK, Gtk::SHRINK );
	m_table.attach( m_combobox, 1, 2, 2, 3, Gtk::EXPAND | Gtk::FILL, Gtk::SHRINK );

	get_vbox()->pack_start( m_table );

	add_button( Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL );
	m_button_save = add_button( Gtk::Stock::SAVE, Gtk::RESPONSE_ACCEPT );

	m_combobox.signal_changed().connect(
			sigc::mem_fun( *this, &DialogExport::handle_savetype_changed ) );

	m_entry_path.signal_icon_release().connect(
			sigc::mem_fun( *this, &DialogExport::handle_browse ) );

	m_entry_path.signal_changed().connect(
			sigc::mem_fun( *this, &DialogExport::handle_path_changed ) );

	m_checkbutton_overwrite.signal_toggled().connect(
			sigc::mem_fun( *this, &DialogExport::handle_overwrite_toggled ) );

	std::string path = diary->get_path();
	path += ".backup";
	m_entry_path.set_text( path );
	m_combobox.set_active( 0 );  // defaults to encrypted diary

	show_all();
}

void
DialogExport::add_saver( Diarysaver *saver )
{
	m_savers.push_back( saver );
	m_combobox.append_text( saver->get_name() );
	saver->signal_updated().connect(
		sigc::mem_fun( *this, &DialogExport::handle_saver_updated ) );
}

void
DialogExport::handle_saver_updated( bool ready )
{
	m_flag_saverready = ready;
	m_button_save->set_sensitive( is_ready( ) );
}

void
DialogExport::handle_savetype_changed( void )
{
	int i = m_combobox.get_active_row_number();
	
	if( m_ptr2widget_options )
	{
		m_table.remove( *m_ptr2widget_options );
		delete m_ptr2widget_options;
	}
	m_ptr2widget_options = m_savers[ i ]->draw_options();
	if( m_ptr2widget_options )
		m_table.attach( *m_ptr2widget_options, 1, 2, 3, 4, Gtk::FILL, Gtk::FILL );
	
	m_flag_saverready = m_savers[ i ]->is_ready();
	
	m_button_save->set_sensitive( is_ready() );
}

void
DialogExport::handle_browse( Gtk::EntryIconPosition, const GdkEventButton* )
{
	DialogSaveDiary *dialog_save = new DialogSaveDiary();

	dialog_save->set_filename( m_entry_path.get_text() );

	if( dialog_save->run() == Gtk::RESPONSE_ACCEPT )
		m_entry_path.set_text( dialog_save->get_filename() );

	delete dialog_save;
}

void
DialogExport::handle_path_changed( void )
{
	m_checkbutton_overwrite.set_active( false );

	if( m_entry_path.get_text().empty() )
	{
		m_flag_fileready = false;
		m_checkbutton_overwrite.set_sensitive( false );
		m_button_save->set_sensitive( false );
	}
	else
	if( Glib::file_test( m_entry_path.get_text(), Glib::FILE_TEST_EXISTS ) )
	{
		m_flag_fileready = false;
		m_checkbutton_overwrite.set_sensitive( true );
		m_button_save->set_sensitive( false );
	}
	else
	{
		m_flag_fileready = true;
		m_checkbutton_overwrite.set_sensitive( false );
		m_button_save->set_sensitive( is_ready() );
	}
}

void
DialogExport::handle_overwrite_toggled( void )
{
	m_flag_fileready = m_checkbutton_overwrite.get_active();
	m_button_save->set_sensitive( is_ready() );
}

inline bool
DialogExport::is_ready( void )
{
//	if( m_flag_fileready == false )
//		return false;
//
//	if( m_flag_saverready == false )
//		return false;

//	return true;
	return( m_flag_fileready && m_flag_saverready );
}

void
DialogExport::on_response( int response )
{
	if( response == Gtk::RESPONSE_ACCEPT )
	{
		m_savers[ m_combobox.get_active_row_number() ]->save(
				m_entry_path.get_text() );
	}
}

// SAVERS & EXPORTERS
Diarysaver::Diarysaver( const Glib::ustring& name, Diary *diary )
	: m_name( name ), m_ptr2diary( diary )
{
}


SaverEncrypted::SaverEncrypted( Diary *diary )
	:	Diarysaver( _( "Encrypted Diary File" ), diary ),
	m_passphrase( "" )
{
}

Gtk::Widget*
SaverEncrypted::draw_options( void )
{
	m_entry_passphrase = Gtk::manage( new Gtk::Entry );
	m_entry_confirm = Gtk::manage( new Gtk::Entry );
	Gtk::Table *table = new Gtk::Table;

	m_entry_passphrase->set_visibility( false );
	m_entry_confirm->set_visibility( false );

	table->set_spacings( 5 );
	table->set_border_width( 5 );
	table->attach(	* Gtk::manage( new Gtk::Label( _( "Password:" ) ) ),
					0, 1, 0, 1, Gtk::SHRINK );
	table->attach( *m_entry_passphrase,
					1, 2, 0, 1, Gtk::EXPAND | Gtk::FILL, Gtk::SHRINK );
	// TRANSLATORS: confirm here means re-enter password
	table->attach(	* Gtk::manage( new Gtk::Label( _( "Confirm:" ) ) ),
					0, 1, 1, 2, Gtk::SHRINK );
	table->attach( *m_entry_confirm,
					1, 2, 1, 2, Gtk::EXPAND | Gtk::FILL, Gtk::SHRINK );

	// TODO: re-use password entry table along with its match indicator icon...
	// ...in DialogPassword

	m_entry_passphrase->signal_changed().connect(
			sigc::mem_fun( *this, &SaverEncrypted::handle_passphrase_changed ) );

	m_entry_confirm->signal_changed().connect(
			sigc::mem_fun( *this, &SaverEncrypted::handle_passphrase_changed ) );

	table->show_all();

	return table;
}

bool
SaverEncrypted::is_ready( void )
{
	return( m_passphrase.size() >= LIFEO::PASSPHRASE_MIN_SIZE );
}

void
SaverEncrypted::handle_passphrase_changed( void )
{
	m_passphrase = m_entry_passphrase->get_text();
	if( m_passphrase != m_entry_confirm->get_text() )
		m_passphrase.clear();
	else
	if( m_passphrase.size() < LIFEO::PASSPHRASE_MIN_SIZE )
		m_passphrase.clear();
	
	m_signal_updated.emit( m_passphrase.size() >= LIFEO::PASSPHRASE_MIN_SIZE );
}

Result
SaverEncrypted::save( const std::string &path )
{
	if( m_passphrase.size() < LIFEO::PASSPHRASE_MIN_SIZE )
		return FAILURE;
	else
		return m_ptr2diary->write_copy( path, m_passphrase );
}


SaverUnencrypted::SaverUnencrypted( Diary *diary )
	: Diarysaver( _( "Unencrypted Diary File" ), diary )
{
}

Result
SaverUnencrypted::save( const std::string &path )
{
	return m_ptr2diary->write_copy( path, "" );
}


ExporterText::ExporterText( Diary *diary )
	: Diarysaver( _( "Plain Text File" ), diary )
{

}

Result
ExporterText::save( const std::string &path )
{
	return m_ptr2diary->write_txt( path );
}
