/***********************************************************************************

	Copyright (C) 2007-2010 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iomanip>

#include "lifeobase.hpp"
#include "diary.hpp"
#include "views.hpp"
#include "gui_tag.hpp"


using namespace LIFEO;


// TAG =========================================================================
TagView::TagView( void )
{
	try
	{
		Lifeobase::builder->get_widget( "vbox_tag", m_vbox );
		Lifeobase::builder->get_widget( "alignment_tag_toolbar", m_alignment_toolbar );
		Lifeobase::builder->get_widget( "togglebutton_tag_filter", m_button_filter );
		Lifeobase::builder->get_widget( "button_tag_dismiss", m_button_dismiss );
		Lifeobase::builder->get_widget( "entry_tag_name", m_entry_name );
		Lifeobase::builder->get_widget( "button_tag_name_apply", m_button_name_apply );
		//Lifeobase::builder->get_widget( "label_tag_period", m_label_period );
		Lifeobase::builder->get_widget( "combobox_tag_category", m_combobox_category );
		Lifeobase::builder->get_widget_derived( "drawingarea_tag_chart", m_widgetchart );

		m_liststore_categories = Gtk::ListStore::create( colrec_category );
	}
	catch( ... ) {}

	Tag::shower = this;

	m_combobox_category->set_model( m_liststore_categories );
	m_combobox_category->pack_start( colrec_category.name );
	m_combobox_category->set_row_separator_func(
			sigc::mem_fun( this, &TagView::is_row_separator ) );

	// SIGNALS
	m_button_filter->signal_toggled().connect(
			sigc::mem_fun( this, &TagView::handle_filter_toggled ) );
	m_button_dismiss->signal_clicked().connect(
			sigc::mem_fun( this, &TagView::dismiss_tag ) );

	m_button_name_apply->signal_clicked().connect(
			sigc::mem_fun( this, &TagView::handle_name_applied ) );
	m_entry_name->signal_changed().connect(
			sigc::mem_fun( this, &TagView::handle_name_changed ) );
	m_entry_name->signal_activate().connect(
			sigc::mem_fun( *m_button_name_apply, &Gtk::Button::clicked ) );

	m_combobox_category->signal_changed().connect(
			sigc::mem_fun( this, &TagView::handle_category_changed ) );
}

void
TagView::handle_filter_toggled( void )
{
	if( Lifeobase::m_internaloperation )
		return;

	Diary::d->set_filter_tag( m_button_filter->get_active() ? m_ptr2tag : NULL );
	Lifeobase::base->update_filter_tag();
	Lifeobase::base->update_tag_icon( m_ptr2tag );
}

void
TagView::handle_name_changed( void )
{
	Glib::ustring name = m_entry_name->get_text();
	Tag *tag = Diary::d->get_tags()->get_tag( name );
	m_button_name_apply->set_sensitive( !( tag || name.empty() ) );
}

void
TagView::handle_name_applied( void )
{
	// a caution against calls by an entry activation event:
	if( ! m_button_name_apply->is_sensitive() )
		return;
	Glib::ustring name = m_entry_name->get_text();
	Diary::d->get_tags()->rename( m_ptr2tag, name );
	Lifeobase::base->m_label_gui_elem->set_label( name );

	m_button_name_apply->set_sensitive( false );

	Gtk::TreeRow row = * Lifeobase::base->get_element_row( m_ptr2tag );
	row[ TagPanel::colrec->name ] = Glib::Markup::escape_text( name );
}

void
TagView::handle_category_changed( void )
{
	if( Lifeobase::m_internaloperation )
		return;

	Gtk::TreeIter iter = m_combobox_category->get_active();
	if( iter )
	{
		Gtk::TreeRow row = *iter;
		if( row[ colrec_category.type ]  == TCT_NEW ) // create new
		{
			CategoryTags *category = Diary::d->create_tag_ctg();
			m_ptr2tag->set_category( category );
			update_combobox_categories();
			Lifeobase::base->update_tag_list();
			category->show();
		}
		else	// assign a category or reset
		{
			m_ptr2tag->set_category( row[ colrec_category.ptr ] );
			Lifeobase::base->update_tag_list();
		}
	}
}

void
TagView::update_combobox_categories( void )
{
	m_liststore_categories->clear();

	Gtk::TreeRow row = * m_liststore_categories->append();

	row[ colrec_category.name ] = _( "None" );
	row[ colrec_category.ptr ] = NULL;
	row[ colrec_category.type ] = TCT_NONE;

	// separator 1:
	row = * m_liststore_categories->append();
	row[ colrec_category.type ] = TCT_SEPARATOR;
	row[ colrec_category.ptr ] = ( CategoryTags* ) 0x1;

	for( PoolCategoriesTags::const_iterator iter = Diary::d->m_tag_categories.begin();
		 iter != Diary::d->m_tag_categories.end();
		 ++iter )
	{
		row = * m_liststore_categories->append();

		row[ colrec_category.name ] = iter->first;
		row[ colrec_category.ptr ] = iter->second;
		row[ colrec_category.type ] = TCT_CATEGORY;
	}

	// separator 2:
	if( Diary::d->m_tag_categories.size() > 0 )
	{
		row = * m_liststore_categories->append();
		row[ colrec_category.type ] = TCT_SEPARATOR;
		row[ colrec_category.ptr ] = ( CategoryTags* ) 0x1;
	}

	row = * m_liststore_categories->append();

	row[ colrec_category.name ] = _( "Create New..." );
	// a dummy value to differentiate it from "None":
	row[ colrec_category.ptr ] = ( CategoryTags* ) 0x1;
	row[ colrec_category.type ] = TCT_NEW;
}

void
TagView::show( Tag &tag )
{
	Lifeobase::m_internaloperation++;

	Lifeobase::base->change_current_element( &tag, m_vbox );

	m_ptr2tag = &tag;

	// HEADER
	Lifeobase::m_image_gui_elem->set( ICONDIR "/tag-32.png" );
	int timespan = 0;
	//Glib::ustring begins, ends;
	if( ! tag.m_items.empty() )
	{
		Glib::Date gdate_begin = ( * tag.m_items.rbegin() )->get_date().get_glib();
		Glib::Date gdate_end = ( * tag.m_items.begin() )->get_date().get_glib();
		timespan = gdate_begin.days_between( gdate_end );
		//begins = gdate_begin.format_string( "%Y.%m.%d" );
		//ends = gdate_end.format_string( "%Y.%m.%d" );
	}
	Lifeobase::base->set_header_texts(
			tag.m_name,
			Glib::ustring::compose(
					_( "%1 entries spanning %2 days" ),
					tag.m_items.size(),
					timespan ) );
	Lifeobase::set_gelem_toolbar( m_alignment_toolbar );

	// BODY
	m_button_filter->set_active( Diary::d->get_filter_tag() == &tag );

	m_entry_name->set_text( tag.m_name );

	for( Gtk::TreeIter iter = m_liststore_categories->children().begin();
		 iter != m_liststore_categories->children().end();
		 ++iter )
	{
		if( ( *iter )[ colrec_category.ptr ] == tag.get_category() )
			m_combobox_category->set_active( iter );
	}

	// CHART DATA
	ChartPoints *points( new ChartPoints );
	Date yearmonth_last( 0 );
	int value( 0 );
	for( SetDiaryElements::reverse_iterator iter = tag.m_items.rbegin();
		 iter != tag.m_items.rend(); ++iter )
	{
		Entry *entry = dynamic_cast< Entry* >( *iter );
		const Date ym_entry( entry->get_date().get_yearmonth() );
		if( yearmonth_last == 0 )
			yearmonth_last = ym_entry;
		else
		if( yearmonth_last != ym_entry )
		{
			points->add( yearmonth_last, value );

			// fill empty months
			for( Date ym_zero = yearmonth_last; ; )
			{
				ym_zero.forward_month();
				if( ym_zero >= ym_entry )
					break;
				points->add( ym_zero, 0 );
			}

			yearmonth_last = ym_entry;
			value = 0;
		}
		value++;
	}
	points->add( yearmonth_last, value );

	m_widgetchart->set_points( points );

	Lifeobase::m_internaloperation--;
}

void
TagView::dismiss_tag( void )
{
	Gtk::MessageDialog message( *Lifeobase::base, "", false,
								Gtk::MESSAGE_WARNING, Gtk::BUTTONS_NONE, true );
	message.set_message( _( "Are you sure, you want to dismiss?" ) );
	message.set_secondary_text( _( "This operation cannot be undone!" ) );
	message.add_button( Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL );
	message.add_button( _( "_DISMISS!" ), Gtk::RESPONSE_ACCEPT );

	if( message.run() != Gtk::RESPONSE_ACCEPT )
		return;

	// clear filter if necessary:
	if( Diary::d->get_filter_tag() == m_ptr2tag )
	{
		Diary::d->set_filter_tag( NULL );
		Lifeobase::base->update_filter_tag();
	}
	Diary::d->dismiss_tag( m_ptr2tag );
	Lifeobase::base->update_tag_list();

	Diary::d->show();
}

bool
TagView::is_row_separator( const Glib::RefPtr< Gtk::TreeModel > &model,
							 const Gtk::TreeIter iter )
{
	Gtk::TreeRow row = * iter;
	return ( row[ colrec_category.type ] == TCT_SEPARATOR );
}

// TAG CATEGORY ================================================================
CategoryTagsView::CategoryTagsView( void )
{
	CategoryTags::shower = this;

	try
	{
		Lifeobase::builder->get_widget( "vbox_categorytags", m_vbox );
		Lifeobase::builder->get_widget( "alignment_categorytags_toolbar", m_alignment_toolbar );
		Lifeobase::builder->get_widget( "entry_categorytags_name", m_entry_name );
		Lifeobase::builder->get_widget( "button_categorytags_name_apply", m_button_name_apply );
		Lifeobase::builder->get_widget( "button_categorytags_dismiss", m_button_dismiss );
	}
	catch( ... ) {}

	m_entry_name->signal_changed().connect(
			sigc::mem_fun( this, &CategoryTagsView::handle_name_changed ) );

	m_entry_name->signal_activate().connect(
			sigc::mem_fun( *m_button_name_apply, &Gtk::Button::clicked ) );

	m_button_name_apply->signal_clicked().connect(
			sigc::mem_fun( this, &CategoryTagsView::handle_name_applied ) );

	m_button_dismiss->signal_clicked().connect(
			sigc::mem_fun( this, &CategoryTagsView::dismiss_category ) );
}

void
CategoryTagsView::show( CategoryTags &category )
{
	Lifeobase::base->change_current_element( &category, m_vbox );

	m_ptr2category = &category;

	// HEADER
	Lifeobase::m_image_gui_elem->set( ICONDIR "/category_tags-32.png" );
	Lifeobase::base->set_header_texts(
			category.m_name,
			Glib::ustring::compose( _( "%1 tags" ), category.size() ) );
	Lifeobase::set_gelem_toolbar( m_alignment_toolbar );

	// BODY
	m_entry_name->set_text( category.m_name );

	//Lifeobase::base->select_date_in_calendar( chapter.m_date_begin );
}

void
CategoryTagsView::handle_name_changed( void )
{
	Glib::ustring name = m_entry_name->get_text();
	PoolCategoriesTags::const_iterator iter = Diary::d->m_tag_categories.find( name );
	m_button_name_apply->set_sensitive(
			!( iter != Diary::d->m_tag_categories.end() || name.empty() ) );
}

void
CategoryTagsView::handle_name_applied( void )
{
	Glib::ustring name = m_entry_name->get_text();
	Diary::d->m_tag_categories.rename_category( m_ptr2category, name );
	Lifeobase::base->m_label_gui_elem->set_label( name );
	Lifeobase::base->update_tag_list();
	Lifeobase::base->update_tag_categories();	// updates the combobox in TagView

	m_button_name_apply->set_sensitive( false );
}

void
CategoryTagsView::dismiss_category( void )
{
	// TODO: this code is repeated in every view, so it should be unified
	Gtk::MessageDialog message( *Lifeobase::base, "", false,
								Gtk::MESSAGE_WARNING, Gtk::BUTTONS_NONE, true );
	message.set_message( _( "Are you sure, you want to dismiss?" ) );
	message.set_secondary_text( _( "This operation cannot be undone!" ) );
	message.add_button( Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL );
	message.add_button( _( "_DISMISS!" ), Gtk::RESPONSE_ACCEPT );

	if( message.run() != Gtk::RESPONSE_ACCEPT )
		return;

	// TODO: offer merging with a category
	Diary::d->show();
	Diary::d->dismiss_tag_ctg( m_ptr2category );
	Lifeobase::base->update_tag_list();
	Lifeobase::base->update_tag_categories();
}

// CHAPTER =====================================================================
ChapterView::ChapterView( void )
{
	Chapter::shower = this;

	try
	{
		Lifeobase::builder->get_widget( "vbox_chapter", m_vbox );
		Lifeobase::builder->get_widget( "entry_chapter_name", m_entry_name );
		Lifeobase::builder->get_widget( "button_chapter_name_apply", m_button_name_apply );
		Lifeobase::builder->get_widget( "entry_chapter_begins", m_entry_begins );
		Lifeobase::builder->get_widget( "label_chapter_ends", m_label_ends );
	}
	catch( ... ) {}

	// SIGNALS
	m_entry_name->signal_changed().connect(
			sigc::mem_fun( *this, &ChapterView::handle_name_changed ) );
	m_entry_name->signal_activate().connect(
			sigc::mem_fun( *m_button_name_apply, &Gtk::Button::clicked ) );
	m_button_name_apply->signal_clicked().connect(
			sigc::mem_fun( *this, &ChapterView::handle_name_applied ) );
}

void
ChapterView::handle_name_changed( void )
{
	Glib::ustring name = m_entry_name->get_text();
	m_button_name_apply->set_sensitive( ! name.empty() &&
										name != m_ptr2chapter->m_name );
}

void
ChapterView::handle_name_applied( void )
{
	// a caution against calls by an entry activation event:
	if( ! m_button_name_apply->is_sensitive() )
		return;
	Glib::ustring name = m_entry_name->get_text();
	m_ptr2chapter->m_name = name;
	m_button_name_apply->set_sensitive( false );
	Lifeobase::base->m_label_gui_elem->set_label( name );
	Gtk::TreeRow row = * Lifeobase::base->get_element_row( m_ptr2chapter );
	row[ ListData::colrec->info ] = m_ptr2chapter->get_2nd_str();
}

void
ChapterView::show( Chapter &chapter )
{
	Lifeobase::base->change_current_element( &chapter, m_vbox );

	m_ptr2chapter = &chapter;

	// HEADER
	int timespan = 0;
	Glib::ustring begins, ends;
	if( ! chapter.m_items.empty() )
	{
		Glib::Date gdate_begin = ( * chapter.m_items.rbegin() )->get_date().get_glib();
		Glib::Date gdate_end = ( * chapter.m_items.begin() )->get_date().get_glib();
		timespan = gdate_begin.days_between( gdate_end );
		begins = chapter.m_date_begin.format_string();
		// TODO: replace this one with the begin date of the next chapter if any:
		ends = gdate_end.format_string( "%Y.%m.%d" );
	}

	Lifeobase::m_image_gui_elem->set( ICONDIR "/chapter-32.png" );
	Lifeobase::base->set_header_texts(
			chapter.m_name,
			Glib::ustring::compose( _( "%1 entries spanning %2 days" ),
					chapter.m_items.size(), timespan ) );
	Lifeobase::set_gelem_toolbar( NULL );

	Lifeobase::base->select_date_in_calendar( chapter.m_date_begin );

	// BODY
	m_entry_name->set_text( chapter.m_name );
	m_entry_begins->set_text( begins );
	m_label_ends->set_text( ends );
}

// THEME =======================================================================
ThemeView::ThemeView( void )
{
	Theme::shower = this;

	try
	{
		Lifeobase::builder->get_widget( "vbox_theme", m_vbox );
		Lifeobase::builder->get_widget( "entry_theme_name", m_entry_name );
		Lifeobase::builder->get_widget( "button_theme_name_apply", m_button_name_apply );
		Lifeobase::builder->get_widget( "alignment_theme_toolbar", m_alignment_toolbar );
		Lifeobase::builder->get_widget( "fontbutton_theme", m_fontbutton );
		Lifeobase::builder->get_widget( "colorbutton_theme_bground", m_colorbutton_bground );
		Lifeobase::builder->get_widget( "colorbutton_theme_text", m_colorbutton_text );
		Lifeobase::builder->get_widget( "colorbutton_theme_heading", m_colorbutton_heading );
		Lifeobase::builder->get_widget( "colorbutton_theme_subheading", m_colorbutton_subheading );
		Lifeobase::builder->get_widget( "colorbutton_theme_highlight", m_colorbutton_highlight );
		Lifeobase::builder->get_widget( "button_theme_duplicate", m_button_duplicate );
		Lifeobase::builder->get_widget( "button_theme_dismiss", m_button_dismiss );
		Lifeobase::builder->get_widget( "button_theme_default", m_button_make_default );
	}
	catch( ... ) { }

	m_entry_name->signal_changed().connect(
			sigc::mem_fun( this, &ThemeView::handle_name_changed ) );

	m_entry_name->signal_activate().connect(
			sigc::mem_fun( *m_button_name_apply, &Gtk::Button::clicked ) );

	m_button_name_apply->signal_clicked().connect(
			sigc::mem_fun( this, &ThemeView::handle_name_applied ) );

	m_fontbutton->signal_font_set().connect(
			sigc::mem_fun( *this, &ThemeView::handle_font_changed ) );
	m_colorbutton_bground->signal_color_set().connect(
			sigc::mem_fun( *this, &ThemeView::handle_color_changed ) );
	m_colorbutton_text->signal_color_set().connect(
			sigc::mem_fun( *this, &ThemeView::handle_color_changed ) );
	m_colorbutton_heading->signal_color_set().connect(
			sigc::mem_fun( *this, &ThemeView::handle_color_changed ) );
	m_colorbutton_subheading->signal_color_set().connect(
			sigc::mem_fun( *this, &ThemeView::handle_color_changed ) );
	m_colorbutton_highlight->signal_color_set().connect(
			sigc::mem_fun( *this, &ThemeView::handle_color_changed ) );

	// TOOLBAR
	m_button_duplicate->signal_clicked().connect(
			sigc::mem_fun( this, &ThemeView::duplicate ) );
	m_button_dismiss->signal_clicked().connect(
			sigc::mem_fun( this, &ThemeView::dismiss_theme ) );
	m_button_make_default->signal_clicked().connect(
			sigc::mem_fun( this, &ThemeView::make_default ) );
}

void
ThemeView::show( Theme &theme )
{
	Lifeobase::base->change_current_element( &theme, m_vbox );

	m_ptr2theme = &theme;

	// HEADER
	Lifeobase::m_image_gui_elem->set( Lifeobase::icons->theme_32 );
	Lifeobase::base->set_header_texts(
			theme.get_name(),
			_( "Custom theme" ) );
	Lifeobase::set_gelem_toolbar( m_alignment_toolbar );

	// BODY
	m_entry_name->set_text( theme.get_name() );
	m_fontbutton->set_font_name( theme.font.to_string() );
	m_colorbutton_bground->set_color( theme.color_base );
	m_colorbutton_text->set_color( theme.color_text );
	m_colorbutton_heading->set_color( theme.color_heading );
	m_colorbutton_subheading->set_color( theme.color_subheading );
	m_colorbutton_highlight->set_color( theme.color_highlight );

	m_vbox->set_sensitive( &theme != &Theme::system_theme );
	m_button_dismiss->set_sensitive( &theme != &Theme::system_theme );
	m_button_make_default->set_visible( &theme != Diary::d->get_default_theme() );
}

void
ThemeView::handle_name_changed( void )
{
	Glib::ustring name = m_entry_name->get_text();
	PoolThemes::const_iterator iter = Diary::d->get_themes().find( name );
	m_button_name_apply->set_sensitive( !( iter != Diary::d->get_themes().end() || name.empty() ) );
}

void
ThemeView::handle_name_applied( void )
{
	Glib::ustring name = m_entry_name->get_text();
	Diary::d->m_themes.rename_theme( m_ptr2theme, name );
	Lifeobase::base->m_label_gui_elem->set_label( name );
	Lifeobase::base->update_theme_list();

	m_button_name_apply->set_sensitive( false );
}

void
ThemeView::handle_font_changed( void )
{
	m_ptr2theme->font = Pango::FontDescription( m_fontbutton->get_font_name() );
}

void
ThemeView::handle_color_changed( void )
{
	m_ptr2theme->color_base = m_colorbutton_bground->get_color();
	m_ptr2theme->color_text = m_colorbutton_text->get_color();
	m_ptr2theme->color_heading = m_colorbutton_heading->get_color();
	m_ptr2theme->color_subheading = m_colorbutton_subheading->get_color();
	m_ptr2theme->color_highlight = m_colorbutton_highlight->get_color();
}

// creating new themes is done by duplicating existing ones
void
ThemeView::duplicate( void )
{
	Theme *new_theme = Diary::d->duplicate_theme( *m_ptr2theme );
	Lifeobase::base->update_theme_list();
	new_theme->show();
}

void
ThemeView::make_default( void )
{
	Diary::d->set_default_theme( m_ptr2theme );
	Lifeobase::base->update_theme_list();
	m_button_make_default->set_visible( false );
}

void
ThemeView::dismiss_theme( void )
{
	// TODO: this code is repeated in every view, so it should be unified
	Gtk::MessageDialog message( *Lifeobase::base, "", false,
								Gtk::MESSAGE_WARNING, Gtk::BUTTONS_NONE, true );
	message.set_message( _( "Are you sure, you want to dismiss?" ) );
	message.set_secondary_text( _( "This operation cannot be undone!" ) );
	message.add_button( Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL );
	message.add_button( _( "_DISMISS!" ), Gtk::RESPONSE_ACCEPT );

	if( message.run() != Gtk::RESPONSE_ACCEPT )
		return;

	Theme *theme = m_ptr2theme; // always be on the safe side
	Theme::system_theme.show();
	Diary::d->dismiss_theme( theme );
	Lifeobase::base->update_theme_list();
}

// DIARY =======================================================================
DiaryView::DiaryView( void )
	:	m_vbox_diary( NULL ), m_dialog_password( NULL ), m_dialog_import( NULL )
{
	Diary::shower = this;
	m_ptr2diary = Diary::d;	// for now no other diary is shown in the view

	Gtk::Button *button_import( NULL );
	Gtk::Button *button_export( NULL );
	Gtk::Button *button_chapters( NULL );
	try
	{
		Lifeobase::builder->get_widget( "vbox_diary", m_vbox_diary );
		Lifeobase::builder->get_widget( "label_path", m_label_path );
		Lifeobase::builder->get_widget( "label_encryption", m_label_encryption );
		Lifeobase::builder->get_widget( "button_log_out_wosaving",
				m_button_logoutwosaving );
		Lifeobase::builder->get_widget( "button_import", button_import );
		Lifeobase::builder->get_widget( "button_export", button_export );
		Lifeobase::builder->get_widget( "button_chapters", button_chapters );
		Lifeobase::builder->get_widget( "button_password", m_button_password );

		Lifeobase::builder->get_widget( "combobox_chapter_ctg",
				m_combobox_chapter_ctg );
		Lifeobase::builder->get_widget( "button_chapter_ctg_dismiss",
				m_button_chapter_ctg_dismiss );
		Lifeobase::builder->get_widget( "button_chapter_ctg_rename",
				m_button_chapter_ctg_rename );

		m_liststore_chapter_ctg = Gtk::ListStore::create( colrec_chapter_ctg );
	}
	catch( ... ) { }

	// CHAPTER CATEGORIES
	m_combobox_chapter_ctg->set_model( m_liststore_chapter_ctg );
	m_combobox_chapter_ctg->set_text_column( colrec_chapter_ctg.name );
	m_combobox_chapter_ctg->set_row_separator_func(
			sigc::mem_fun( this, &DiaryView::is_row_separator ) );

	// SIGNALS
	m_button_password->signal_clicked().connect(
			sigc::mem_fun( *this, &DiaryView::start_dialog_password ) );
	button_import->signal_clicked().connect(
			sigc::mem_fun( *this, &DiaryView::start_dialog_import ) );
	button_export->signal_clicked().connect(
			sigc::mem_fun( *this, &DiaryView::start_dialog_export ) );
	button_chapters->signal_clicked().connect(
			sigc::mem_fun( *this, &DiaryView::start_dialog_chaptereditor ) );
	m_button_logoutwosaving->signal_clicked().connect(
			sigc::bind( sigc::mem_fun( *Lifeobase::base, &Lifeobase::logout ), false ) );

	m_combobox_chapter_ctg->signal_changed().connect(
			sigc::mem_fun( this, &DiaryView::handle_cur_chapter_ctg_changed ) );
	m_combobox_chapter_ctg->get_entry()->signal_activate().connect(
			sigc::mem_fun( this, &DiaryView::rename_cur_chapter_ctg ) );

	m_button_chapter_ctg_rename->signal_clicked().connect(
			sigc::mem_fun( this, &DiaryView::rename_cur_chapter_ctg ) );

	m_button_chapter_ctg_dismiss->signal_clicked().connect(
			sigc::mem_fun( this, &DiaryView::dismiss_cur_chapter_ctg ) );
}

DiaryView::~DiaryView( void )
{
	Diary::shower = NULL;
}

/*Gtk::Frame*
add_group( const Glib::ustring& str_label, Gtk::Widget& content )
{
	Gtk::Frame *frame = Gtk::manage( new Gtk::Frame );
	Gtk::Label *label = Gtk::manage( new Gtk::Label );

	Gtk::Alignment *alignment = Gtk::manage(
			new Gtk::Alignment( .5, .5, .5, .5 ) );

	label->set_markup( str_label );
	frame->set_shadow_type( Gtk::SHADOW_NONE );
	frame->set_label_widget( *label );
	alignment->add( content );
	frame->add( *alignment );

	return frame;
}*/

void
DiaryView::show( Diary &diary )
{
	Lifeobase::base->change_current_element( &diary, m_vbox_diary );

	// not necessary for now as there is only one diary to show in the view:
	// m_ptr2diary = &diary;

	// HEADER
	int timespan = 0;
	if( ! diary.m_entries.empty() )
	{
		Glib::Date gdate_begin = diary.m_entries.rbegin()->second->get_date().get_glib();
		Glib::Date gdate_end = diary.m_entries.begin()->second->get_date().get_glib();
		timespan = gdate_begin.days_between( gdate_end );
	}

	Lifeobase::m_image_gui_elem->set( ICONDIR "/diary-32.png" );
	Lifeobase::base->set_header_texts( diary.get_name(),
									   Glib::ustring::compose(
												_( "%1 entries spanning %2 days" ),
												diary.m_entries.size(),
												timespan ) );
	Lifeobase::set_gelem_toolbar( NULL );

	// BODY
	m_button_password->set_label( diary.m_passphrase.empty() ? _( "Encrypt..." ) :
												_( "Change Password..." ) );
	m_label_encryption->set_label( diary.m_passphrase.empty() ? _( "Unencrypted" ) :
												_( "Encrypted" ) );

#if GTK_CHECK_VERSION(2,18,0)
	m_label_path->set_markup( Glib::ustring::compose( "<a href=\"file://%1\">%2</a>",
			Glib::path_get_dirname( diary.m_path ), diary.m_path ) );
#else
	m_label_path->set_label( m_path );
#endif
}

void
DiaryView::start_dialog_password( void )
{
	try
	{
		m_dialog_password = new DialogPassword(	m_ptr2diary->m_passphrase,
				m_ptr2diary->m_passphrase.size() > 0 ?
		// TRANSLATORS: this is the title of the password dialog
						_( "Change Password" ) :
		// TRANSLATORS: this is the title of the password dialog
						_( "Add Password" ) );
	}
	catch( ... )
	{
		// TODO: throw
		return;
	}

	m_dialog_password->run();

	if( m_dialog_password->is_passphrase_set() )
	// TODO: if( m_dialog_password->run() )
	{
		if( m_ptr2diary->set_passphrase( m_dialog_password->get_passphrase() ) )
		{
			m_label_encryption->set_label( _( "Encrypted" ) );
			m_button_password->set_label( _( "Change Password..." ) );
		}
	}

	delete m_dialog_password;
	m_dialog_password = NULL;
}

#include "dialog_chaptereditor.hpp"

void
DiaryView::start_dialog_chaptereditor( void )
{
	DialogChaptereditor *dialog_chaptereditor = new DialogChaptereditor( m_ptr2diary );

	dialog_chaptereditor->run();

	if( m_ptr2diary->m_option_sorting_criteria == SC_DATE )
	{
		Lifeobase::base->update_entry_list();
	}

	delete dialog_chaptereditor;
}

void
DiaryView::start_dialog_import( void )
{
	if( m_dialog_import == NULL )
	{
		Lifeobase::builder->get_widget_derived( "dialog_import", m_dialog_import );
	}

	int response = m_dialog_import->run();
	m_dialog_import->hide();

	if( response == RESPONSE_GO )
	{
		Lifeobase::base->update_entry_list();
		Lifeobase::base->update_tag_list();
		//Lifeobase::base->update_theme_list();
	}
}

void
DiaryView::start_dialog_export( void )
{
	DialogExport *dialog_export = new DialogExport( m_ptr2diary );
	dialog_export->run();

	delete dialog_export;
}

//void
//DiaryView::open_diary_folder( void )
//{
//	std::string uri = "file://" + Glib::path_get_dirname( m_path );
//	GError *err = NULL;
//	gtk_show_uri( NULL, uri.c_str(), GDK_CURRENT_TIME, &err);
//}

void
DiaryView::update_combobox_chapter_ctg( void )
{
	m_liststore_chapter_ctg->clear();

	Gtk::TreeRow row = * m_liststore_chapter_ctg->append();

	row[ colrec_chapter_ctg.name ] = _( "None" );
	row[ colrec_chapter_ctg.ptr ] = NULL;
	row[ colrec_chapter_ctg.type ] = CCCT_NONE;

	// setting current item:
	if( m_ptr2diary->m_ptr2chapter_ctg_cur == NULL )
	{
		m_combobox_chapter_ctg->set_active( row );
		m_button_chapter_ctg_rename->set_visible( false );
		m_button_chapter_ctg_dismiss->set_visible( true );
		m_button_chapter_ctg_dismiss->set_sensitive( false );
		m_combobox_chapter_ctg->get_child()->set_sensitive( false );
	}

	// separator 1:
	row = * m_liststore_chapter_ctg->append();
	row[ colrec_chapter_ctg.type ] = CCCT_SEPARATOR;

	for( PoolCategoriesChapters::const_iterator iter =
				m_ptr2diary->m_chapter_categories.begin();
		 iter != m_ptr2diary->m_chapter_categories.end();
		 ++iter )
	{
		row = * m_liststore_chapter_ctg->append();

		CategoryChapters *category = iter->second;

		row[ colrec_chapter_ctg.name ] = category->get_name();
		row[ colrec_chapter_ctg.ptr ] = category;
		row[ colrec_chapter_ctg.type ] = CCCT_CATEGORY;

		// setting current item:
		if( m_ptr2diary->m_ptr2chapter_ctg_cur == category )
		{
			m_combobox_chapter_ctg->set_active( row );
			m_button_chapter_ctg_rename->set_visible( false );
			m_button_chapter_ctg_dismiss->set_visible( true );
			m_button_chapter_ctg_dismiss->set_sensitive( true );
			m_combobox_chapter_ctg->get_child()->set_sensitive( true );
		}
	}

	// separator 2:
	if( m_ptr2diary->m_chapter_categories.size() > 0 )
	{
		row = * m_liststore_chapter_ctg->append();
		row[ colrec_chapter_ctg.type ] = CCCT_SEPARATOR;
	}

	row = * m_liststore_chapter_ctg->append();

	row[ colrec_chapter_ctg.name ] = _( "Create New..." );
	// a dummy value to differentiate it from "None":
	row[ colrec_chapter_ctg.ptr ] = ( CategoryChapters* ) 0x1;
	row[ colrec_chapter_ctg.type ] = CCCT_NEW;
}

void
DiaryView::handle_cur_chapter_ctg_changed( void )
{
	if( Lifeobase::m_internaloperation ) return;

	Gtk::TreeIter iter = m_combobox_chapter_ctg->get_active();
	bool flag_renamed = ! ( iter ), flag_existing_ctg = false;

	if( flag_renamed )
	{
		Glib::ustring name = m_combobox_chapter_ctg->get_active_text();
		bool flag_name_usable = m_ptr2diary->m_chapter_categories.count( name ) < 1;
		m_button_chapter_ctg_rename->set_sensitive( flag_name_usable );
	}
	else
	{
		Gtk::TreeRow row = *iter;
		switch( row[ colrec_chapter_ctg.type ] )
		{
			case CCCT_NEW: // create new
				m_ptr2diary->m_ptr2chapter_ctg_cur = m_ptr2diary->create_chapter_ctg();
				update_combobox_chapter_ctg();
				flag_existing_ctg = true;
				break;
			case CCCT_CATEGORY:
				flag_existing_ctg = true;
			case CCCT_NONE:
				m_ptr2diary->m_ptr2chapter_ctg_cur = row[ colrec_chapter_ctg.ptr ];
				Lifeobase::base->update_entry_list();
				break;
			default:	// for separators
				break;
		}

		m_button_chapter_ctg_dismiss->set_sensitive( flag_existing_ctg );
	}

	m_button_chapter_ctg_rename->set_visible( flag_renamed );
	m_button_chapter_ctg_dismiss->set_visible( ! flag_renamed );
	m_combobox_chapter_ctg->get_child()->set_sensitive( flag_existing_ctg || flag_renamed );
}

void
DiaryView::rename_cur_chapter_ctg( void )
{
	Glib::ustring name = m_combobox_chapter_ctg->get_active_text();
	if( name == m_ptr2diary->m_ptr2chapter_ctg_cur->get_name() )
		return;
	m_ptr2diary->rename_chapter_ctg( m_ptr2diary->m_ptr2chapter_ctg_cur, name );
	update_combobox_chapter_ctg();
}

void
DiaryView::dismiss_cur_chapter_ctg( void )
{
	Gtk::MessageDialog message( *Lifeobase::base, "", false,
								Gtk::MESSAGE_WARNING, Gtk::BUTTONS_NONE, true );
	message.set_message( _( "Are you sure, you want to dismiss?" ) );
	message.set_secondary_text( _( "This operation cannot be undone!" ) );
	message.add_button( Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL );
	message.add_button( _( "_DISMISS!" ), Gtk::RESPONSE_ACCEPT );

	if( message.run() != Gtk::RESPONSE_ACCEPT )
		return;

	m_ptr2diary->dismiss_chapter_ctg( m_ptr2diary->m_ptr2chapter_ctg_cur );
	update_combobox_chapter_ctg();
}

bool
DiaryView::is_row_separator( const Glib::RefPtr< Gtk::TreeModel > &model,
							 const Gtk::TreeIter iter )
{
	Gtk::TreeRow row = * iter;
	return ( row[ colrec_chapter_ctg.type ] == CCCT_SEPARATOR );
}
