/**
 * \file script.h
 *
 * see script_lua.c
 * see fetchnews.c::getarticle()
 * see store.c::store_stream()
 *
 * see script-test-store.c for testing all this.
 *
 * clemens fischer <ino-news@spotteswoode.dnsalias.org>
 */

#include "leafnode.h"

#if ! defined(__unused)
#define __unused __attribute__((unused))
#endif

#if ! defined(__noreturn)
#define __noreturn __attribute__((noreturn))
#endif

#if defined(SCRIPT_TEST)
#define BODY_MAX 399
#endif

#if ! defined(MAX_SCRIPT_ERRORS)
#define MAX_SCRIPT_ERRORS 13
#endif

/*
 * This looks ridiculous, but it is intentional.  There is yet no syntax
 * in the filtering-engine for specifying body filters.  Since there is
 * normally no "Date:" header in the body of an article, filters may
 * trigger on eg. "maxage".  Until such syntax such as a "body" keyword
 * is implemented, the feature has to be switched off!
 *
 */
#define WITHOUT_FILTER_BODY_PCRE 1
#if ! defined(WITHOUT_FILTER_BODY_PCRE)
#define FILTER_BODY_PCRE 1
#endif

#if defined(WITH_SCRIPT_LUA) || defined(WITH_SCRIPT_PYTHON) || defined(WITH_SCRIPT_PERL)
#define WITH_SCRIPT "oh_yes"
#endif

/*
 * "LN_(UN)?AUTHENTICATED_USER" is the name of a global variable in scripts
 * "LN_CURRENTGROUP" is the name of a global variable in scripts
 * "SCRIPT_UNAUTHENTICATED_USER" is the value of a constant while user unknown
 * "SCRIPT_CURRENT_GROUP" is the value of a constant when group name unavailable
 */
#define LN_AUTHENTICATED_USER "AUTHENTICATED_USER"
#define LN_UNAUTHENTICATED_USER "UNAUTHENTICATED_USER"
#define SCRIPT_UNAUTHENTICATED_USER "UNAUTHENTICATED_USER"
#define LN_CURRENTGROUP "CURRENT_GROUP"
#define SCRIPT_CURRENT_GROUP "CURRENT_GROUP"

/* the following definitions are always needed */

#if ! defined(BODY_MAX)
#define BODY_MAX 55000
#endif

/* script status returns */
#define SCRIPT_NOERROR 0             /* ok */
#define SCRIPT_REJECT 1              /* reject article or group */
#define SCRIPT_NOCHANGE 2            /* neither header nor body changed */
#define SCRIPT_NOCHANGE_HEADER 3     /* header unchanged */
#define SCRIPT_NOCHANGE_BODY 4       /* body unchanged */
#define SCRIPT_UNAVAILABLE 5         /* no error, but item unavailable */
#define SCRIPT_IGNORE_ARTICLE 7      /* ignore this item */
#define SCRIPT_IGNORE_GROUP 8        /* ignore this item */
#define SCRIPT_ILLEGAL_SEQ 9         /* API fun called out of sequence,
                                        handled within API code */
#define SCRIPT_ILLEGAL_API_ARGS 10   /* API fun got problematic args */
#define SCRIPT_API_RETURN_PROB 11    /* API fun returned problem */
#define SCRIPT_ERROR_UNSPEC 12       /* unspecified error (shouldn't
                                        happen, ignore or report) */
#define SCRIPT_WRONG_RESULT 13       /* API returned wrong result type! */
#define SCRIPT_RUNTIME_PROB 14       /* backend runtime prob */
#define SCRIPT_STATUS_MAX 19         /* status values MUST be less than this! */

/*
 * script result retrieval "commands":
 * you can issue any command, but the data might not be available.
 * the reason can be "illegal state" (of the FSM),
 * "ignored" (if API currently doesn't return this item) etc.
 */
#define SCRIPT_GET_NOTHING 0        /* retrieve nothing */
#define SCRIPT_GET_STATUS 1         /* retrieve status of last operation */
#define SCRIPT_GET_NEWSGROUPS 2     /* get CSVs of newsgroups to store into */
#define SCRIPT_GET_HEADER 3         /* get possibly changed header */
#define SCRIPT_GET_BODY 4           /* get possibly changed body */

/* -ino: XXX redefine as appropriate
 *
 * NB: all script functions return status numbers as defined above or
 *     error messages as C-strings, only the script_*_get_result
 *     function(s) return leafnodes main data type - "mastr" pointers -
 *     which is automatically extended as needed and carries a string,
 *     its length and the currently used buffer size.
 *
 */
typedef const /*unsigned*/ char * script_return_t;
typedef mastr * script_result_t;
typedef mastr * script_data_t;
typedef int script_cmd_t;

/* debugging */
const char * readable_status(script_return_t);

#if defined(WITH_SCRIPT)
#warning "compiling real script functions"

int script_init_ready(int);

/* prototypes fetchnews */
script_return_t script_fn_init(void);
script_return_t script_fn_finish(int);

script_return_t script_fn_init_article(void);
script_return_t script_fn_finish_article(int);

script_return_t script_fn_add_header(script_data_t);
script_return_t script_fn_add_body(script_data_t);

script_return_t script_fn_init_group(char *);
script_return_t script_fn_finish_group(int);

script_return_t script_fn_filter_header_table(void);
script_return_t script_fn_filter_header(script_data_t);
script_return_t script_fn_filter_header_body(script_data_t, script_data_t);

script_result_t script_fn_get_result(script_cmd_t);

/* prototypes leafnode */
script_return_t script_ln_init(void);
script_return_t script_ln_chk_grp_access(const char *, const char *);
int script_ln_register_user(const char *);

/* check structured script return values */
unsigned long int
check_script_return(script_return_t, const char *, const char *, script_cmd_t, script_result_t *);

#else /* not defined(WITH_SCRIPT) */
#warning "compiling dummy script functions"

#define script_init_ready(_1) 0
#define script_ln_chk_grp_access(_1, _2) ((script_return_t) SCRIPT_UNAVAILABLE)
#define script_ln_register_user(_1) 0
#define script_ln_init() ((script_return_t) SCRIPT_UNAVAILABLE)
#define script_fn_init() ((script_return_t) SCRIPT_UNAVAILABLE)
#define script_fn_init_article(_1) ((script_return_t) SCRIPT_UNAVAILABLE)
#define script_fn_init_group(_1) ((script_return_t) SCRIPT_UNAVAILABLE)
#define script_fn_finish_group(_1) ((script_return_t) SCRIPT_UNAVAILABLE)
#define script_fn_get_result(_1) ((script_return_t) SCRIPT_UNAVAILABLE)
#define script_fn_add_header(_1) ((script_return_t) SCRIPT_UNAVAILABLE)
#define script_fn_filter_header_table(_1) ((script_return_t) SCRIPT_UNAVAILABLE)
#define script_fn_filter_header(_1) ((script_return_t) SCRIPT_UNAVAILABLE)
#define script_fn_filter_header_body(_1, _2) ((script_return_t) SCRIPT_UNAVAILABLE)
#define script_fn_add_body(_1) ((script_return_t) SCRIPT_UNAVAILABLE)
#define script_fn_finish(_1) ((script_return_t) SCRIPT_UNAVAILABLE)
#define script_fn_finish_article(_1) ((script_return_t) SCRIPT_UNAVAILABLE)
#define check_script_return(_1, _2, _3, _4, _5) ((long unsigned int) SCRIPT_NOERROR)

#endif

#define IS_STATUSCODE(status) ((unsigned long int) status < SCRIPT_STATUS_MAX)
#define IS_STRINGRESULT(status) (((unsigned long int) status > SCRIPT_STATUS_MAX) && (strlen(status) > 0))
#define IS_NULLRESULT(status) ((status == NULL) || \
        (((unsigned long int) status > SCRIPT_STATUS_MAX) && (strlen(status) == 0)))
