/*
 * ldm.c
 * LTSP display manager.
 * Manages spawning a session to a server.
 *
 * (c) Scott Balneaves, sbalneav@ltsp.org
 *
 * This software is licensed under the GPL v2 or later.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <syslog.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/wait.h>
#include <glib.h>
#include <glib-object.h>

#include "ldm.h"


#include <config.h>
#include <libintl.h>
#include <locale.h>
#define _(text) gettext(text)


struct ldm_info ldminfo;
FILE *ldmlog;

void
usage()
{
    fprintf(stderr, "Usage: ldm <vt[1-N]> <:[0-N]> [xserver options]\n");
    exit(1);
}

/*
 * die()
 *
 * Close display manager down with an error message.
 */

void
die(char *msg)
{
    fprintf(ldmlog, "%s\n", msg);
    fclose(ldmlog);

    /*
     * Shut things down gracefully if we can
     */ 

    if (ldminfo.greeterpid)
        close_greeter();
    if (ldminfo.sshpid)
        ssh_endsession();
    if (ldminfo.xserverpid) {
        kill(ldminfo.xserverpid, SIGTERM);      /* Kill Xorg server off */
        ldm_wait(ldminfo.xserverpid);
    }

    exit(1);
}

/*
 * Print out command lines.  For debugging.
 */

void
dump_cmdline(char *argv[])
{
    int i;

    for (i = 0; argv[i]; i++)
        fprintf(ldmlog, "%s ", argv[i]);
    fprintf(ldmlog, "\n");
}
/*
 * scopy()
 *
 * Copy a string.  Used to move data in and out of our ldminfo structure.
 * Note: if the source string is null, or points to a valid string of '\0',
 * both result in a dest string length of 0.
 */

char *
scopy(char *dest, char *source)
{
    if (!source)
        *dest = '\0';
    else {
        strncpy(dest, source, LDMSTRSZ - 1);
        *(dest + LDMSTRSZ - 1) = '\0';     /* ensure null termination */
    }

    return dest;
}

/*
 * ldm_spawn:
 *
 * Execute commands.  Ignore stdin and stdout.
 */

GPid
ldm_spawn (char **argv)
{
    GPid pid;
    gboolean res;
  
    res = g_spawn_async(NULL, argv, NULL,
                  G_SPAWN_DO_NOT_REAP_CHILD |
                  G_SPAWN_STDOUT_TO_DEV_NULL |
                  G_SPAWN_STDERR_TO_DEV_NULL,
                  NULL, NULL, &pid, NULL);

    if (!res) {
        fprintf(ldmlog, "ldm_spawn failed to execute:\n");
        dump_cmdline(argv);
        die("Exiting ldm\n");
    }

    return pid;
}

int
ldm_wait(GPid pid)
{
    int status;

    if (waitpid (pid, &status, 0) < 0)
        die(_("Error: wait() call failed"));
    if (!WIFEXITED (status)) {
        fprintf(ldmlog, "Process returned no status\n");
        return 1;
    } else
        return WEXITSTATUS (status);
}

void
create_xauth()
{
    GPid xauthpid;
    int status;

    char *xauth_command[] = {
       "/usr/bin/xauth", 
       "-i",
       "-n",
       "-f",
       ldminfo.authfile,
       "generate",
       ldminfo.display,
       NULL };

    /*
     * Since we're talking to the X server, we might have to give it a moment
     * or two to start up.  So do this in a loop.
     */

    do {
        sleep(1);
        xauthpid = ldm_spawn(xauth_command);
        status = ldm_wait(xauthpid);
    } while (status);
}

void
launch_x(char **xserver_args)
{
    char *argv[MAXARGS];
    int i = 0;
    int fd;
    int x = 0;

    /* FIXME: Should generate xauth here using libXau
              like gdm instead of screen.d/ldm */

    /* Spawn the X server */
    argv[i++] = "/usr/bin/X";
    argv[i++] = "-auth";
    argv[i++] = ldminfo.authfile;
    // set default X server options
    argv[i++] = "-br";
    argv[i++] = "-noreset";
    if (xserver_args[3]) {
        // ldm was passed additional options beyond vt and display,
        // pass these on to the X server
        for (x=3; xserver_args[x]; x++)
            argv[i++]=xserver_args[x];
    }
    argv[i++] = ldminfo.vty;
    argv[i++] = ldminfo.display;
    argv[i++] = NULL;
        
    ldminfo.xserverpid = ldm_spawn(argv);
}

void
rc_files(char *action)
{
    GPid rcpid;

    char *rc_cmd[] = {
        "/bin/sh",
        RC_DIR "/ldm-script",
        action,
        NULL };

    /* start */
        
    rcpid = ldm_spawn(rc_cmd);
    ldm_wait(rcpid);
}

void
x_session()
{
    char *cmd[MAXARGS];
    char displayenv[ENVSIZE];
    char ltspclienv[ENVSIZE];
    char soundcmd1[ENVSIZE];
    char soundcmd2[ENVSIZE];
    char lang1[ENVSIZE];
    char lang2[ENVSIZE];
    char lang3[ENVSIZE];
    char *esdcmd[] = {
        "/usr/bin/esd",
        "-nobeeps",
        "-public", 
        "-tcp",
        NULL };
    GPid xsessionpid;
    GPid esdpid = 0;
    int i = 0;

    snprintf(ltspclienv, sizeof ltspclienv, "LTSP_CLIENT=%s", ldminfo.ipaddr);
    if (ldminfo.directx)
        snprintf(displayenv, sizeof displayenv, 
                "DISPLAY=%s%s ", ldminfo.ipaddr, ldminfo.display);
    if (*(ldminfo.lang) != '\0') {
        snprintf(lang1, sizeof lang1, "LC_ALL=%s", ldminfo.lang);
        snprintf(lang2, sizeof lang2, "LANGUAGE=%s", ldminfo.lang);
        snprintf(lang3, sizeof lang3, "LANG=%s", ldminfo.lang);
    }

    cmd[i++] = "/usr/bin/ssh";
    cmd[i++] = "-Y";
    cmd[i++] = "-t";
    cmd[i++] = "-S";
    cmd[i++] = ldminfo.control_socket;
    cmd[i++] = "-l";
    cmd[i++] = ldminfo.username;
    cmd[i++] = ldminfo.server;
    cmd[i++] = ltspclienv;

    /*
     * Set our language, if a different one is picked.
     */

    if (*(ldminfo.lang) != '\0') {
        cmd[i++] = lang1;
        cmd[i++] = lang2;
        cmd[i++] = lang3;
    }

    /*
     * Set the DISPLAY env, if not running over encrypted ssh
     */

    if (ldminfo.directx)
        cmd[i++] = displayenv;

    /*
     * Handle sound
     */

    if (ldminfo.sound) {
        char *daemon = ldminfo.sound_daemon;  

        if (!daemon || !strncmp(daemon, "pulse", 5)) {
            snprintf(soundcmd1, sizeof soundcmd1, "PULSE_SERVER=tcp:%s:4713",
                     ldminfo.ipaddr);
            snprintf(soundcmd2, sizeof soundcmd2, "ESPEAKER=%s:16001",
                     ldminfo.ipaddr);
            cmd[i++] = soundcmd1;
            cmd[i++] = soundcmd2;
        } else if (!strncmp(daemon, "esd", 3)) {
            snprintf(soundcmd1, sizeof soundcmd1, "ESPEAKER=%s:16001",
                     ldminfo.ipaddr);
            cmd[i++] = soundcmd1;
            esdpid = ldm_spawn(esdcmd);         /* launch ESD */
        } else if (!strncmp(daemon, "nasd", 4)) {
            snprintf(soundcmd1, sizeof soundcmd1, "AUDIOSERVER=%s:0",
                     ldminfo.ipaddr);
            cmd[i++] = soundcmd1;
        }
    }
            
    cmd[i++] = ldminfo.session;

    if (ldminfo.localdev) {
        cmd[i++] = ";";        /* closes bug number 121254 */
        cmd[i++] = "/usr/sbin/ltspfsmounter";
        cmd[i++] = "all";
        cmd[i++] = "cleanup";
    }

    cmd[i++] = ";";
    cmd[i++] = "kill";
    cmd[i++] = "-1";
    cmd[i++] = "$PPID";
    cmd[i++] = NULL;

    xsessionpid = ldm_spawn(cmd);
    ldm_wait(xsessionpid);
    if (esdpid) {
        kill(esdpid, SIGTERM);
        ldm_wait(esdpid);
    }
}

/*
 * Load guest info
 */

void load_guestinfo()
{
       char **hosts_char = NULL;

        // Get all info for autologin, without greeter
        fprintf(ldmlog, "Logging in as guest\n");
        scopy(ldminfo.username, getenv("LDM_USERNAME"));
        scopy(ldminfo.password, getenv("LDM_PASSWORD"));
        if (*(ldminfo.username) == '\0')
        {
            char hostname[128];
            gethostname(hostname, 128);
            scopy(ldminfo.username, hostname);
        }
        if (*(ldminfo.password) == '\0')
            scopy(ldminfo.password, ldminfo.username);

        char autoservers[128];
        scopy(autoservers, getenv("LDM_GUEST_SERVER"));
        if (*(autoservers) == '\0')
            scopy(autoservers, getenv("LDM_AUTOLOGIN_SERVER"));
        if (*(autoservers) == '\0')
            scopy(autoservers, getenv("LDM_SERVER"));

        // Make sure that if there are multiple servers in the list
        // always take the first one
        hosts_char = g_strsplit(autoservers, " ", -1);

        if ( hosts_char[0] != NULL )
            scopy(ldminfo.server, hosts_char[0]);                                
        return;
}

/*
 * mainline
 */
        
int
main(int argc, char *argv[])
{
    /* decls */
    char display_env[ENVSIZE], xauth_env[ENVSIZE];
    char server_env[ENVSIZE], socket_env[ENVSIZE];
    gboolean err_flag = FALSE;
    char *err_msg = NULL;
    char *p = NULL;
    int i;

#ifdef ENABLE_NLS
     setlocale (LC_ALL, "");
     bindtextdomain (GETTEXT_PACKAGE, LOCALE_DIR);
     bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
     textdomain (GETTEXT_PACKAGE);
#endif

    g_type_init();

    /*
     * Zero out our info struct.
     */

    bzero(&ldminfo, sizeof ldminfo);

    /*
     * Process command line args.  Need to get our vt, and our display number
     * Since we don't have any fancy args, we won't bother to do this with
     * getopt, rather, we'll just do it manually.
     */

    if (argc < 3)
        usage();

    if (strncmp(argv[1], "vt", 2))      /* sanity */
        usage();
    if (*argv[2] != ':')            /* more sanity */
        usage();

    scopy(ldminfo.vty, argv[1]);
    scopy(ldminfo.display, argv[2]);

    /*
     * Open our log.  Since we're on a terminal, logging to syslog is preferred,
     * as then we'll be able to collect the logging on the server.
     * Since we're handling login info, log to AUTHPRIV
     */

    if (!(ldmlog = fopen("/var/log/ldm.log", "a")))
        die(_("Couldn't open /var/log/ldm.log"));

    setbuf(ldmlog, NULL);      /* Unbuffered */

    /*
     * openlog("ldm", LOG_PID | LOG_PERROR , LOG_AUTHPRIV);
     */

    fprintf(ldmlog, "LDM2 starting\n");

    /*
     * Get our IP address.
     */

    get_ipaddr();
    fprintf(ldmlog, "LDM2 running on ip address %s\n", ldminfo.ipaddr);

    // put ip address in environment so that it is available to to the greeter
    setenv("LDMINFO_IPADDR", ldminfo.ipaddr, 1);

    /*
     * Get some of the environment variables we'll need.
     */

    ldminfo.allowguest = ldm_getenv_bool("LDM_GUESTLOGIN");
    ldminfo.sound = ldm_getenv_bool("SOUND");
    scopy(ldminfo.sound_daemon, getenv("SOUND_DAEMON"));
    ldminfo.localdev = ldm_getenv_bool("LOCALDEV");
    scopy(ldminfo.override_port, getenv("SSH_OVERRIDE_PORT"));
    ldminfo.directx = ldm_getenv_bool("LDM_DIRECTX");
    ldminfo.autologin = ldm_getenv_bool("LDM_AUTOLOGIN");
    scopy(ldminfo.lang, getenv("LDM_LANGUAGE"));
    scopy(ldminfo.session, getenv("LDM_SESSION"));
    if (*ldminfo.session == '\0') {
        if(!access("/etc/X11/xinit/Xsession", X_OK)) {
            /* Fedora, RHEL, CentOS, etc. */
            scopy(ldminfo.session, "/etc/X11/xinit/Xsession");
        }
        else if(!access("/etc/X11/Xsession", X_OK)) {
            /* Debian, Ubuntu */
            scopy(ldminfo.session, "/etc/X11/Xsession");
        }
        else if(!access("/usr/lib/X11/xdm/Xsession", X_OK)) {
            /* Gentoo */
            scopy(ldminfo.session, "/usr/lib/X11/xdm/Xsession");
        }
        else if(!access("/etc/X11/xdm/Xsession", X_OK)) {
            /* OpenSUSE */
            scopy(ldminfo.session, "/etc/X11/xdm/Xsession");
        }
    }
    scopy(ldminfo.greeter_prog, getenv("LDM_GREETER"));
    if (*ldminfo.greeter_prog == '\0')
        scopy(ldminfo.greeter_prog, LDM_EXEC_DIR "/ldmgtkgreet");
    scopy(ldminfo.authfile, "/root/.Xauthority");

    snprintf(display_env, sizeof display_env,  "DISPLAY=%s", ldminfo.display);
    snprintf(xauth_env, sizeof xauth_env, "XAUTHORITY=%s", ldminfo.authfile);

    /* 
     * Update our environment with a few extra variables.
     */

    putenv(display_env);
    putenv(xauth_env);

    /*
     * Begin running display manager
     */

    fprintf(ldmlog, "Launching Xorg\n");
    launch_x(argv);
    
    if (!ldminfo.autologin) {
        fprintf(ldmlog, "Spawning greeter: %s\n", ldminfo.greeter_prog);

        spawn_greeter();
    
        // ask greeter to fill ldminfo.username and ldminfo.server
        if (get_userid())
            die(_("ERROR: get_userid from greeter failed"));

        if (!ldminfo.autologin)
            if (get_passwd())
                die(_("ERROR: get_passwd from greeter failed"));

        if (get_host())
            die(_("ERROR: get_host from greeter failed"));

        if (get_language())
            die(_("ERROR: get_language from greeter failed"));
            
        if (get_session())
            die(_("ERROR: get_session from greeter failed"));

    }

    if(ldminfo.autologin)
      load_guestinfo();

    // Verify that we have all info needed to connect
    if (*(ldminfo.username) == '\0'){
        err_msg = g_strconcat(_("ERROR: username variable empty"), "\n", NULL );
        err_flag = TRUE;
    }
    if (*(ldminfo.password) == '\0'){
        err_msg = g_strconcat(_("ERROR: password variable empty"), "\n", NULL );
        err_flag = TRUE;
    }
    if (*(ldminfo.server) == '\0'){
        err_msg = g_strconcat(_("ERROR: server variable empty"), "\n", NULL );
        err_flag = TRUE;
    }

    if (err_flag){
        fprintf(ldmlog, "%s", err_msg);
        die(_("Fatal error, missing mandatory information"));
    }

    /*
     * If we run multiple ldm sessions on multiply vty's we need separate 
     * control sockets.
     */

    snprintf(ldminfo.control_socket, sizeof ldminfo.control_socket,
             "/var/run/ldm_socket_%s_%s", ldminfo.vty, ldminfo.server);
    snprintf(socket_env, sizeof socket_env, "LDM_SOCKET=%s", 
             ldminfo.control_socket);
    putenv(socket_env);
    snprintf(server_env, sizeof server_env,  "LDM_SERVER=%s", ldminfo.server);
    putenv(server_env);

    fprintf(ldmlog, "Establishing a session with %s\n", ldminfo.server);

    while(TRUE) {
        int retval;
        fprintf(ldmlog, "Attempting ssh session as %s\n", ldminfo.username);
        retval = ssh_session();             /* Log in via ssh */
        if (!retval)
            break;
        if (retval == 2) {
            ldm_wait(ldminfo.sshpid);
            close(ldminfo.sshfd);
        }
    }

    /*
     * Clear out the password so it's not sitting in memory anywhere
     */

    bzero(ldminfo.password, sizeof ldminfo.password);

    fprintf(ldmlog, "Established ssh session.\n");
    if (ldminfo.greeterpid)
        close_greeter();

    fprintf(ldmlog, "Executing rc files.\n");
    rc_files("start");                      /* Execute any rc files */
    fprintf(ldmlog, "Beginning X session.\n");
    x_session();                            /* Start X session up */
    fprintf(ldmlog, "X session ended.\n");

    /* x_session's exited.  So, clean up. */

    fprintf(ldmlog, "Executing rc files.\n");
    rc_files("stop");                       /* Execute any rc files */

    fprintf(ldmlog, "Killing X server.\n");
    kill(ldminfo.xserverpid, SIGTERM);      /* Kill Xorg server off */
    ldm_wait(ldminfo.xserverpid);

    fprintf(ldmlog, "Ending ssh session.\n");
    ssh_endsession();                       /* Log out of server */

    fclose(ldmlog);
    exit(0);
}
