.\"
.\" $Id: rfio_preseek.man,v 1.1 2005/03/31 13:13:03 baud Exp $
.\"
.\" @(#)$RCSfile: rfio_preseek.man,v $ $Revision: 1.1 $ $Date: 2005/03/31 13:13:03 $ CERN IT-PDP/DM Jean-Philippe Baud
.\" Copyright (C) 1990-2002 by CERN/IT/PDP/DM
.\" All rights reserved
.\"
.TH RFIO_PRESEEK 3 "$Date: 2005/03/31 13:13:03 $" CASTOR "Rfio Library Functions"
.SH NAME
rfio_preseek \- prefetch chunks of data from a file
.SH SYNOPSIS
.B #include <sys/types.h>
.br
\fB#include "rfio_api.h"\fR
.sp
.BI "int rfio_preseek (int " s ", const struct iovec *" iov ", int " iovnb ");"
.sp
Under Linux, for large files:
.br
.B #define _LARGEFILE64_SOURCE
.br
.B #include <sys/types.h>
.br
\fB#include "rfio_api.h"\fR
.sp
.BI "int rfio_preseek64 (int " s ", const struct iovec64 *" iov ", int " iovnb ");"
.br
.sp
For large files, under other systems:
.br
.B #include <sys/types.h>
.br
\fB#include "rfio_api.h"\fR
.sp
.BI "int rfio_preseek64 (int " s ", const struct iovec64 *" iov ", int " iovnb ");"
.SH DESCRIPTION
.B rfio_preseek
prefetches chunks of data at given offsets into an internal buffer (on
the client side), using the descriptor
.BI s
generated by a previous
.BR rfio_open .
The actual offset values and the lengths are given in the array of structures
.BR iov .
The number of chunks is specified by
.BR iovnb .
.LP
.B rfio_preseek64
does the prefetch for large files, using an array of structures of type
.BR iovec64 
rather than an array of 
.BR iovec .
.LP
.B RFIO_READOPT
must be set to RFIO_READBUF, which is the default.
The default internal buffer size is 128 kB, but the buffer size can be set with
an entry RFIO IOBUFSIZE in 
.BR shift.conf .
Then 
.B rfio_read
gets the data from that buffer.
.SH EXAMPLES
.nf
.ft CW
/*	tpreseek - write NBRECORDS_TOWRITE records and
		read back NBRECORDS_TOREAD using the rfio_preseek function */

#include <fcntl.h>
#include <stdio.h>
#if defined(_WIN32)
#include <winsock2.h>
#endif
#include "rfio_api.h"
#define NBRECORDS_TOREAD 5
#define NBRECORDS_TOWRITE 10

main(argc, argv)
int argc;
char **argv;
{
	char buf[65536];
	int errflg = 0;
	int fd;
	int i;
	struct iovec iov[NBRECORDS_TOREAD];
	int iovnb = NBRECORDS_TOREAD;
	int j;
	static int lengths[NBRECORDS_TOWRITE] = {4096, 32768, 16384, 8192,
				   65536, 32768, 16384, 4096, 65536, 8192};
	static int records_toread[NBRECORDS_TOREAD] = {2, 4, 5, 8, 9};
#if defined(_WIN32)
	WSADATA wsadata;
#endif

	if (argc != 2) {
		fprintf (stderr, "usage: tpreseek pathname\\n");
		exit (1);
	}
#if defined(_WIN32)
	if (WSAStartup (MAKEWORD (2, 0), &wsadata)) {
		fprintf (stderr, "WSAStartup unsuccessful\\n");
		exit (2);
	}
#endif
	while (! errflg) {

		/* Write variable length records.
		 * Each record is filled with the record index
		 */

		if ((fd = rfio_open (argv[1],
			    O_WRONLY|O_CREAT|O_TRUNC, 0644)) < 0) {
			rfio_perror ("rfio_open");
			errflg++;
			break;
		}
		for (j = 0; j < NBRECORDS_TOWRITE; j++) {
			for (i = 0; i < lengths[j]; i++)
				buf[i] = j;
			if (rfio_write (fd, buf, lengths[j]) < 0) {
				rfio_perror ("rfio_write");
				errflg++;
				break;
			}
		}
		(void)rfio_close (fd);
		if (errflg) break;

		/* Prefetch a few records: the actual offsets and lengths
		 * of the records is set in the array of iov structures
		 */

		if ((fd = rfio_open (argv[1], O_RDONLY)) < 0) {
			rfio_perror ("rfio_open");
			errflg++;
			break;
		}
		for (j = 0; j < NBRECORDS_TOREAD; j++) {
			/* compute record offset */
			iov[j].iov_base = 0;
			for (i = 0; i < records_toread[j]; i++)
				iov[j].iov_base = (char *) iov[j].iov_base +
				    lengths[i];
			/* set length */
			iov[j].iov_len = lengths[records_toread[j]];
		}
		if (rfio_preseek (fd, iov, iovnb) < 0) {
			rfio_perror ("rfio_preseek");
			errflg++;
			break;
		}

		/* Read back the records and check their cpntents */

		for (j = 0; j < NBRECORDS_TOREAD; j++) {
			if (rfio_lseek (fd, (off_t) iov[j].iov_base,
			    SEEK_SET) < 0) {
				rfio_perror ("rfio_lseek");
				errflg++;
				break;
			}
			if (rfio_read (fd, buf, iov[j].iov_len) < 0) {
				rfio_perror ("rfio_read");
				errflg++;
				break;
			}
			for (i = 0; i < iov[j].iov_len; i++) {
				if (buf[i] != records_toread[j]) {
					fprintf (stderr,
					    "incorrect data read, record %d\\n",
					    records_toread[j]);
					errflg++;
					break;
				}
			}
			if (errflg) break;
		}
		(void) rfio_close (fd);
		break;
	}
	if (rfio_unlink (argv[1]) < 0) {
		rfio_perror ("rfio_unlink");
		errflg++;
	}
#if defined(_WIN32)
	WSACleanup();
#endif
	exit (errflg ? 1 : 0);
}
.ft
.fi
.SH RETURN VALUE
This routine returns 0 if the operation was successful or -1 if the operation
failed. In the latter case, 
.B serrno
is set appropriately.
.SH ERRORS
.TP 1.3i
.B EBADF
.I s
is not a valid descriptor.
.TP
.B EINVAL
RFIO_READOPT is not set to RFIO_READBUF.
.TP
.B SENOSHOST
Host unknown.
.TP
.B SENOSSERV
Service unknown.
.TP
.B SETIMEDOUT
Timed out.
.TP
.B SEBADVERSION
Version ID mismatch.
.TP
.B SECONNDROP
Connection closed by remote end.
.TP
.B SECOMERR
Communication error.
.TP
.B SENORCODE
Host did not return error number.
.SH SEE ALSO
.BR rfio_lseek(3) ,
.BR rfio_open(3) ,
.BR rfio_read(3)
.SH AUTHOR
\fBCASTOR\fP Team <castor.support@cern.ch>
