/***************************************************************************
                          kpgtable.cpp  -  description
                             -------------------
    begin                : � led 6 2004
    copyright            : (C) 2004 by Lumir Vanek
    email                : lvanek@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "kpgtable.h"

 
// include files for KDE
#include <kiconloader.h>
#include <kdebug.h>
#include <klocale.h>
#include <kcursor.h>

// application specific includes
#include "../kpogreview.h"
#include "../kpgconfiguration.h"
#include "../kpggeneralsettings.h"
#include "kpgserver.h"
#include "kpgdatabase.h"
#include "kpgtablesfolder.h"
#include "kpgtablecolumnsfolder.h"
#include "kpgtableconstraintsfolder.h"
#include "kpgtableindexesfolder.h"
#include "kpgrulesfolder.h"
#include "kpgtriggersfolder.h"
#include "../kpgsqldialog.h"


KPGTable::KPGTable(KPGTablesFolder *parent, const QString name, pqxx::oid _oid)
  : KPGObject(parent, name, _oid)
{
    m_bIsLoaded = false;
    setPixmap(0, *m_pIconTable);
	
	m_pFolderColumns = new KPGTableColumnsFolder(this);
	m_pFolderConstraints = new KPGTableConstraintsFolder(this, m_pFolderColumns);
	m_pFolderIndexes = new KPGTableIndexesFolder(this, m_pFolderConstraints);
	m_pFolderRules = new KPGRulesFolder(this, m_pFolderIndexes);
	m_pFolderTriggers = new KPGTriggersFolder(this, m_pFolderRules);
}

KPGTable::KPGTable(KPGTablesFolder *parent, KPGTable *after, const QString name, pqxx::oid _oid)
  : KPGObject(parent, after, name, _oid)
{
    m_bIsLoaded = false;
    setPixmap(0, *m_pIconTable);
	
	m_pFolderColumns = new KPGTableColumnsFolder(this);
	m_pFolderConstraints = new KPGTableConstraintsFolder(this, m_pFolderColumns);
	m_pFolderIndexes = new KPGTableIndexesFolder(this, m_pFolderConstraints);
	m_pFolderRules = new KPGRulesFolder(this, m_pFolderIndexes);
	m_pFolderTriggers = new KPGTriggersFolder(this, m_pFolderRules);
}

KPGTable::~KPGTable()
{
}

void KPGTable::activate()
{
	if(!m_bIsLoaded)
	{ 
		listView()->setCursor(KCursor::waitCursor());
		try
		{
			refresh();
		}
		catch (const KPGSqlException &e)
		{
			listView()->setCursor(KCursor::arrowCursor());
            KPGSqlDialog dlg(0, e.sql(), e.message());
			dlg.exec();
		}
		listView()->setCursor(KCursor::arrowCursor());
		setOpen(true);
	}
}

void KPGTable::refresh() throw(const KPGSqlException &)
{
    //kdDebug() << "Refreshing table: " << text(0) << endl;
    refreshItem();
    
    refreshChildItems();
			
	m_bIsLoaded = true;
}

void KPGTable::refreshChildItems() throw(const KPGSqlException &)
{
    try
    {
		m_pFolderColumns->refresh(m_oid);
		m_pFolderConstraints->refresh(m_oid);
		m_pFolderIndexes->refresh(m_oid);
		m_pFolderRules->refresh(m_oid);
		m_pFolderTriggers->refresh(m_oid);
	}
	catch (const KPGSqlException &e)
	{
		kdError() << k_funcinfo << "Routing exception up. Table: " << text(0) << endl;
		throw; // if it is KPGSqlException re-throw it
	}
}

// Refresh only table info, without childs objects
void KPGTable::refreshItem() throw(const KPGSqlException &)
{
	// Get pointer to server for version info
	KPGDatabase *pDatabase = static_cast <KPGDatabase *> (parent()->parent()->parent());
	KPGServer *pServer = static_cast <KPGServer *> (pDatabase->parent());
	
	bool bVersion80_OrNewer = false;
	bool bVersion81_OrNewer = false;
	    
	// Is it 8.0 or newer ?
	if(pServer->versionMajor() > 7)
    {             
       bVersion80_OrNewer = true;
    }     
	    
    // Is it 8.1 or newer ?
	if(((pServer->versionMajor() == 8) && (pServer->versionMiddle() >= 1)) || ((pServer->versionMajor() > 8))) 
	{
		bVersion81_OrNewer = true;
	}
		
	// obtain table info
	QString strQuery("SELECT rel.oid, rel.relname, rel.reltype, description, pg_get_userbyid(rel.relowner) AS relowner");
	strQuery.append(", rel.relhasoids, rel.relhassubclass, rel.reltuples, c.conname, c.conkey, rel.relacl");
    strQuery.append(", CASE rel.relkind WHEN 'r' THEN 'Ordinary table' WHEN 't' THEN 'Toast table' END AS kind");
    strQuery.append(", toast.oid AS toast_table_oid, toast.relname AS toast_table");
	
	if(bVersion80_OrNewer)
	{
		strQuery.append(QString(", CASE rel.reltablespace WHEN 0 THEN %1 ELSE rel.reltablespace END AS reltablespace, ts.spcname").arg(pDatabase->oidTablespace()));
	}
		
	if(bVersion81_OrNewer) 
	{
		strQuery.append(", pg_catalog.pg_size_pretty(pg_catalog.pg_relation_size(rel.oid)) AS size_pretty");
		strQuery.append(", pg_catalog.pg_size_pretty(pg_catalog.pg_total_relation_size(rel.oid)) AS size_pretty_total");
		strQuery.append(", pg_catalog.pg_relation_size(rel.oid) AS table_size");
		strQuery.append(", pg_catalog.pg_total_relation_size(rel.oid) AS table_size_total");
	}
		
	strQuery.append(" FROM pg_catalog.pg_class rel ");
	strQuery.append("LEFT OUTER JOIN pg_catalog.pg_description des ON des.objoid=rel.oid AND des.objsubid=0 ");
	strQuery.append("LEFT OUTER JOIN pg_catalog.pg_constraint c ON c.conrelid=rel.oid AND c.contype='p' ");
	
	strQuery.append("LEFT JOIN pg_catalog.pg_class toast ON rel.reltoastrelid=toast.oid ");
	
	if(bVersion80_OrNewer)
	{	
		strQuery.append("LEFT JOIN pg_catalog.pg_tablespace ts ON CASE rel.reltablespace ");
    	strQuery.append(QString("WHEN 0 THEN %1 ELSE rel.reltablespace END=ts.oid ").arg(pDatabase->oidTablespace()));
	}	
	strQuery.append("WHERE rel.oid = " + QString("%1").arg(m_oid));

	try
	{
		pqxx::result pqxxResultTables = pDatabase->connection()->runQuery(strQuery);
		
		if(pqxxResultTables.size() != 1)
		{
            kdError() << k_funcinfo "Expect one row in result !" <<  endl;
            return;
		}
		
		setProperties(pqxxResultTables[0], bVersion80_OrNewer, bVersion81_OrNewer, pDatabase->connection());
	}
	catch (const std::exception &e)
	{
		kdError() << k_funcinfo << e.what() << endl;
        throw KPGSqlException(pDatabase->connection(), e.what(), strQuery);
	} 
}

// Set table properties
void KPGTable::setProperties(const pqxx::result::tuple & pqxxTuple, 
    bool bVersion80_OrNewer, 
	bool bVersion81_OrNewer,
 	const KPGConnection *pConnection)
{
    pqxxTuple["reltype"].to(m_oidReltype);
	m_strOwner = pqxxTuple["relowner"].c_str();
	m_strACL = pqxxTuple["relacl"].c_str();
	m_strDescription = pConnection->toUnicode(pqxxTuple["description"].c_str());
	m_strPrimaryKeyName = pqxxTuple["conname"].c_str();
	m_strPrimaryKeyColNumbers = pqxxTuple["conkey"].c_str(); 
	pqxxTuple["relhasoids"].to(m_bHasOids);
	pqxxTuple["relhassubclass"].to(m_bHasSubclass);
	pqxxTuple["reltuples"].to(m_dEstimatedRows);
	m_strKind = pqxxTuple["kind"].c_str();
	
	pqxxTuple["toast_table_oid"].to(m_oidToastTable);   
    m_strToastTable = pqxxTuple["toast_table"].c_str(); 
	
	if(m_strKind == "Toast table")
	   setPixmap(0, *m_pIconToastTable);
	
	if(!bVersion80_OrNewer)
	{
	   m_oidTablespace = 0;
	}
	else
	{   
	   pqxxTuple["reltablespace"].to(m_oidTablespace);
	   m_strTablespace = pqxxTuple["spcname"].c_str(); 
    }
    
    if(bVersion81_OrNewer)
	{
		pqxxTuple["table_size"].to(m_llSize);
		m_strPrettySize = pqxxTuple["size_pretty"].c_str();
		pqxxTuple["table_size_total"].to(m_llSizeTotal);
		m_strPrettySizeTotal = pqxxTuple["size_pretty_total"].c_str(); 
	}
}

const pqxx::result & KPGTable::resultColumns() const
{
	return m_pFolderColumns->resultColumns();
}

const pqxx::result & KPGTable::resultConstraints() const
{
	return m_pFolderConstraints->resultConstraints();
}

const pqxx::result & KPGTable::resultIndexes() const
{
	return m_pFolderIndexes->resultIndexes();
}


const pqxx::result & KPGTable::resultRules() const
{
	return m_pFolderRules->resultRules();
}


const pqxx::result & KPGTable::resultTriggers() const
{
	return m_pFolderTriggers->resultTriggers();
}

const pqxx::result KPGTable::queryStatistics() const
{
	QString strQuery("SELECT relname, seq_scan, seq_tup_read, idx_scan, idx_tup_fetch, n_tup_ins, n_tup_upd, n_tup_del FROM pg_catalog.pg_stat_all_tables WHERE relid = ");
    strQuery.append(QString("%1;").arg(m_oid));
    
    KPGConnection *pConnection = connection();
    try
	{
        return pConnection->runQuery(strQuery);
	}
	catch (const std::exception &e)
	{
		kdError() << k_funcinfo << e.what() << endl;
        throw KPGSqlException(pConnection, e.what(), strQuery);
	}
}


const pqxx::result KPGTable::queryIoStatistics() const
{
	QString strQuery("SELECT relname, heap_blks_read, heap_blks_hit, idx_blks_read, idx_blks_hit, toast_blks_read, toast_blks_hit, tidx_blks_read, tidx_blks_hit FROM \
pg_catalog.pg_statio_all_tables WHERE relid = ");
    strQuery.append(QString("%1;").arg(m_oid));
    
    KPGConnection *pConnection = connection();
    try
	{
        return pConnection->runQuery(strQuery);
	}
	catch (const std::exception &e)
	{
		kdError() << k_funcinfo << e.what() << endl;
        throw KPGSqlException(pConnection, e.what(), strQuery);
	}
}

    
