/***************************************************************************
                          mymoneybanking.cpp
                             -------------------
    begin                : Thu Aug 26 2004
    copyright            : (C) 2004 Martin Preuss
    email                : aquamaniac@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

// ----------------------------------------------------------------------------
// QT Includes

#include <qmessagebox.h>
#include <qlayout.h>

// ----------------------------------------------------------------------------
// KDE Includes

#include <klocale.h>
#include <kmessagebox.h>
#include <kgenericfactory.h>
#include <kaction.h>
#include <kglobal.h>
#include <kstandarddirs.h>
#include <kpopupmenu.h>
#include <kiconloader.h>
#include <kguiitem.h>

// ----------------------------------------------------------------------------
// Library Includes

#include <aqbanking/imexporter.h>
#include <aqbanking/jobgettransactions.h>
#include <aqbanking/job.h>
#include <qbanking/qbpickstartdate.h>
#include <qbanking/qbgui.h>
#include <gwenhywfar/logger.h>
#include <gwenhywfar/debug.h>
//#include <kbanking/settings.h>

// ----------------------------------------------------------------------------
// Project Includes

#include "mymoneybanking.h"
#include "kbjobview.h"
#include "kbsettings.h"
#include <kmymoney/mymoneyfile.h>
#include <kmymoney/kmymoneyview.h>


K_EXPORT_COMPONENT_FACTORY( kmm_kbanking,
                            KGenericFactory<KBankingPlugin>( "kmm_kbanking" ) )

KBankingPlugin::KBankingPlugin(QObject *parent, const char* name, const QStringList&) :
  KMyMoneyPlugin::OnlinePlugin(parent, name )
{
  m_kbanking=new KMyMoneyBanking(this, "KMyMoney");

  if (m_kbanking) {
    QBGui *gui;

    gui=new QBGui(m_kbanking);
    GWEN_Gui_SetGui(gui->getCInterface());
    GWEN_Logger_SetLevel(0, GWEN_LoggerLevel_Info);
    GWEN_Logger_SetLevel(AQBANKING_LOGDOMAIN, GWEN_LoggerLevel_Debug);
    if (m_kbanking->init() == 0) {
      // Tell the host application to load my GUI component
      setInstance(KGenericFactory<KBankingPlugin>::instance());
      setXMLFile("kmm_kbanking.rc");

      // create view
      createJobView();

      // create actions
      createActions();
    }
    else {
      kdWarning() << "Could not initialize KBanking online banking interface" << endl;
      delete m_kbanking;
      m_kbanking = 0;
    }
  }
}



KBankingPlugin::~KBankingPlugin()
{
  if (m_kbanking) {
    m_kbanking->fini();
    delete m_kbanking;
  }
}



void KBankingPlugin::protocols(QStringList& protocolList) const
{
  std::list<std::string> list = m_kbanking->getActiveProviders();
  std::list<std::string>::iterator it;
  for(it = list.begin(); it != list.end(); ++it)
    protocolList << (*it);
}



void KBankingPlugin::createJobView(void)
{
  KMyMoneyViewBase* view = viewInterface()->addPage(i18n("Outbox"), "onlinebanking");
  QWidget* frm = dynamic_cast<QWidget*>(view->parent());
  QWidget* w = new KBJobView(m_kbanking, view, "JobView");
  viewInterface()->addWidget(view, w);
  connect(viewInterface(), SIGNAL(viewStateChanged(bool)), frm, SLOT(setEnabled(bool)));
  connect(viewInterface(), SIGNAL(accountSelected(const MyMoneyAccount&)), this, SLOT(slotAccountSelected(const MyMoneyAccount&)));
}



void KBankingPlugin::createActions(void)
{
  new KAction(i18n("Configure Aq&Banking..."), "configure", 0, this, SLOT(slotSettings()), actionCollection(), "settings_aqbanking");
  new KAction(i18n("AqBanking importer..."), "", 0, this, SLOT(slotImport()), actionCollection(), "file_import_aqbanking");
  new KAction(i18n("Map to AqBanking account..."), "news_subscribe", 0, this, SLOT(slotAccountOnlineMap()), actionCollection(), "account_map_aqbanking");
  new KAction(i18n("Online update using AqBanking..."), "reload", 0, this, SLOT(slotAccountOnlineUpdate()), actionCollection(), "account_update_aqbanking");

  connect(viewInterface(), SIGNAL(viewStateChanged(bool)), action("file_import_aqbanking"), SLOT(setEnabled(bool)));
}

void KBankingPlugin::slotSettings(void)
{
  KBankingSettings bs(m_kbanking);
  if (bs.init())
    kdWarning() << "Error on ini of settings dialog." << endl;
  else {
    bs.exec();
    if (bs.fini())
      kdWarning() << "Error on fini of settings dialog." << endl;
  }
}



void KBankingPlugin::slotAccountOnlineMap(void)
{
  if (!m_account.id().isEmpty()) {
    MyMoneyFile* file = MyMoneyFile::instance();

    QString bankId;
    QString accountId;
    // extract some information about the bank. if we have a sortcode
    // (BLZ) we display it, otherwise the name is enough.
    try {
      const MyMoneyInstitution &bank = file->institution(m_account.institutionId());
      bankId = bank.name();
      if(!bank.sortcode().isEmpty())
        bankId = bank.sortcode();
    } catch(MyMoneyException *e) {
      // no bank assigned, we just leave the field emtpy
      delete e;
    }

    // extract account information. if we have an account number
    // we show it, otherwise the name will be displayed
    accountId = m_account.number();
    if(accountId.isEmpty())
      accountId = m_account.name();

    // do the mapping. the return value of this method is either
    // true, when the user mapped the account or false, if he
    // decided to quit the dialog. So not really a great thing
    // to present some more information.
    m_kbanking->askMapAccount(m_account.id(),
                              bankId.utf8(),
                              accountId.utf8());
  }
}



const bool KBankingPlugin::accountIsMapped(const QCString& id)
{
  AB_ACCOUNT* ab_acc;
  ab_acc = AB_Banking_GetAccountByAlias(m_kbanking->getCInterface(), id);
  return ab_acc != 0;
}



void KBankingPlugin::slotAccountOnlineUpdate(void)
{
  if (!m_account.id().isEmpty()) {
    AB_ACCOUNT *ba;
    AB_JOB *job;
    int rv;
    int days;
    int year, month, day;
    QDate qd;

    /* get AqBanking account */
    ba=AB_Banking_GetAccountByAlias(m_kbanking->getCInterface(),
				    m_account.id());
    if (!ba) {
      QMessageBox::critical(0,
			    i18n("Account Not Mapped"),
                            i18n("<qt>"
                                 "<p>"
                                 "The given application account "
                                 "has not been mapped to banking "
                                 "accounts."
                                 "</p>"
                                 "</qt>"
                            ),
			    QMessageBox::Ok,QMessageBox::NoButton);
      return;
    }

    /* create getTransactions job */
    job=AB_JobGetTransactions_new(ba);
    rv=AB_Job_CheckAvailability(job, 0);
    if (rv) {
      DBG_ERROR(0, "Job \"GetTransactions\" is not available (%d)", rv);
      QMessageBox::critical(0,
                            i18n("Job not Available"),
                            i18n("<qt>"
                                 "The update job is not supported by the "
                                 "bank/account/backend.\n"
                                 "</qt>"),
                            i18n("Dismiss"), QString::null);
      AB_Job_free(job);
      return;
    }

    days=AB_JobGetTransactions_GetMaxStoreDays(job);
    if (days>0) {
      GWEN_TIME *ti1;
      GWEN_TIME *ti2;

      ti1=GWEN_CurrentTime();
      ti2=GWEN_Time_fromSeconds(GWEN_Time_Seconds(ti1)-(60*60*24*days));
      GWEN_Time_free(ti1);
      ti1=ti2;

      if (GWEN_Time_GetBrokenDownDate(ti1, &day, &month, &year)) {
	DBG_ERROR(0, "Bad date");
	qd=QDate();
      }
      else
	qd=QDate(year, month+1, day);
      GWEN_Time_free(ti1);
    }

    // get last statement request date from application account object
    // and start from the next day if the date is valid
    QDate lastUpdate = QDate::fromString(m_account.value("lastImportedTransactionDate"), Qt::ISODate);
    if(lastUpdate.isValid())
      lastUpdate = lastUpdate.addDays(1);

    QBPickStartDate psd(m_kbanking, qd, lastUpdate, 
                        lastUpdate.isValid() ? 2 : 3, 0,
			"PickStartDate", true);
    if (psd.exec() != QDialog::Accepted) {
      AB_Job_free(job);
      return;
    }

    qd=psd.getDate();
    if (qd.isValid()) {
      GWEN_TIME *ti1;

      ti1=GWEN_Time_new(qd.year(), qd.month()-1, qd.day(), 0, 0, 0, 0);
      AB_JobGetTransactions_SetFromTime(job, ti1);
      GWEN_Time_free(ti1);
    }

    rv=m_kbanking->enqueueJob(job);
    if (rv) {
      AB_Job_free(job);
      DBG_ERROR(0, "Error %d", rv);
      QMessageBox::critical(0,
                            i18n("Error"),
                            i18n("<qt>"
                                 "Could not enqueue the job.\n"
                                 "</qt>"),
                            i18n("Dismiss"), QString::null);
      return;
    }

    // ask if the user want's to execute this job right away or spool it
    // for later execution
    KIconLoader *ic = KGlobal::iconLoader();
    KGuiItem executeButton(i18n("&Execute"),
                          QIconSet(ic->loadIcon("wizard",
                            KIcon::Small, KIcon::SizeSmall)),
                          i18n("Close this window"),
                          i18n("Use this button to close the window"));

    KGuiItem queueButton(i18n("&Queue"),
                          QIconSet(ic->loadIcon("fileexport",
                            KIcon::Small, KIcon::SizeSmall)),
                          i18n("Close this window"),
                          i18n("Use this button to close the window"));

    if(KMessageBox::questionYesNo(0,
        i18n("Do you want to execute or queue this job in the outbox?"),
        i18n("Execution"), executeButton, queueButton) == KMessageBox::Yes) {
      AB_JOB_LIST2* jobList = AB_Job_List2_new();
      AB_Job_List2_PushBack(jobList, job);

      AB_IMEXPORTER_CONTEXT *ctx;

      ctx=AB_ImExporterContext_new();
      rv=m_kbanking->executeQueue(ctx);
      if (!rv)
        m_kbanking->importContext(ctx, 0);
      else {
        DBG_ERROR(0, "Error: %d", rv);
      }
      AB_ImExporterContext_free(ctx);

      // let application emit signals to inform views
      m_kbanking->accountsUpdated();

      AB_Job_List2_free(jobList);
    }
    AB_Job_free(job);
  }
}



void KBankingPlugin::slotAccountSelected(const MyMoneyAccount& acc)
{
  MyMoneyInstitution institution;
  bool state = false;
  m_account = acc;

  action("account_map_aqbanking")->setEnabled(false);
  action("account_update_aqbanking")->setEnabled(false);

  if (!MyMoneyFile::instance()->isStandardAccount(acc.id())) {
    switch(m_account.accountGroup()) {
      case MyMoneyAccount::Asset:
      case MyMoneyAccount::Liability:
        state = true;
        break;

      default:
        break;
    }
  }

  if (state == true) {
    if (accountIsMapped(acc.id())) {
      action("account_update_aqbanking")->setEnabled(true);
    } else {
      action("account_map_aqbanking")->setEnabled(true);
    }
  }
}



void KBankingPlugin::slotImport(void)
{
  if (!m_kbanking->interactiveImport())
    kdWarning() << "Error on import dialog" << endl;
}



bool KBankingPlugin::importStatement(MyMoneyStatement& s)
{
  return statementInterface()->import(s);
}






KMyMoneyBanking::KMyMoneyBanking(KBankingPlugin* parent, const char* appname, const char* fname)
:KBanking(appname, fname)
,m_parent(parent)
{
}



const AB_ACCOUNT_STATUS* KMyMoneyBanking::_getAccountStatus(AB_IMEXPORTER_ACCOUNTINFO *ai)
{
  const AB_ACCOUNT_STATUS *ast;
  const AB_ACCOUNT_STATUS *best;

  best=0;
  ast=AB_ImExporterAccountInfo_GetFirstAccountStatus(ai);
  while(ast) {
    if (!best)
      best=ast;
    else {
      const GWEN_TIME *tiBest;
      const GWEN_TIME *ti;

      tiBest=AB_AccountStatus_GetTime(best);
      ti=AB_AccountStatus_GetTime(ast);

      if (!tiBest) {
	best=ast;
      }
      else {
        if (ti) {
          double d;

          /* we have two times, compare them */
          d=GWEN_Time_Diff(ti, tiBest);
          if (d>0)
            /* newer */
            best=ast;
        }
      }
    }
    ast=AB_ImExporterAccountInfo_GetNextAccountStatus(ai);
  } /* while */
  return best;
}



void KMyMoneyBanking::_xaToStatement(const AB_TRANSACTION *t,
                                     MyMoneyStatement &ks)
{
  const GWEN_STRINGLIST *sl;
  QString s;
  const char *p;
  const AB_VALUE *val;
  const GWEN_TIME *ti;
  const GWEN_TIME *startTime=0;
  MyMoneyStatement::Transaction kt;

  kt.m_fees = MyMoneyMoney();

  // bank's transaction id
  p=AB_Transaction_GetFiId(t);
  if (p)
    kt.m_strBankID=QString("ID ")+QString::fromUtf8(p);

  // payee
  s.truncate(0);
  sl=AB_Transaction_GetRemoteName(t);
  if (sl) {
    GWEN_STRINGLISTENTRY *se;

    se=GWEN_StringList_FirstEntry(sl);
    if (se) {
      p=GWEN_StringListEntry_Data(se);
      assert(p);
      s = QString::fromUtf8(p);
    }
  }
  kt.m_strPayee=s;

  // memo
  s.truncate(0);
  sl=AB_Transaction_GetPurpose(t);
  if (sl) {
    GWEN_STRINGLISTENTRY *se;

    se=GWEN_StringList_FirstEntry(sl);
    while (se) {
      p = GWEN_StringListEntry_Data(se);
      assert(p);
      if (!s.isEmpty())
        s += " ";
      s += QString::fromUtf8(p);
      se = GWEN_StringListEntry_Next(se);
    } // while
  }
  kt.m_strMemo = s;

  // date
  ti=AB_Transaction_GetDate(t);
  if (!ti)
    ti=AB_Transaction_GetValutaDate(t);
  if (ti) {
    int year, month, day;

    if (!startTime)
      startTime=ti;
    else {
      if (GWEN_Time_Diff(ti, startTime)<0)
        startTime=ti;
    }

    if (!GWEN_Time_GetBrokenDownDate(ti, &day, &month, &year))
      kt.m_datePosted=QDate(year, month+1, day);
  } else {
    DBG_WARN(0, "No date for transaction");
  }

  // value
  val=AB_Transaction_GetValue(t);
  if (val) {
    if (ks.m_strCurrency.isEmpty()) {
      p=AB_Value_GetCurrency(val);
      if (p)
	ks.m_strCurrency=p;
    }
    else {
      p=AB_Value_GetCurrency(val);
      if (p)
	s=p;
      if (ks.m_strCurrency.lower()!=s.lower()) {
	// TODO: handle currency difference
	DBG_ERROR(0, "Mixed currencies currently not allowed");
      }
    }

    kt.m_amount=MyMoneyMoney(AB_Value_GetValueAsDouble(val));
  }
  else {
    DBG_WARN(0, "No value for transaction");
  }

  if (startTime) {
    int year, month, day;

    if (!GWEN_Time_GetBrokenDownDate(startTime, &day, &month, &year)) {
      QDate d(year, month+1, day);

      if (!ks.m_dateBegin.isValid())
        ks.m_dateBegin=d;
      else if (d<ks.m_dateBegin)
        ks.m_dateBegin=d;

      if (!ks.m_dateEnd.isValid())
        ks.m_dateEnd=d;
      else if (d>ks.m_dateEnd)
        ks.m_dateEnd=d;
    }
  }
  else {
    DBG_WARN(0, "No date in current transaction");
  }

  // store transaction
  DBG_INFO(0, "Adding transaction");
  ks.m_listTransactions+=kt;
}



bool KMyMoneyBanking::importAccountInfo(AB_IMEXPORTER_ACCOUNTINFO *ai,
					uint32_t /*flags*/)
{
  QString s;
  const char *p;
  const AB_TRANSACTION *t;
  MyMoneyStatement ks;
  const AB_ACCOUNT_STATUS *ast;
  const AB_VALUE *val;
  const GWEN_TIME *ti;

  DBG_INFO(0, "Importing account...");

  // account number
  p=AB_ImExporterAccountInfo_GetAccountNumber(ai);
  if (p)
    ks.m_strAccountNumber=p;

  // account name
  p=AB_ImExporterAccountInfo_GetAccountName(ai);
  if (p)
    ks.m_strAccountName=p;

  // account type
  switch(AB_ImExporterAccountInfo_GetType(ai)) {
  case AB_AccountType_Bank:
    ks.m_eType=MyMoneyStatement::etSavings;
    break;
  case AB_AccountType_CreditCard:
    ks.m_eType=MyMoneyStatement::etCreditCard;
    break;
  case AB_AccountType_Checking:
    ks.m_eType=MyMoneyStatement::etCheckings;
    break;
  case AB_AccountType_Savings:
    ks.m_eType=MyMoneyStatement::etSavings;
    break;
  case AB_AccountType_Investment:
    ks.m_eType=MyMoneyStatement::etInvestment;
    break;
  case AB_AccountType_Cash:
    ks.m_eType=MyMoneyStatement::etNone;
    break;
  default:
    ks.m_eType=MyMoneyStatement::etNone;
  }

  // account status
  ast=_getAccountStatus(ai);
  if (ast) {
    const AB_BALANCE *bal;

    bal=AB_AccountStatus_GetBookedBalance(ast);
    if (!bal)
      bal=AB_AccountStatus_GetNotedBalance(ast);
    if (bal) {
      val=AB_Balance_GetValue(bal);
      if (val) {
	DBG_INFO(0, "Importing balance");
        ks.m_closingBalance=MyMoneyMoney(AB_Value_GetValueAsDouble(val));
        p=AB_Value_GetCurrency(val);
        if (p)
          ks.m_strCurrency=p;
      }
      ti=AB_Balance_GetTime(bal);
      if (ti) {
        int year, month, day;

        if (!GWEN_Time_GetBrokenDownDate(ti, &day, &month, &year))
          ks.m_dateEnd=QDate(year, month+1, day);
      }
      else {
        DBG_WARN(0, "No time for balance");
      }
    }
    else {
      DBG_WARN(0, "No account balance");
    }
  }
  else {
    DBG_WARN(0, "No account status");
  }

  // get all transactions
  t=AB_ImExporterAccountInfo_GetFirstTransaction(ai);
  while(t) {
    _xaToStatement(t, ks);
    t=AB_ImExporterAccountInfo_GetNextTransaction(ai);
  }

  // import them
  if (!m_parent->importStatement(ks)) {
    if (QMessageBox::critical(0,
                              i18n("Critical Error"),
                              i18n("Error importing statement."),
                              i18n("Continue"),
                              i18n("Abort"), 0, 0)!=0) {
      DBG_ERROR(0, "User aborted");
      return false;
    }
  }
  return true;
}


#include "mymoneybanking.moc"
