/**************************************************************************/
/*  Klavaro - a flexible touch typing tutor                               */
/*  Copyright (C) 2005 - 2008  Felipe Castro                              */
/*                                                                        */
/*  This program is free software, licensed under the terms of the GNU    */
/*  General Public License as published by the Free Software Foundation,  */
/*  which is also included in this package, in a file named "COPYING".    */
/**************************************************************************/

/*
 * Charts management
 */
#include <stdio.h>
#include <string.h>
#include <locale.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>

#include "support.h"
#include "main.h"
#include "tutor.h"
#include "basic.h"
#include "gtkdatabox.h"
#include "plot.h"

# define PLOT_SCORE "#60C0F0"

GtkWidget *window_stat_ = NULL;

struct
{
	gfloat y_data[DATA_POINTS];
	gfloat x_data[DATA_POINTS];
} plot;

/*******************************************************************************
 * Functions to manage the plottings on the progress window
 */
void
plot_initialize (GtkWidget * widget)
{
	gint i;
	gint index;
	GdkColor color;
	GtkWidget *databox;

	/*
	 * Initialize X data
	 */
	for (i = 0; i < DATA_POINTS; i++)
		plot.x_data[i] = i + 1;

	/*
	 * Data Box
	 */
	databox = gtk_databox_new ();
	gtk_widget_set_name (databox, "databox_stat");
	gtk_widget_ref (databox);
	gtk_object_set_data_full (GTK_OBJECT (window_stat_), "databox_stat",
				  databox, (GtkDestroyNotify) gtk_widget_unref);
	gtk_container_add (GTK_CONTAINER (widget), databox);
	gtk_widget_show (databox);

	/*
	 * Grid lines
	 */
	gdk_color_parse (TUTOR_YELLOW_GREEN, &color);
	index = gtk_databox_data_add_x_y (GTK_DATABOX (databox), DATA_POINTS,
					  plot.x_data, plot.y_data, color, GTK_DATABOX_GRID, 1);
	gtk_databox_data_set_grid_config (GTK_DATABOX (databox), index, 9, 10);

	/*
	 * Background color
	 */
	gdk_color_parse (TUTOR_WHITE, &color);
	gtk_databox_set_background_color (GTK_DATABOX (databox), color);

	plot_draw_chart (1);
}

/**********************************************************************
 * Plots the statistics
 */
void
plot_draw_chart (gint field)
{
	gint i;
	gint index;
	guint64 aux;
	gchar *tmp;
	gchar *tmp_locale;
	gchar *tmp_name;
	gchar tmp_str[2000];
	FILE *fh;
	gfloat accur[DATA_POINTS];
	gfloat velo[DATA_POINTS];
	gfloat fluid[DATA_POINTS];
	gfloat score[DATA_POINTS];
	gchar data[DATA_POINTS][20];
	gchar hour[DATA_POINTS][20];
	gint nchars[DATA_POINTS];
	gchar lesson[DATA_POINTS][299];
	GdkColor color;
	GtkWidget *wg;
	GtkDataboxValue max = { 0, 0 };
	GtkDataboxValue min = { 0, 0 };

	/* Blank the chart
	 */
	wg = lookup_widget (window_stat_, "databox_stat");
	gtk_databox_data_remove_all (GTK_DATABOX (wg));
	gtk_databox_redraw (GTK_DATABOX (wg));

	/*
	 * Formats the charts
	 */
	min.x = 0;
	min.y = 0;
	max.x = DATA_POINTS + 5;
	max.y = 100;
	wg = lookup_widget (window_stat_, "label_stat_unit");
	switch (field)
	{
	case 1:
		min.y = 80;
		gtk_label_set_text (GTK_LABEL (wg), _("Accuracy (%)"));
		break;

	case 2:
		gtk_label_set_text (GTK_LABEL (wg), _("Velocity (WPM)"));
		break;

	case 3:
		gtk_label_set_text (GTK_LABEL (wg), _("Fluidness (%)"));
		break;
	case 4:
		max.y = 10;
		tmp = g_strconcat (_("Score"), " (0-10)", NULL);
		gtk_label_set_text (GTK_LABEL (wg), tmp);
		g_free (tmp);
		break;
	}
	wg = lookup_widget (window_stat_, "databox_stat");
	gtk_databox_rescale_with_values (GTK_DATABOX (wg), min, max);

	/*
	 * Auxiliar variable to track the current lesson
	 */
	aux = basic_get_lesson ();
	aux -= basic_get_lesson_increased ()? 1 : 0;
	if (aux == 0)
		aux++;
	sprintf (tmp_str, "%d", (gint) aux);
	gtk_label_set_text (GTK_LABEL (lookup_widget (window_stat_, "label_stat_lesson_n")),
			    tmp_str);

	/*
	 * Get the file name
	 */
	if (field != 4)
		tmp_name =
			g_strconcat (main_get_user_dir (), "stat_", tutor_get_type_name (), ".txt",
				     NULL);
	else
		tmp_name = g_strconcat (main_get_user_dir (), "scores_fluid.txt", NULL);

	/*
	 * Open the data file
	 */
	fh = (FILE *) g_fopen (tmp_name, "r");
	if (!fh)
	{
		g_message ("no data yet, no statistic file:\n%s", tmp_name);
		g_free (tmp_name);
		return;
	}

	/*
	 * Reads the first line (header)
	 */
	if (!fgets (tmp_str, 2000, fh))
	{
		g_message ("no data on the statistic file:\n%s", tmp_name);
		g_free (tmp_name);
		fclose (fh);
		return;
	}

	/*
	 * Changing to "C" locale
	 */
	tmp_locale = g_strdup (setlocale (LC_NUMERIC, NULL));
	if (tmp_locale != NULL)
		setlocale (LC_NUMERIC, "C");

	/*
	 * Reads the first DATA_POINTS points
	 */
	for (i = 0; i < DATA_POINTS; i++)
		plot.y_data[i] = -1000;

	i = 0;
	while (1)
	{
		gint itens;

		if (field < 4)
			itens = fscanf (fh, "%f%f%f%s%s", &accur[i], &velo[i], &fluid[i], data[i],
					hour[i]);
		else
			itens = fscanf (fh, "%f%s%s%i", &score[i], data[i], hour[i], &nchars[i]);

		if (itens != 5 && field < 4)
			break;
		else if (itens != 4 && field == 4)
			break;

		if (fgets (lesson[i], 299, fh) == NULL)
			break;

		if (strlen (lesson[i]) == 299)
		{
			if (field < 4)
				g_warning
					("file name of custom lesson too long in line %i of %s: "
					 "the limit is 299 characters.", i, tmp_name);
			else
				g_warning
					("language name too long in line %i of %s: "
					 "the limit is 299 characters.", i, tmp_name);
		}

		if (accur[i] > 100)
			accur[i] = 100;
		if (accur[i] < 80)
			accur[i] = 80;
		if (velo[i] > 100)
			velo[i] = 100;
		if (velo[i] < 0)
			velo[i] = 0;
		if (fluid[i] > 100)
			fluid[i] = 100;
		if (fluid[i] < 0)
			fluid[i] = 0;
		if (score[i] < 0)
			score[i] = 0;

		if (tutor_get_type () == TT_BASIC && g_ascii_strtoull (lesson[i], NULL, 10) != aux)
			continue;
		switch (field)
		{
		case 1:
			plot.y_data[i] = accur[i];
			break;
		case 2:
			plot.y_data[i] = velo[i];
			break;
		case 3:
			plot.y_data[i] = fluid[i];
			break;
		case 4:
			plot.y_data[i] = score[i];
			break;
		}
		if (++i == DATA_POINTS)
			break;
	}

	/*
	 * Reads until the end, keeping the last DATA_POINTS points 
	 */
	while (1)
	{
		gint itens;

		if (field < 4)
			itens = fscanf (fh, "%f%f%f%s%s", &accur[i], &velo[i], &fluid[i], data[i],
					hour[i]);
		else
			itens = fscanf (fh, "%f%s%s%i", &score[i], data[i], hour[i], &nchars[i]);

		if (itens != 5 && field < 4)
			break;
		else if (itens != 4 && field == 4)
			break;

		if (fgets (lesson[i], 299, fh) == NULL)
			break;
		if (strlen (lesson[i]) == 299)
		{
			if (field < 4)
				g_warning
					("file name of custom lesson too long in line %i of %s: "
					 "the limit is 299 characters.", i, tmp_name);
			else
				g_warning
					("language name too long in line %i of %s: "
					 "the limit is 299 characters.", i, tmp_name);
		}

		if (accur[i] > 100)
			accur[i] = 100;
		if (accur[i] < 80)
			accur[i] = 80;
		if (velo[i] > 100)
			velo[i] = 100;
		if (velo[i] < 0)
			velo[i] = 0;
		if (fluid[i] > 100)
			fluid[i] = 100;
		if (fluid[i] < 0)
			fluid[i] = 0;
		if (score[i] < 0)
			score[i] = 0;

		if (tutor_get_type () == TT_BASIC && g_ascii_strtoull (lesson[i], NULL, 10) != aux)
			continue;
		for (i = 0; i < DATA_POINTS - 1; i++)
			plot.y_data[i] = plot.y_data[i + 1];
		switch (field)
		{
		case 1:
			plot.y_data[i] = accur[i + 1];
			break;
		case 2:
			plot.y_data[i] = velo[i + 1];
			break;
		case 3:
			plot.y_data[i] = fluid[i + 1];
			break;
		case 4:
			plot.y_data[i] = score[i + 1];
			break;
		}
		i++;
	}
	fclose (fh);
	g_free (tmp_name);

	/*
	 * Coming back to the right locale
	 */
	if (tmp_locale != NULL)
		setlocale (LC_NUMERIC, tmp_locale);
	g_free (tmp_locale);

	if (i == 0)
	{
		g_message ("no valid data to plot.");
		return;
	}

	/*
	 * Replot the data
	 */
	wg = lookup_widget (window_stat_, "databox_stat");
	if (field < 4)
		gdk_color_parse (TUTOR_GREEN, &color);
	else
		gdk_color_parse (PLOT_SCORE, &color);
	gtk_databox_data_add_x_y (GTK_DATABOX (wg), i, plot.x_data, plot.y_data, color,
				  GTK_DATABOX_POINTS, 3);
	gdk_color_parse (TUTOR_BLACK, &color);
	gtk_databox_data_add_x_y (GTK_DATABOX (wg), i, plot.x_data, plot.y_data, color,
				  GTK_DATABOX_POINTS, 5);
	gtk_databox_data_add_x_y (GTK_DATABOX (wg), i, plot.x_data, plot.y_data, color,
				  GTK_DATABOX_LINES, 1);

	gdk_color_parse (TUTOR_YELLOW_GREEN, &color);
	index = gtk_databox_data_add_x_y (GTK_DATABOX (wg), i, plot.x_data, plot.y_data, color,
					  GTK_DATABOX_GRID, 1);
	gtk_databox_data_set_grid_config (GTK_DATABOX (wg), index, 9, 10);

	/*
	 * Redraw the plot
	 */
	gtk_databox_redraw (GTK_DATABOX (wg));
}
