#include <stdlib.h>
#include <string.h>

#include <glib-object.h>
#include <glib.h>

#include "kpcalendartime.h"

static void       kp_calendar_time_class_init         (GObjectClass *klass,
                                                       gpointer data);
static void       kp_calendar_time_instance_init      (GObject *object,
                                                       gpointer data);
static void       kp_calendar_time_instance_finalize  (GObject *object);
/*static gboolean   is_valid_time_str                   (const gchar *time_str);*/
static gboolean   is_valid_datetime_str               (const gchar *date);

GType
kp_calendar_time_get_type ()
{
  static GType kp_calendar_time_type = 0;

  if (!kp_calendar_time_type) {
    static const GTypeInfo kp_calendar_time_info = {
      sizeof (KPCalendarTimeClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc) kp_calendar_time_class_init,
      (GClassFinalizeFunc) NULL,
      NULL,
      sizeof (KPCalendarTime),
      0,
      (GInstanceInitFunc) kp_calendar_time_instance_init,
      NULL
    };

    kp_calendar_time_type = g_type_register_static (G_TYPE_OBJECT, "KPCalendarTime",
                                                 &kp_calendar_time_info, 0);
  }
  return kp_calendar_time_type;
}

static void
kp_calendar_time_class_init (GObjectClass * klass,
                          gpointer data)
{
  GObjectClass *object_class;
  
  object_class = G_OBJECT_CLASS (klass);
  object_class->finalize = kp_calendar_time_instance_finalize;
}

static void
kp_calendar_time_instance_init (GObject * object,
                             gpointer data)
{
  KP_CALENDAR_TIME (object)->date = NULL;
  KP_CALENDAR_TIME (object)->h = 0;
  KP_CALENDAR_TIME (object)->m = 0;
  KP_CALENDAR_TIME (object)->s = 0;
  KP_CALENDAR_TIME (object)->t = 0;
}

static void
kp_calendar_time_instance_finalize (GObject * object)
{
  GObjectClass *parent_class;

  if (KP_CALENDAR_TIME (object)->date)
    g_free (KP_CALENDAR_TIME (object)->date);
  
  parent_class = g_type_class_peek_parent (G_OBJECT_GET_CLASS (object));
  parent_class->finalize (object);
}

KPCalendarTime *
kp_calendar_time_new (void)
{
  return g_object_new (kp_calendar_time_get_type (), NULL);
}

KPCalendarTime *
kp_calendar_time_new_dmyhms (guint d,
                          guint m,
                          guint y,
                          gint h,
                          gint min,
                          gint s)
{
  KPCalendarTime *ct;

  ct = g_object_new (kp_calendar_time_get_type (), NULL);

  kp_calendar_time_set_dmy (ct, d, m, y);

  if (h >= 0 && min >= 0 && s >= 0)
    kp_calendar_time_set_hmst (ct, h, min, s, 0);

  return ct;
}

gboolean
kp_calendar_time_set_dmy (KPCalendarTime * ct,
                       guint d,
                       guint m,
                       guint y)
{
  g_return_val_if_fail (KP_IS_CALENDAR_TIME (ct), FALSE);

  ct->date = g_date_new_dmy (d, m, y);
  g_return_val_if_fail (g_date_valid (ct->date) != FALSE, FALSE);

  return TRUE;
}

gboolean
kp_calendar_time_set_hmst (KPCalendarTime *ct, guint h, guint m, guint s,
                           guint t)
{
  g_return_val_if_fail (KP_IS_CALENDAR_TIME (ct), FALSE);
  g_return_val_if_fail (h < 24, FALSE);
  g_return_val_if_fail (m < 60, FALSE);
  g_return_val_if_fail (s < 60, FALSE);
  g_return_val_if_fail (t < 1000, FALSE);

  ct->h = h;
  ct->m = m;
  ct->s = s;
  ct->t = t;

  return TRUE;
}

/**
 * kp_calendar_time_set_hms_str:
 * @ct: a KPCalendarTime
 * @time: string which contains time in format %.2u:%.2u:%.2u (hh:mm:ss) or 
 * %.2u:%.2u:%.2u.%3u (hh:mm:ss.ddd)
 *
 * Returns: TRUE if successful and FALSE otherwise.
 */
gboolean
kp_calendar_time_set_hmst_str (KPCalendarTime *ct,
                               const gchar *time)
{
  gchar *str;
  guint len;
  guint h;
  guint m;
  guint s;
  guint t;

  g_return_val_if_fail (time != NULL, FALSE);
  g_return_val_if_fail (KP_IS_CALENDAR_TIME (ct), FALSE);

  len = g_utf8_strlen (time, -1);

  if (len != 8 && len != 12) {
    g_warning ("Time string is not either 8 or 12 characters long! (%s)", time);
    return FALSE;
  }

  str = g_strdup (time);

  str[2] = str[5] = '\0';

  h = strtod (&str[0], NULL);
  m = strtod (&str[3], NULL);
  s = strtod (&str[6], NULL);

  if (len == 12) {
    str[7] = '\0';
    t = strtod (&str[9], NULL);
  }
  else
    t = 0;

  kp_calendar_time_set_hmst (ct, h, m, s, t);
  free (str);

  return TRUE;
}

gboolean
kp_calendar_time_set_date (KPCalendarTime * ct,
                        GDate * date)
{
  g_return_val_if_fail (g_date_valid (date), FALSE);
  ct->date = date;
  return TRUE;
}

gboolean
kp_calendar_time_set_datetime (KPCalendarTime *ct, const gchar *datetime)
{
  guint d, m, y;
  gchar *date;

  if (is_valid_datetime_str (datetime) == FALSE)
    return FALSE;

  date = g_strdup (datetime);
  date[4] = date[7] = date[10] = date[13] = date[16] = '\0';

  y = (guint) strtod (&date[0], NULL);
  m = (guint) strtod (&date[5], NULL);
  d = (guint) strtod (&date[8], NULL);

  ct->date = g_date_new_dmy (d, m, y);
  g_return_val_if_fail (g_date_valid (ct->date) != FALSE, FALSE);

  ct->h = (guint) strtod (&date[11], NULL);
  ct->m = (guint) strtod (&date[14], NULL);
  ct->s = (guint) strtod (&date[17], NULL);
  g_free (date);

  return TRUE;
}

GDate *
kp_calendar_time_get_date (KPCalendarTime *ct)
{
  g_return_val_if_fail (ct != NULL, NULL);
  g_return_val_if_fail (ct->date != NULL, NULL);

  return ct->date;
}

/*static gboolean
is_valid_time_str (const gchar *time_str)
{
  gchar **str;
  guint i;

  str = g_strsplit (time_str, ":", 4);

  if (str[0] == NULL)
    return FALSE;

  if (!g_unichar_isdigit (str[0][0]))
    goto error;

  for (i = 0; str[i] != NULL && i < 4; i++) {
    if (g_utf8_strlen (str[i], -1) > 2)
      goto error;
    if (!g_unichar_isdigit (str[i][0]))
      goto error;
    if (str[i][1] != '\0' && !g_unichar_isdigit (str[i][1]))
      goto error;
  }
  if (i != 3)
    goto error;
  return TRUE;

error:
  g_strfreev (str);
  return FALSE;
}*/

/**
 * is_valid_xml_date:
 * @date: date-string in format: CCYY-MM-DDThh:mm:ss.
 *
 * Checks if syntax of date is valid.
 **/
static gboolean
is_valid_datetime_str (const gchar * date)
{
  gboolean val = TRUE;
  guint i;

  if (strlen (date) < 19)
    val = 0;
  else {
    if (date[4] != '-' || date[7] != '-' || date[10] != 'T'
        || date[13] != ':' || date[16] != ':') {
      val = 0;
    }
  }

  for (i = 0; i < 19; i++)
    if (i != 4 && i != 7 && i != 10 && i != 13 && i != 16)
      if (!g_unichar_isdigit (date[i])) {
        val = 0;
        break;
      }

  if (!val) {
    g_warning ("Invalid date-format: %s", date);
    return FALSE;
  }

  return TRUE;
}

gchar *
kp_calendar_time_to_str_hmst (KPCalendarTime *ct)
{
  g_return_val_if_fail (ct != NULL, NULL);
  g_return_val_if_fail (KP_IS_CALENDAR_TIME (ct), NULL);

  return g_strdup_printf ("%.2u:%.2u:%.2u.%.3u", ct->h, ct->m, ct->s, ct->t);
}
