/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  $Id: kz-xml.h 3752 2009-02-22 09:46:16Z ikezoe $
 */

#ifndef KZ_XML_H
#define KZ_XML_H

#include <glib-object.h>

G_BEGIN_DECLS

#define KZ_TYPE_XML		(kz_xml_get_type ())
#define KZ_XML(obj)		(G_TYPE_CHECK_INSTANCE_CAST ((obj), KZ_TYPE_XML, KzXML))
#define KZ_XML_CLASS(klass)	(G_TYPE_CHECK_CLASS_CAST ((klass),  KZ_TYPE_XML, KzXMLClass))
#define KZ_IS_XML(obj)		(G_TYPE_CHECK_INSTANCE_TYPE ((obj), KZ_TYPE_XML))
#define KZ_IS_XML_CLASS(klass)	(G_TYPE_CHECK_CLASS_TYPE ((klass),  KZ_TYPE_XML))
#define KZ_XML_GET_CLASS(obj)	(G_TYPE_INSTANCE_GET_CLASS ((obj),  KZ_TYPE_XML, KzXMLClass))

typedef struct _KzXML        KzXML;
typedef struct _KzXMLClass   KzXMLClass;
typedef struct _KzXMLNode    KzXMLNode;
typedef struct _KzXMLElement KzXMLElement;
typedef struct _KzXMLAttr    KzXMLAttr;

typedef enum
{
	KZ_XML_NODE_INVALID,
	KZ_XML_NODE_DOC_ROOT,
	KZ_XML_NODE_XML_DECL,
	KZ_XML_NODE_DOCTYPE,
	KZ_XML_NODE_ELEMENT,
	KZ_XML_NODE_TEXT,
	KZ_XML_NODE_COMMENT,
	KZ_XML_NODE_PI,
	KZ_XML_NODE_CDATA,
	KZ_XML_NODE_OTHER,
	KZ_XML_N_NODE_TYPES
} KzXMLNodeType;

struct _KzXML
{
	GObject parent;

	gchar *file;
	gchar *dtd;
	gchar *encoding;

	KzXMLNode *root;
};

struct _KzXMLClass
{
	GObjectClass parent_class;
};

struct _KzXMLNode
{
	KzXMLNodeType  type;
	gpointer       content;
	KzXMLNode     *parent;
	GList         *children;
	gint           ref_count;
};

struct _KzXMLElement
{
	gchar *name;
	GList *attrs;
};

struct _KzXMLAttr
{
	gchar *name;
	gchar *value;
};

/* XML file */
GType        kz_xml_get_type               (void) G_GNUC_CONST;
KzXML       *kz_xml_new                    (void);
gboolean     kz_xml_load                   (KzXML         *xml,
					    const gchar   *filename);
gboolean     kz_xml_save                   (KzXML         *xml,
					    const gchar   *filename);

/* parsing XML */
gboolean     kz_xml_load_xml               (KzXML         *xml,
					    const gchar   *buffer,
					    guint          length);

/*  */
void         kz_xml_clear_content          (KzXML         *xml);

/* node */
KzXMLNode   *kz_xml_node_new               (KzXML         *xml,
					    KzXMLNodeType  type);
KzXMLNode   *kz_xml_get_root_element       (KzXML         *xml);
void         kz_xml_set_root_element       (KzXML         *xml,
					    KzXMLNode     *node);
KzXMLNode   *kz_xml_node_ref               (KzXMLNode     *node);
void         kz_xml_node_unref             (KzXMLNode     *node);
gboolean     kz_xml_node_is_element        (KzXMLNode     *node);
KzXMLNode   *kz_xml_node_next              (KzXMLNode     *node);
KzXMLNode   *kz_xml_node_prev              (KzXMLNode     *node);
KzXMLNode   *kz_xml_node_parent            (KzXMLNode     *node);
KzXMLNode   *kz_xml_node_first_child       (KzXMLNode     *node);
KzXMLNode   *kz_xml_node_last_child        (KzXMLNode     *node);
gchar       *kz_xml_node_to_xml            (KzXMLNode     *node);
gchar       *kz_xml_node_to_str            (KzXMLNode     *node);

/* for element node */
KzXMLNode   *kz_xml_element_node_new       (const gchar   *name);
const gchar *kz_xml_node_name              (KzXMLNode     *node);
gboolean     kz_xml_node_name_is           (KzXMLNode     *node,
					    const gchar   *name);
gboolean     kz_xml_node_has_attr          (KzXMLNode     *node,
					    const gchar   *attr_name);
const gchar *kz_xml_node_get_attr          (KzXMLNode     *node,
					    const gchar   *attr_name);
void         kz_xml_node_set_attr          (KzXMLNode     *node,
					    const gchar   *attr_name,
					    const gchar   *attr_value);
const GList *kz_xml_node_get_attrs         (KzXMLNode     *node);

void         kz_xml_node_append_child      (KzXMLNode     *node,
					    KzXMLNode     *child);
void         kz_xml_node_insert_before     (KzXMLNode     *node,
					    KzXMLNode     *child,
					    KzXMLNode     *sibling);
KzXMLNode   *kz_xml_node_remove_child      (KzXMLNode     *node,
					    KzXMLNode     *child);
KzXMLNode   *kz_xml_node_replace_child     (KzXMLNode     *node,
					    KzXMLNode     *new_child,
					    KzXMLNode     *old_child);

/* for text node */
KzXMLNode   *kz_xml_text_node_new          (const gchar *text);
gboolean     kz_xml_node_is_text           (KzXMLNode   *node);
void         kz_xml_text_node_replace_text (KzXMLNode   *node,
					    const gchar *text);
gboolean     kz_xml_node_is_space          (KzXMLNode   *node);

gboolean     kz_xml_node_remove_trailing_blank_line
					   (KzXMLNode   *node);
void         kz_xml_node_arrange_indent    (KzXMLNode   *parent,
					    guint        indent_level);

G_END_DECLS

#endif /* KZ_XML_H */
