/**
 * SJIS2EUCJPOutputStream.java
 *
 * Copyright (c) 2006 Ying-Chun Liu (PaulLiu)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *  1. Redistributions of source code must retain the copyright notice,
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in
 *     the documentation and/or other materials provided with the
 *     distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND
 * CONTRIBUTORS ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

package org.jlhafrontend;

import java.io.*;

/**
 * This class re-encodes SHIFT-JIS to EUC-JP when output
 *
 */

public class SJIS2EUCJPOutputStream extends java.io.FilterOutputStream {

    private boolean lastByteFlag = false;
    private int lastByte = 0;
    
    /** 
     * Creates an output stream built on top of the specified underlying
     * output stream.
     *
     * @param out the underlying output stream to be assigned
     */
    public SJIS2EUCJPOutputStream(java.io.OutputStream out) {
	super(out);
    }

    /**
     * Writes the specified byte to the output stream 
     *
     * @param b the byte
     */
    public void write(int b) throws java.io.IOException {
        b = b & 0x00ff;
	if (!lastByteFlag) {
	    if (b <= 0x80 || b > 0xfc) { /* US-ASCII compatible */
		super.write(b);
		return;
	    } else if (0xa0 <= b && b < 0xe0) { /* kana */
		super.write(0x8e);
		super.write(b);
		return;
	    } else {
		lastByteFlag=true;
		lastByte = b;
		return;
	    }
	} else {
	    lastByteFlag=false;
	    if (lastByte >= 0xa0) {
		lastByte -= 0x40;
	    }
	    lastByte = (lastByte-0x81)*2+0x21;
	    if (b >= 0x9f) {
		b = b - 0x9f + 0x21;
		lastByte++;
	    } else {
		if (b >= 0x7f) {
		    b--;
		}
		b -= 0x1f;
	    }
	    b = b & 0xff;
	    lastByte = lastByte & 0xff;
	    b = b | 0x80;
	    lastByte = lastByte | 0x80;
	    super.write(lastByte);
	    lastByte=0;
	    super.write(b);
	}	    
    }

    /**
     * Writes len bytes from the specified byte array starting at 
     * offset off to this output stream.
     *
     * @param b the data
     * @param off the start offset
     * @param len the number of bytes to write
     */
    public void write(byte[] b, int off, int len) throws java.io.IOException{
	int i;
	for (i=0 ; i<len && (off+i)<b.length ; i++) {
	    this.write(b[(off+i)]);
	}
    }

    /**
     * Writes b.length bytes to this output stream.
     * 
     * @param b the data
     */
    public void write(byte[] b) throws java.io.IOException {
	this.write(b,0,b.length);
    }

    /**
     * Test function 
     */
    public static void main (String[] args) {
	int bufRead;
	byte[] buf = new byte[1024];
	InputStream in = System.in;
	OutputStream out = new SJIS2EUCJPOutputStream(System.out);
	while (true) {
	    try {
		bufRead=in.read(buf);
	    } catch (Exception e) {
		break;
	    }
	    if (bufRead<=0) break;
	    try {
		out.write(buf,0,bufRead);
	    } catch (Exception e) {
		break;
	    }
	}
	try {
	    in.close();
	} catch (Exception e) {
	}
	try {
	    out.close();
	} catch (Exception e) {
	}
    }

}

