;;;; octet-stream.lisp -- like string-streams, but with (VECTOR (UNSIGNED-BYTE 8))

(in-package :crypto)


;;; portability definitions

(eval-when (:compile-toplevel :load-toplevel :execute)
(defvar *binary-input-stream-class*
  (quote
   #+sbcl sb-gray:fundamental-binary-input-stream
   #+openmcl ccl:fundamental-binary-input-stream
   #-(or sbcl openmcl) (error "octet streams not supported in this implementation")))

(defvar *binary-output-stream-class*
  (quote
   #+sbcl sb-gray:fundamental-binary-output-stream
   #+openmcl ccl:fundamental-binary-output-stream
   #-(or sbcl openmcl) (error "octet streams not supported in this implementation")))

(defvar *stream-element-type-function*
  (quote
   #+sbcl sb-gray::stream-element-type
   #+openmcl cl:stream-element-type
   #-(or sbcl openmcl) (error "octet streams not supported in this implementation")))

(defvar *stream-read-byte-function*
  (quote
   #+sbcl sb-gray:stream-read-byte
   #+openmcl ccl:stream-read-byte
   #-(or sbcl openmcl) (error "octet streams not supported in this implementation")))

(defvar *stream-write-byte-function*
  (quote
   #+sbcl sb-gray:stream-write-byte
   #+openmcl ccl:stream-write-byte
   #-(or sbcl openmcl) (error "octet streams not supported in this implementation")))

;;; FIXME: would be nice to support STREAM-{READ,WRITE}-SEQUENCE, too.  The
;;; function name hacking is here, but the actual implementation (and
;;; possible arglist headaches) are not.
(defvar *stream-read-sequence-function*
  (quote
   #+sbcl sb-gray:stream-read-sequence
   #+openmcl ccl:stream-read-vector
   #-(or sbcl openmcl) (error "octet streams not supported in this implementation")))

(defvar *stream-write-sequence-function*
  (quote
   #+sbcl sb-gray:stream-write-sequence
   #+openmcl ccl:stream-write-vector
   #-(or sbcl openmcl) (error "octet streams not supported in this implementation")))
)


;;; implementation via Gray streams

;;; These could be specialized for particular implementations by hooking
;;; in directly to the "native" stream methods for the implementation.

(defclass octet-stream ()
  ((buffer :accessor buffer :initarg :buffer :type (simple-array (unsigned-byte 8) (*)))))

(defmethod #.*stream-element-type-function* ((stream octet-stream))
  '(unsigned-byte 8))


;;; input streams

(defclass octet-input-stream (octet-stream #.*binary-input-stream-class*)
  ((index :accessor index :initarg :index :type index)
   (end :accessor end :initarg :end :type index)))

(defmethod #.*stream-read-byte-function* ((stream octet-input-stream))
  (let ((buffer (buffer stream))
        (index (index stream)))
    (declare (type (simple-array (unsigned-byte 8) (*)) buffer))
    (cond
      ((>= index (end stream)) :eof)
      (t
       (setf (index stream) (1+ index))
       (aref buffer index)))))

(defun make-octet-input-stream (buffer &optional (start 0) end)
  "As MAKE-STRING-INPUT-STREAM, only with octets instead of characters."
  (declare (type (simple-array (unsigned-byte 8) (*)) buffer)
           (type index start)
           (type (or index null) end))
  (let ((end (or end (length buffer))))
    (make-instance 'octent-input-stream
                   :buffer buffer :index start :end end)))


;;; output streams

(defclass octet-output-stream (octet-stream #.*binary-output-stream-class*)
  ((index :accessor index :initform 0 :type index)))

(defmethod #.*stream-write-byte-function* ((stream octet-output-stream) integer)
  (declare (type (unsigned-byte 8) integer))
  (let* ((buffer (buffer stream))
         (length (length buffer))
         (index (index stream)))
    (declare (type (simple-array (unsigned-byte 8) (*)) buffer))
    (when (>= index (length buffer))
      (let ((new-buffer (make-array (* 2 length)
                                    :element-type '(unsigned-byte 8))))
        (declare (type (simple-array (unsigned-byte 8) (*)) new-buffer))
        (replace new-buffer buffer)
        (setf buffer new-buffer
              (buffer stream) new-buffer)))
    (setf (aref buffer index) integer
          (index stream) (1+ index))
    integer))

(defun get-output-stream-octets (stream)
  "As GET-OUTPUT-STREAM-STRING, only with an octet output-stream instead
of a string output-stream."
  (let ((buffer (buffer stream))
        (index (index stream)))
    (setf (index stream) 0)
    (subseq buffer 0 index)))

(defun make-octet-output-stream ()
  "As MAKE-STRING-OUTPUT-STREAM, only with octets instead of characters."
  (make-instance 'octet-output-stream
                 :buffer (make-array 128 :element-type '(unsigned-byte 8))))
