/*************************************************************************
 * 
 * irmp3 - Multimedia Audio Jukebox for Linux
 * http://irmp3.sourceforge.net
 *
 * $Source: /cvsroot/irmp3/irmp3/src/irmp3d/mod_browser.c,v $ -- interactive file and directory browser
 * $Id: mod_browser.c,v 1.11 2004/02/17 20:40:36 boucman Exp $
 *
 * Copyright (C) by Alexander Fedtke
 *
 * Please contact the current maintainer, Jeremy Rosen <jeremy.rosen@enst-bretagne.fr>
 * for information and support regarding irmp3.
 *
 *
 */
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <dirent.h>
#include <fnmatch.h>
#include <unistd.h>
#include <sys/types.h>
#include <fcntl.h>

#include "config.h"
#include "irmp3tools.h"
#include "irmp3log.h"
#include "irmp3config.h"
#include "irmp3mod.h"
#include "mod_browser.h"


/*************************************************************************
 * GLOBALS
 */


#define MOD_BROWSER_MAXENTRY 1024	// maximum number of entries per directory

#define MOD_BROWSER_DIRSTYPE_OTH 0
#define MOD_BROWSER_DIRSTYPE_DIR 1
#define MOD_BROWSER_DIRSTYPE_MP3 2
#define MOD_BROWSER_DIRSTYPE_M3U 3
#define MOD_BROWSER_DIRSTYPE_M3L 4
#define MOD_BROWSER_DIRSTYPE_EXE 5
#define MOD_BROWSER_DIRSTYPE_HID 6	// will not be displayed

char mod_browser_dirstype_icon[7] = {'?', '>', '+', '*', '=', '!', 'x'};

char mod_browser_path[512];
char mod_browser_root[512];
char mod_browser_exec[512];
char mod_browser_execmsg[512];
char mod_browser_dirs[MOD_BROWSER_MAXENTRY][512];
int  mod_browser_dirstype[MOD_BROWSER_MAXENTRY];

char mod_browser_mask_audio[512];
char mod_browser_mask_list[512];
char mod_browser_mask_listlist[512];
char mod_browser_mask_exec[512];
char mod_browser_mask_hide[512];

int  mod_browser_mask_sensitiveflag;

int  mod_browser_cdup               = 1;	// insert ".." directory?
int  mod_browser_cdup_initial_updir = 0;	// stay on ".." after entering directory?
int  mod_browser_pwddepth           = 1;
int  mod_browser_dirspos            = 0;
int  mod_browser_dirscount          = 0;

/*************************************************************************
 * MODULE INFO
 */
mod_t mod_browser = {
	"mod_browser",
	mod_browser_deinit,	// deinit
	mod_browser_reload,	// reload
	0,			// watchfd
	NULL,			// poll
	NULL,			// update
	mod_browser_message,	// message
	NULL,			// SIGCHLD handler
	mod_browser_init,
	NULL,			// avoid warning
};


/*************************************************************************
 * PRINT CURRENT CURSOR
 */
void mod_browser_printcursor(void)
{
    char *c, buf[512];
    
    strcpy(buf, mod_browser_dirs[mod_browser_dirspos]);
    if ((mod_browser_dirstype[mod_browser_dirspos]==MOD_BROWSER_DIRSTYPE_MP3) ||
        (mod_browser_dirstype[mod_browser_dirspos]==MOD_BROWSER_DIRSTYPE_M3U) ||
        (mod_browser_dirstype[mod_browser_dirspos]==MOD_BROWSER_DIRSTYPE_M3L) ||
	(mod_browser_dirstype[mod_browser_dirspos]==MOD_BROWSER_DIRSTYPE_EXE)) {
	if ((c=strrchr(buf, '.')))		//check if there is an extension
	    *c=0;				//cut extension
    }    

    mod_sendmsgf(MSGTYPE_INFO, "browser cursor %c%s",
		    mod_browser_dirstype_icon[mod_browser_dirstype[mod_browser_dirspos]],
		    buf);
    mod_sendmsgf(MSGTYPE_INFO, "browser cursorpos %d",
		    (mod_browser_dirscount == 1)                     ? 0  : 
		    (mod_browser_dirspos   == 0)                     ? 1  :
		    (mod_browser_dirspos+1 == mod_browser_dirscount) ? 99 : 100*mod_browser_dirspos/(mod_browser_dirscount-1));
}



/*************************************************************************
 * PRINT PARENT WORKING DIRECTORY
 */
void mod_browser_printpwd(void)
{
    char *p, *r;
    int i, c=0;

    r = strlen(mod_browser_root)==1 ? mod_browser_path : &mod_browser_path[strlen(mod_browser_root)];	//cut leading 'browser_root'
    if (*r) {
	for (i=strlen(r)-1; i>=0 && c<mod_browser_pwddepth; i--)  //check pwd depth
	    if (r[i]=='/') c++;
	p=&r[i+1];
	if (p!=r) p++; 						  //remove leading '/' if not browser_root
	mod_sendmsgf(MSGTYPE_INFO, "browser pwd %s", p);
    } else
	mod_sendmsgf(MSGTYPE_INFO, "browser pwd /");
}


/*************************************************************************
 * CHECK FILE TYPE
 */
int mod_browser_checkdirstype(char *dest)	//dest = path+filename
{
    struct stat st;

    if (!dest)
	return MOD_BROWSER_DIRSTYPE_OTH;

    if (stat(dest, &st)) {
	log_printf(LOG_DEBUG, "mod_browser_checkdirstype(): Can't stat '%s'. Skipping.\n", dest);
	return MOD_BROWSER_DIRSTYPE_OTH;
    }

    if (S_ISDIR(st.st_mode)) return MOD_BROWSER_DIRSTYPE_DIR;

    if (!fnmatch(mod_browser_mask_audio, dest, FNM_NOESCAPE | mod_browser_mask_sensitiveflag))
	return MOD_BROWSER_DIRSTYPE_MP3;

    if (!fnmatch(mod_browser_mask_list, dest, FNM_NOESCAPE | mod_browser_mask_sensitiveflag))
	return MOD_BROWSER_DIRSTYPE_M3U;
    
    if (!fnmatch(mod_browser_mask_listlist, dest, FNM_NOESCAPE | mod_browser_mask_sensitiveflag))
	return MOD_BROWSER_DIRSTYPE_M3L;

    if (mod_browser_exec && !strncasecmp(mod_browser_exec, dest, strlen(mod_browser_exec)))
	if (!fnmatch(mod_browser_mask_exec, dest, FNM_NOESCAPE | mod_browser_mask_sensitiveflag))
	    return MOD_BROWSER_DIRSTYPE_EXE;

    if (!fnmatch(mod_browser_mask_hide, dest, FNM_NOESCAPE | mod_browser_mask_sensitiveflag))
	return MOD_BROWSER_DIRSTYPE_HID;

    return MOD_BROWSER_DIRSTYPE_OTH;
}


/*************************************************************************
 * SCAN DIRECTORY
 */
int mod_browser_intodirscan(char *dest)
{
    DIR *dir;
    struct dirent *dirent;
    struct stat st;
    
    char buf[512];
    char filelist[MOD_BROWSER_MAXENTRY][512];
    int  dircount=0, filecount=0, hidecount=0;
    int  i;

    if (!dest)
	return 0;
    
    log_printf(LOG_DEBUG, "mod_browser_intodirscan(): Scanning dir '%s'.\n", dest);

    dir = opendir(dest);
    if (!dir) {
	log_printf(LOG_DEBUG, "mod_browser_intodirscan(): Unable to read dir '%s'.\n", mod_browser_path);
	return 0;
    }

    while (1) {
	dirent = readdir(dir);
	if (!dirent)
	    break;
	if (!strcmp(dirent->d_name, "."))
	    continue;
	if (!strcmp(dirent->d_name, ".."))
	    if (!mod_browser_cdup)
		continue;
	if (!strcasecmp(dest, "/"))
	    snprintf(buf, sizeof(buf)-1, "/%s", dirent->d_name);
	else
	    snprintf(buf, sizeof(buf)-1, "%s/%s", dest, dirent->d_name);

	if (stat(buf, &st)) {
	    log_printf(LOG_DEBUG, "mod_browser_intodirscan(): Can't stat '%s'. Skipping.\n", buf);
	    continue;
	}
	switch(mod_browser_checkdirstype(buf)) {
	case MOD_BROWSER_DIRSTYPE_DIR:
					strcpy(mod_browser_dirs[dircount], dirent->d_name);
					dircount++;
					break;		//break switch statement
	case MOD_BROWSER_DIRSTYPE_HID:
					hidecount++;
					continue;	//continue next while loop
	default:
					strcpy(filelist[filecount], dirent->d_name);
					filecount++;
					break;		//break switch statement
	}

	if (dircount+filecount >= MOD_BROWSER_MAXENTRY) 
	    break;
    }
    closedir(dir);

    strcpy(mod_browser_path, dest);		//Set current path
    mod_browser_dirspos = 0;			//first entry
    if (!(dircount+filecount)) {		//Dir empty
	strcpy(mod_browser_dirs[0], "EMPTY");
	mod_browser_dirstype[0]=MOD_BROWSER_DIRSTYPE_OTH;
	mod_browser_dirscount=1;
	log_printf(LOG_DEBUG, "mod_browser_intodirscan(): directory is empty.\n");
    } else {
	//sort dirs and files (but dont mix them)
	qsort(mod_browser_dirs, dircount, 512, (void*)strcasecmp);
	qsort(filelist, filecount, 512, (void*)strcasecmp);

	//concatenate dirs+files
	for (i=0; i<filecount; i++)
	    strcpy(mod_browser_dirs[dircount+i], filelist[i]);

	mod_browser_dirscount = dircount+filecount;	//# of entries

	//check file type again
	for (i=0; i<mod_browser_dirscount; i++) {
	    if (!strcasecmp(dest, "/"))
		snprintf(buf, sizeof(buf)-1, "/%s", mod_browser_dirs[i]);
	    else
		snprintf(buf, sizeof(buf)-1, "%s/%s", dest, mod_browser_dirs[i]);
	    mod_browser_dirstype[i]=mod_browser_checkdirstype(buf);
	}
	log_printf(LOG_DEBUG, "mod_browser_intodirscan(): %d directories, %d files found.\n", dircount, filecount);

        if (!mod_browser_cdup_initial_updir) {
	    // display 1st entry instead of ".."
	    if (mod_browser_cdup && (mod_browser_dirscount > 1))
		mod_browser_dirspos++;
	}

    }
    log_printf(LOG_DEBUG, "mod_browser_intodirscan(): %d hidden files.\n", hidecount);

    return 1;						//success
}



/*************************************************************************
 * SCROLL WITHIN CURRENT DIRECTORY
 */
void mod_browser_cursor(char *p)
{
    int	 step;
    char *current;

    if (!p) 
	return;

    if (p[1]==0 && (p[0]=='+' || p[0]=='-')) {	//search next letter
	step = (p[0]=='+') ? 1 : -1;		//search direction

	current=mod_browser_dirs[mod_browser_dirspos];
	while (mod_browser_dirspos+step >= 0 &&
	       mod_browser_dirspos+step <= mod_browser_dirscount-1 &&
	       !strncasecmp(mod_browser_dirs[mod_browser_dirspos], current, 1)) {
	       
	       mod_browser_dirspos+=step;
	}
    
    } else {					//scroll x positions up/down

	if (*p!='+' && *p!='-') 		//set absolute position
            mod_browser_dirspos = 0;

	step=atoi(p);
        if (mod_browser_dirspos+step < 0) 
	    mod_browser_dirspos = 0;
        else if (mod_browser_dirspos+step > mod_browser_dirscount-1)
	    mod_browser_dirspos = mod_browser_dirscount-1;
        else 
	    mod_browser_dirspos += step;
    }
}


/*************************************************************************
 * CHANGE DIRECTORY BACK (cd ..)
 */
int mod_browser_back()
{
    char buf[512], oldbuf[512], *old, *b;

    if (strcasecmp(mod_browser_path, mod_browser_root)) {
	snprintf(oldbuf, sizeof(buf)-1, "%s", mod_browser_path);
	old=strrchr(oldbuf, '/');
	if (old) old++;

	snprintf(buf, sizeof(buf)-1, "%s", mod_browser_path);
	b=strrchr(buf, '/');
	if (b==buf) *++b=0;
	else *b=0;

	if (!mod_browser_intodirscan(buf)) {
	    log_printf(LOG_DEBUG, "mod_browser_back(): Unable to change into directory '%s'.\n", buf);
	    return 0;	
	}

	if (old) {	//search for old directory
	    for (mod_browser_dirspos=mod_browser_dirscount-1; mod_browser_dirspos>=0; mod_browser_dirspos--)
		if (!strcasecmp(mod_browser_dirs[mod_browser_dirspos], old)) 
		    break;
	    if (mod_browser_dirspos < 0) mod_browser_dirspos=0;
	} else mod_browser_dirspos=0;

	log_printf(LOG_DEBUG, "mod_browser_back(): new pwd is '%s'.\n", mod_browser_path);
	return 1;
    }
    log_printf(LOG_DEBUG, "mod_browser_back(): browser_root '%s' reached.\n", mod_browser_root);
    return 0;	//cant cd ..
}



/*************************************************************************
 * CHANGE INTO DIRECTORY
 */
int mod_browser_into()
{
    char buf[512];

    if (!strcmp(mod_browser_dirs[mod_browser_dirspos], ".."))
	return mod_browser_back();
    
    if (!strcmp(mod_browser_path, "/"))
	snprintf(buf, sizeof(buf)-1, "/%s", mod_browser_dirs[mod_browser_dirspos]);
    else
	snprintf(buf, sizeof(buf)-1, "%s/%s", mod_browser_path, mod_browser_dirs[mod_browser_dirspos]);

    if (mod_browser_dirstype[mod_browser_dirspos] == MOD_BROWSER_DIRSTYPE_DIR)
	if (mod_browser_intodirscan(buf)) {
	    log_printf(LOG_DEBUG, "mod_browser_into(): Changed into '%s'.\n", buf);
	    return 1;	//success
	}

    //error
    log_printf(LOG_DEBUG, "mod_browser_into(): Unable to change into: '%s'.\n", buf);
    return 0;	
}



int mod_browser_script(char *command)
{
	char type[16];
	int tmp;
	switch(mod_browser_dirstype[mod_browser_dirspos]) {
		case MOD_BROWSER_DIRSTYPE_DIR:	strcpy(type, "dir");
						break;
		case MOD_BROWSER_DIRSTYPE_MP3:	strcpy(type, "audio");
						break;
		case MOD_BROWSER_DIRSTYPE_M3U:	strcpy(type, "list");
						break;
		case MOD_BROWSER_DIRSTYPE_M3L:	strcpy(type, "listlist");
						break;
		case MOD_BROWSER_DIRSTYPE_EXE:	strcpy(type, "exec");
						break;
		default:			strcpy(type, "other");
						break;	
	}
	tmp = system_block(0,NULL,NULL,"/bin/sh %s %s %s %s %s",command,mod_browser_execmsg,mod_browser_path,mod_browser_dirs[mod_browser_dirspos],type);
	return tmp;
}


/*************************************************************************
 * EXECUTE SHELL SCRIPT
 */
void mod_browser_execute(char *command) {

    int  fd;
    char buf[512];

    if (!command)
	return;

    //clear message file
    if (mod_browser_execmsg)
	if (!truncate(mod_browser_execmsg, 0))
	    log_printf(LOG_DEBUG, "mod_browser_exec(): Message file '%s' cleared.\n", mod_browser_execmsg);
	else	
	    log_printf(LOG_DEBUG, "mod_browser_exec(): Error clearing message file '%s'.\n", mod_browser_execmsg);
    else
	log_printf(LOG_DEBUG, "mod_browser_exec(): Message file disabled.\n");


    log_printf(LOG_DEBUG, "mod_browser_exec(): Executing '%s'...\n", command);

    if (!mod_browser_script(command)) {

        log_printf(LOG_DEBUG, "mod_browser_exec(): Executing done.\n");

	if (mod_browser_execmsg) {	//try to open msg file and display it
	
	    log_printf(LOG_DEBUG, "mod_browser_exec(): Reading message file '%s'.\n", mod_browser_execmsg);

	    fd=open(mod_browser_execmsg, O_RDONLY);
	    if (fd >= 0) {
		if (readline(fd, buf, sizeof(buf)) >= 0) {
		    log_printf(LOG_DEBUG, "mod_browser_exec(): Script reports '%s'.\n", buf);
		    mod_sendmsgf(MSGTYPE_INFO, "browser info %s", buf);
		} else {
		    log_printf(LOG_DEBUG, "mod_browser_exec(): Script reports nothing.\n");
		}
		//read multiple irmp3 commands from message file
		while (readline(fd, buf, sizeof(buf)) >= 0) {
		    log_printf(LOG_DEBUG, "mod_browser_exec(): Script issues command '%s'.\n", buf);
		    mod_sendmsgf(MSGTYPE_INPUT, "%s", buf);
		}
		close(fd);
	    } else {
		log_printf(LOG_DEBUG, "mod_browser_exec(): Message file '%s' not found.\n", mod_browser_execmsg);
		mod_sendmsgf(MSGTYPE_INFO, "browser info Done");
	    }
	} else {
	    log_printf(LOG_DEBUG, "mod_browser_exec(): Message file disabled.\n");
	    mod_sendmsgf(MSGTYPE_INFO, "browser info Done");
	}
    } else {
	log_printf(LOG_DEBUG, "mod_browser_exec(): Script execution failed.\n");
	mod_sendmsgf(MSGTYPE_INFO, "browser info Error");
    }
    return;
}



/*************************************************************************
 * ADD CURRENT CURSOR TO PLAYLIST
 */
void mod_browser_add()
{
    char buf[512];
    
    // disable browser_add on ".."
    if (!strcmp(mod_browser_dirs[mod_browser_dirspos], "..")) {
	log_printf(LOG_DEBUG, "mod_browser_add(): Sorry, cannot add '..' to playlist.\n");
	return;
    }

    snprintf(buf, sizeof(buf)-1, "%s/%s", mod_browser_path, mod_browser_dirs[mod_browser_dirspos]);
    switch (mod_browser_dirstype[mod_browser_dirspos]) {
	case MOD_BROWSER_DIRSTYPE_DIR: 	strcat(buf, "/");
					strcat(buf, mod_browser_mask_audio);
					mod_sendmsgf(MSGTYPE_INPUT, "playlist loaddirplus %s", buf);
					log_printf(LOG_DEBUG, "mod_browser_add(): Adding directory '%s'.\n", buf);
					break;
	case MOD_BROWSER_DIRSTYPE_MP3: 	mod_sendmsgf(MSGTYPE_INPUT, "playlist addplus %s", buf);
					log_printf(LOG_DEBUG, "mod_browser_add(): Adding mp3 file '%s'.\n", buf);
					break;
	case MOD_BROWSER_DIRSTYPE_M3U: 	mod_sendmsgf(MSGTYPE_INPUT, "playlist loadplus %s", buf);
					log_printf(LOG_DEBUG, "mod_browser_add(): Adding playlist '%s'.\n", buf);
					break;
	case MOD_BROWSER_DIRSTYPE_M3L: 	//Maybe future playlistlist support
					//mod_sendmsgf(MSGTYPE_INPUT, "playlistlist loadplus %s", buf);
					//log_printf(LOG_DEBUG, "mod_browser_add(): Adding playlist-list '%s'\n", buf);
					//break;
					break;
	case MOD_BROWSER_DIRSTYPE_EXE:	mod_browser_execute(buf);
					break;
	default: 			mod_sendmsg(MSGTYPE_INFO, "browser info Error");
					log_printf(LOG_DEBUG, "mod_browser_add(): Ignoring entry '%s'.\n", buf);
					break;
    }
}


/*************************************************************************
 * PLAY CURRENT CURSOR
 */
void mod_browser_play()
{
    if (mod_browser_dirstype[mod_browser_dirspos] != MOD_BROWSER_DIRSTYPE_EXE) {
	log_printf(LOG_DEBUG, "mod_browser_play(): stopping and clearing playlist.\n");
        mod_sendmsg(MSGTYPE_INPUT, "stop");			//stop playing
        mod_sendmsg(MSGTYPE_PLAYER, "stop");			//notify mod_playlist that we are stopped
        mod_sendmsg(MSGTYPE_INPUT, "playlist clearnostop");	//clear playlist w/o issuing STOP command
        mod_sendmsg(MSGTYPE_PLAYER, "browser clrscr");		//remove directory listing from screen
    }
    log_printf(LOG_DEBUG, "mod_browser_play(): Adding current cursor.\n");
    mod_browser_add();					//add and start playing automatically
}


/*************************************************************************
 * SET CURRENT DIRECTORY
 */
int mod_browser_setpwd(char *dest)
{
    char buf[512];
    
    if (!dest) {
        log_printf(LOG_ERROR, "mod_browser_setpwd(): unable to set NULL directory.\n");
	return 0;
    }

    // relative path?
    if (dest[0]!='/') {
	if (strlen(mod_browser_root)==1)
	    snprintf(buf, sizeof(buf)-1, "/%s", dest);
	else
	    snprintf(buf, sizeof(buf)-1, "%s/%s", mod_browser_root, dest);
        log_printf(LOG_DEBUG, "mod_browser_setpwd(): relative -> absolute path '%s'.\n", buf);
    } else
	strncpy(buf, dest, 512);	//already absolute path

    //subdir of browser_root?
    if (strncasecmp(mod_browser_root, buf, strlen(mod_browser_root))) {
        log_printf(LOG_ERROR, "mod_browser_setpwd(): '%s' is not a subdir of browser_root '%s'.\n", buf, mod_browser_root);
	return 0;
    }	

    //dir?
    if (mod_browser_checkdirstype(buf) != MOD_BROWSER_DIRSTYPE_DIR) {
	log_printf(LOG_ERROR, "mod_browser_setpwd(): not a directory '%s'.\n", buf);
	return 0;
    }

    //directory scan ok?
    if (!mod_browser_intodirscan(buf)) {
	log_printf(LOG_ERROR, "mod_browser_setpwd(): unable to scan directory '%s'.\n", buf);
	return 0;
    }
    
    log_printf(LOG_DEBUG, "mod_browser_setpwd(): set to '%s'.\n", buf);
    return 1;
}

/*************************************************************************
 * REFRESH DISPLAY
 */
void mod_browser_refresh()
{
    mod_browser_printpwd();
    mod_browser_printcursor();
    return;
}

/*************************************************************************
 * RECEIVE MESSAGE
 */
void mod_browser_message(int msgtype, char *msg,const char __attribute__((unused))*sender)
{
    char *c1, *c2, *c3;

    if (msgtype == MSGTYPE_INPUT) {

	c1 = strtok(msg, " \t");
	if(!c1) return;
	c2 = strtok(NULL, " \t");
	if(!c2) return;
	

	if ( !strcasecmp(c1, "browser")) {
		if ( !strcasecmp(c2, "cursor") ) {
			c3 = strtok(NULL, "");
			if(!c3)return;
			mod_browser_cursor(c3);
			mod_browser_refresh();
		} else if ( !strcasecmp(c2, "back")) {
			mod_browser_back();
			mod_browser_refresh();
		} else if ( !strcasecmp(c2, "into")) {
			mod_browser_into();
			mod_browser_refresh();
		} else if ( !strcasecmp(c2, "add")) {
			mod_browser_add();
			mod_browser_refresh();
		} else if ( !strcasecmp(c2, "setpwd") ) {
			c3 = strtok(NULL, "");
			if(!c3)return;
			mod_browser_setpwd(c3);
			mod_browser_refresh();
		} else if ( !strcasecmp(c2, "clear")) {
			mod_sendmsg(MSGTYPE_INPUT,  "playlist clear");
			mod_sendmsg(MSGTYPE_INFO, "browser info Playlist cleared");
			mod_browser_refresh();
		} else if ( !strcasecmp(c2, "show")) {
			mod_browser_refresh();
		} else if ( !strcasecmp(c2, "play")) {
			mod_browser_play();
		} else if ( !strcasecmp(c2, "exec") ) {
			c3 = strtok(NULL, "");
			if(!c3)return;
			mod_browser_execute(c3);
		} else {
			c3 = strtok(NULL, "");
			log_printf(LOG_DEBUG, "mod_browser_message(): ignoring '%s %s %s'.\n", c1, c2, c3);
		}

		return;
	}
    }
}


/*************************************************************************
 * MODULE RELOAD FUNCTION
 */
char *mod_browser_reload(void)
{

    log_printf(LOG_DEBUG, "mod_browser_reload(): reloading\n");

    //set audio file mask
    strcpy(mod_browser_mask_audio, config_getstr("browser_mask_audio", "/*/*.[Mm][Pp]3"));
    log_printf(LOG_DEBUG, "mod_browser_reload(): mask_audio is '%s'\n", mod_browser_mask_audio);

    //set playlist file mask
    strcpy(mod_browser_mask_list, config_getstr("browser_mask_list", "*.[Mm]3[Uu]"));
    log_printf(LOG_DEBUG, "mod_browser_reload(): mask_list is '%s'\n", mod_browser_mask_list);

    //set playlist-list file mask
    strcpy(mod_browser_mask_listlist, config_getstr("browser_mask_listlist", "*.[Mm]3[Ll]"));
    log_printf(LOG_DEBUG, "mod_browser_reload(): mask_listlist is '%s'\n", mod_browser_mask_listlist);

    //set executable file mask
    strcpy(mod_browser_mask_exec, config_getstr("browser_mask_exec", "*.[Ss][Hh]"));
    log_printf(LOG_DEBUG, "mod_browser_reload(): mask_exec is '%s'\n", mod_browser_mask_exec);

    //set execlude file mask
    strcpy(mod_browser_mask_hide, config_getstr("browser_mask_hide", "*"));
    log_printf(LOG_DEBUG, "mod_browser_reload(): mask_hide is '%s'\n", mod_browser_mask_hide);

    //case sensitive matching
    //according to fnmatch(2) mod_browser_mask_sensitiveflag=0 means sensitive
    //and mod_browser_mask_sensitiveflag=FNM_CASEFOLD means NOT case sensitive
    if (!strcasecmp(config_getstr("browser_mask_casesensitive", "yes"), "yes"))
	mod_browser_mask_sensitiveflag = 0;
    else
	mod_browser_mask_sensitiveflag = 0; //should be FNM_CASEFOLD (defined in fnmatch.h) but only when GNU source....????;
    log_printf(LOG_DEBUG, "mod_browser_reload(): mask case sensitive: %s\n", mod_browser_mask_sensitiveflag ? "no" : "yes");

    // set pwd path depth to show on LCD
    mod_browser_pwddepth=config_getnum("browser_pwddepth", 2);
    if (mod_browser_pwddepth<1) mod_browser_pwddepth=1;
    log_printf(LOG_DEBUG, "mod_browser_reload(): Setting pwd path depth to '%d'\n", mod_browser_pwddepth);

    // show ".." directory?
    mod_browser_cdup=(strcasecmp("yes", config_getstr("browser_cdup", "no")) == 0);
    log_printf(LOG_DEBUG, "mod_browser_reload(): show '..' directory='%s'\n", mod_browser_cdup ? "yes" : "no");

    // stay on ".." entry after entering directory?
    mod_browser_cdup_initial_updir = (strcasecmp("yes", config_getstr("browser_cdup_initial_updir", "no")) == 0);
    log_printf(LOG_DEBUG, "mod_browser_reload(): stay on '..' after cd='%s'\n", mod_browser_cdup_initial_updir ? "yes" : "no");
    //set chroot directory
    strcpy(mod_browser_root, config_getstr("browser_root", "/"));
    if (!mod_browser_root || mod_browser_root[0]!='/')	//allow ablosute path only
	strcpy(mod_browser_root, "/");	
    if (!mod_browser_intodirscan(mod_browser_root))	//only exsting directories
	strcpy(mod_browser_root, "/");
    log_printf(LOG_DEBUG, "mod_browser_reload(): root dir is '%s'\n", mod_browser_root);

    //set exec directory
    if (strcasecmp(config_getstr("browser_exec", "-"), "-")) {	//enabled
	strcpy(mod_browser_exec, mod_browser_root);
	if (mod_browser_exec[strlen(mod_browser_exec)-1] != '/')
	    strcat(mod_browser_exec, "/");
	strcat(mod_browser_exec, config_getstr("browser_exec", ""));
	if ( (mod_browser_exec[strlen(mod_browser_exec)-1] == '/') && (strlen(mod_browser_exec)>1) )
	    mod_browser_exec[strlen(mod_browser_exec)-1]=0;
	log_printf(LOG_DEBUG, "mod_browser_reload(): exec dir is '%s'\n", mod_browser_exec);
    } else {
	mod_browser_exec[0]=0;
	log_printf(LOG_DEBUG, "mod_browser_reload(): exec dir disabled\n");
    }

    //set exec output file
    strcpy(mod_browser_execmsg, config_getstr("browser_execmsg", "/tmp/irmp3.tmp.execmsg"));
    if (!mod_browser_execmsg)
	strcpy(mod_browser_execmsg, "/irmp3.tmp.execmsg");	//DONT allow empty execmsg because of command line paramter count
    log_printf(LOG_DEBUG, "mod_browser_reload(): exec messages file is '%s'\n", mod_browser_execmsg);

    //set home directory
    strcpy(mod_browser_path, mod_browser_root);
    if (mod_browser_path[strlen(mod_browser_path)-1] != '/')
	strcat(mod_browser_path, "/");
    strcat(mod_browser_path, config_getstr("browser_home", "/"));
    if ( (mod_browser_path[strlen(mod_browser_path)-1] == '/') && (strlen(mod_browser_path)>1) )
	mod_browser_path[strlen(mod_browser_path)-1]=0;
    if (!mod_browser_intodirscan(mod_browser_path)) {
	log_printf(LOG_ERROR, "mod_browser_reload(): unable to read home dir '%s'. Using '%s'.\n", mod_browser_path, mod_browser_root);
	if (!mod_browser_intodirscan(mod_browser_root))
	    return strcpy("error scanning fallback home directory ", mod_browser_root);
    }
    log_printf(LOG_DEBUG, "mod_browser_reload(): home dir is '%s'\n", mod_browser_path);



    return NULL;
}



/*************************************************************************
 * MODULE INIT FUNCTION
 */
char *mod_browser_init(void)
{
	char *p;
	
	log_printf(LOG_DEBUG, "mod_browser_init(): initializing\n");

	if ((p=mod_browser_reload())) return p;
	
	return NULL;
}


/*************************************************************************
 * MODULE DEINIT FUNCTION
 */
void mod_browser_deinit(void)
{
	log_printf(LOG_DEBUG, "mod_browser_deinit(): deinitialized\n");
}


/*************************************************************************
 * EOF
 */
