/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2011 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "idatasubject.h"


#include "ibuffer.h"
#include "idata.h"
#include "idataconsumer.h"
#include "idatalimits.h"
#include "idatareader.h"
#include "ifileloader.h"
#include "ierror.h"
#include "ihistogrammaker.h"
#include "iprobefilter.h"

//
//  Templates
//
#include "iarraytemplate.h"
#include "ibuffertemplate.h"


IOBJECT_DEFINE_TYPE(iDataSubject,Data,-d,iObjectType::_Data);


IOBJECT_DEFINE_KEY(iDataSubject,DataPresent,dp,Bool,1);
IOBJECT_DEFINE_KEY(iDataSubject,FixedLimits,fl,Bool,1);
IOBJECT_DEFINE_KEY(iDataSubject,FileName,fn,String,1);
IOBJECT_DEFINE_KEY(iDataSubject,FixedStretch,fs,Bool,0);
IOBJECT_DEFINE_KEY(iDataSubject,LowerLimit,lo,Float,0);
IOBJECT_DEFINE_KEY(iDataSubject,Max,max,Float,0);
IOBJECT_DEFINE_KEY(iDataSubject,Min,min,Float,0);
IOBJECT_DEFINE_KEY(iDataSubject,NumVars,n,Int,1);
IOBJECT_DEFINE_KEY(iDataSubject,Name,na,String,0);
IOBJECT_DEFINE_KEY(iDataSubject,Stretch,s,Int,0);
IOBJECT_DEFINE_KEY(iDataSubject,UpperLimit,up,Float,0);

IOBJECT_DEFINE_KEY(iDataSubject,ResizeLimits,-rl,Int,1); // hidden key


bool iDataSubject::IsTypeMatch(const iString &prefix)
{
	iString pre = prefix.Section("-",0,0);
	if(pre == "d") return true; else return iDataSubject::Type().IsMatch(pre);
}


iString iDataSubject::GetDataSubjectName(const iString &prefix)
{
	static iString none;
	//
	//  Set the subject name
	//
	int i1 = prefix.Find("-");
	if(i1 < 0) return none;
	int i2 = prefix.Find(iObjectKey::PrefixSeparator());
	if(i2 < 0) i2 = prefix.Length();
	return prefix.Part(i1+1,i2-i1-1);
}


iDataSubject::iDataSubject(iFileLoader *fl, const iString &name) : iObject(name), mViewModule(fl?fl->GetReader()->GetViewModule():0), mId(fl?fl->NumStreams():-1)
{
	IERROR_ASSERT(fl);

	mLoader = fl;

	//
	//  Add self as a new stream
	//
	iFileLoader::Stream *s = fl->CreateNewStream(); IERROR_ASSERT(s);
	s->Subject = this; 
	fl->mStreams.Add(s);

	mLimits = 0;
	mFixedLimits = false;

	mLimitsNum = 1;
	mLimitsName = "Component";
}


iDataSubject::~iDataSubject()
{
	if(mLimits != 0) mLimits->Delete();
}


const iObjectType& iDataSubject::GetObjectType() const
{
	return iDataSubject::Type();
}


iDataLimits* iDataSubject::CreateLimits() const
{
	return iDataLimits::New(this,mLimitsNum,mLimitsName);
}


void iDataSubject::ConfigureLimits(int num, const iString &name)
{
	if(num > 0) mLimitsNum = num;
	mLimitsName = name;
}


bool iDataSubject::IsThereData() const
{
	return mLoader->IsThereData(mId);
}


vtkDataSet* iDataSubject::GetData() const
{
	return mLoader->GetData(mId);
}


iDataLimits* iDataSubject::GetLimits() const
{
	if(mLimits == 0)
	{
		mLimits = this->CreateLimits();
	}
	return mLimits;
}


void iDataSubject::NotifyDependencies()
{
	//
	//  Default implementation, but can be changed in a child
	//
	this->GetLimits()->NotifyDependencies(-1);
}


void iDataSubject::SetFixedLimits(bool s)
{
	mFixedLimits = s;
	this->ClearCache();
}


void iDataSubject::PackCompleteState(iString &s) const
{
	iDataLimits *limits = this->GetLimits();
	//
	//  Save the currently loaded Vars
	//
	limits->BackupVars();
	limits->BlockNotifications(true);
	limits->AssignVars();
	limits->BlockNotifications(false);
	//
	//  Pack state without caching
	//
	s.Clear();
	this->PackValue(s,KeyResizeLimits(),limits->GetSize());
	this->PackStateBody(s);
	//
	//  Restore the currently loaded Vars
	//
	limits->RestoreVars();
}


void iDataSubject::UnPackCompleteState(const iString &s)
{
	iDataLimits *limits = this->GetLimits();
	//
	//  Save the currently loaded Vars
	//
	if(this->IsThereData())
	{
		limits->BackupVars();
	}
	//
	//  Unpack state and clear the cache
	//
	this->UnPackStateBody(s);
	this->ClearCache();
	//
	//  Restore the currently loaded Vars
	//
	if(this->IsThereData())
	{
		limits->RestoreVars();
	}
}


void iDataSubject::PackStateBody(iString &s) const
{
	static iBuffer<int> itmp;
	static iBuffer<bool> btmp;
	static iBuffer<float> ftmp;
	static iBuffer<iString> stmp;
	int i, n;

	iDataLimits *limits = this->GetLimits();

	this->PackValue(s,KeyFixedLimits(),mFixedLimits);
	this->PackValue(s,KeyDataPresent(),this->IsThereData());
	this->PackValue(s,KeyFileName(),mLoader->GetLastFileName());

	n = limits->GetNumVars();
	this->PackValue(s,KeyNumVars(),n);

	if(n > 0)
	{
		for(i=0; i<n; i++) itmp[i] = limits->GetStretch(i); 
		this->PackValue(s,KeyStretch(),itmp,n);

		for(i=0; i<n; i++) ftmp[i] = limits->GetMin(i);
		this->PackValue(s,KeyMin(),ftmp,n);

		for(i=0; i<n; i++) ftmp[i] = limits->GetMax(i);
		this->PackValue(s,KeyMax(),ftmp,n);

		for(i=0; i<n; i++) ftmp[i] = limits->GetLowerLimit(i);
		this->PackValue(s,KeyLowerLimit(),ftmp,n);

		for(i=0; i<n; i++) ftmp[i] = limits->GetUpperLimit(i);
		this->PackValue(s,KeyUpperLimit(),ftmp,n);

		for(i=0; i<n; i++) btmp[i] = limits->GetFixedStretch(i);
		this->PackValue(s,KeyFixedStretch(),btmp,n);

		for(i=0; i<n; i++) stmp[i] = limits->GetName(i);
		this->PackValue(s,KeyName(),stmp,n);
	}

	this->DataSubjectPackStateBody(s);
}


void iDataSubject::UnPackStateBody(const iString &s)
{
	static iBuffer<int> itmp;
	static iBuffer<float> ftmp, ftmp2;
	static iBuffer<iString> stmp;
	int i, n; bool b;
	
	iDataLimits *limits = this->GetLimits();
	limits->BlockNotifications(true);

	if(this->UnPackValue(s,KeyResizeLimits(),n))
	{
		limits->Resize(n);
		if(!this->IsThereData()) limits->AssignVars();
		this->ClearCache();
	}
	if(this->UnPackValue(s,KeyFixedLimits(),b)) this->SetFixedLimits(b);

	n = limits->GetNumVars();
	if(this->UnPackValue(s,KeyNumVars(),n))
	{
		this->ClearCache();
		n = limits->GetNumVars();
	}

	if(n > 0)
	{
		for(i=0; i<n; i++) itmp[i] = limits->GetStretch(i);
		if(this->UnPackValue(s,KeyStretch(),itmp,n))
		{
			for(i=0; i<n; i++) limits->SetStretch(i,itmp[i]);
			this->ClearCache();
		}

		for(i=0; i<n; i++) ftmp[i] = limits->GetMin(i);
		if(this->UnPackValue(s,KeyMin(),ftmp,n))
		{
			for(i=0; i<n; i++) ftmp2[i] = limits->GetMax(i);
			if(this->UnPackValue(s,KeyMax(),ftmp2,n))
			{
				for(i=0; i<n; i++) limits->SetMinMax(i,ftmp[i],ftmp2[i]);
				this->ClearCache();
			}
			else
			{
				for(i=0; i<n; i++) limits->SetMin(i,ftmp[i]);
				this->ClearCache();
			}
		}
		else
		{
			for(i=0; i<n; i++) ftmp[i] = limits->GetMax(i);
			if(this->UnPackValue(s,KeyMax(),ftmp,n))
			{
				for(i=0; i<n; i++) limits->SetMax(i,ftmp[i]);
				this->ClearCache();
			}
		}

		for(i=0; i<n; i++) ftmp[i] = limits->GetLowerLimit(i);
		if(this->UnPackValue(s,KeyLowerLimit(),ftmp,n))
		{
			for(i=0; i<n; i++) limits->SetLowerLimit(i,ftmp[i]);
			this->ClearCache();
		}

		for(i=0; i<n; i++) ftmp[i] = limits->GetUpperLimit(i);
		if(this->UnPackValue(s,KeyUpperLimit(),ftmp,n))
		{
			for(i=0; i<n; i++) limits->SetUpperLimit(i,ftmp[i]);
			this->ClearCache();
		}

		for(i=0; i<n; i++) stmp[i] = limits->GetName(i);
		if(this->UnPackValue(s,KeyName(),stmp,n))
		{
			for(i=0; i<n; i++) limits->SetName(i,stmp[i]);
			this->ClearCache();
		}
	}

	this->DataSubjectUnPackStateBody(s);

	limits->BlockNotifications(false);
	limits->NotifyDependencies(-1);
}


iProbeFilter* iDataSubject::CreateProbeFilter(iViewSubject *vo) const
{
	return iProbeFilter::New(vo);
}


iHistogramMaker* iDataSubject::CreateHistogramMaker() const
{
	return iHistogramMaker::New(this->GetViewModule());
}


const iDataSyncRequest& iDataSubject::Request(int var) const
{
	return iDataConsumer::Request(this->GetDataType(),var);
}

