/*
   Copyright (C) 1999, 2000 PolyWog and Javaman for Ghetto.Org
   This file is part of the PCR-1000 API Library.

   The PCR-1000 API Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The PCR-1000 API Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the PCR-1000 API Library; see the file LICENSE.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
 */

#include "plog.h"
#include <errno.h>

/* 
 * The constructor: pass the filename and the file
 * mode as a string.  "r" is for read access, "w" is
 * for write access and overwrite the existing file,
 * and "a" is to append to the existing file.
 */

PLog :: PLog (char *FileName, char *FileMode)
{
	if ((logfile_p = fopen(FileName, FileMode)) == NULL) {
		perror("PLog fopen");
		exit (-1);
	} 
	
	if (strcmp(FileMode, "r") == 0) {
		writefile = false;
		readfile = true;
		CountChannels();
		ReadChannels();
	} else if (strcmp(FileMode, "w") == 0) {
		Intro();
		writefile = true;
		readfile = false;
	} else if (strcmp(FileMode, "a") == 0) {
		writefile = true;
		readfile = false;
	}

}

// Nothing in the destructor as of yet.
PLog :: ~PLog()
{
	if(readfile) {
		free(chanfreqs);
	}
}

// Intro() adds the header to the log file.  This is
// a private method, don't worry too much about it.
void PLog :: Intro() 
{
	CurrentTime(temp);
	fprintf(logfile_p, "//PCR-1000 Log, generated by IcomLib from Ghetto.org\n//Started: %s\n//Freq\t\tDate/Time\t\tSigStrength\n", temp);
}

// Again, this is another private method.  It places a formatted
// string containing the current time and date into the string
// pointed to by the argument of the method.
void PLog :: CurrentTime(char *copyto) 
{
	struct tm *gmcurtime;            // Struct from gmtime.
	time_t *curtime;                 // Current system time.

	gmcurtime = (struct tm *) malloc(sizeof(struct tm));
	curtime = (time_t *) malloc(sizeof(time_t));

	time(curtime);
	gmcurtime = gmtime(curtime);
	sprintf(copyto, "%02i/%02i/%i %02i:%02i:%02i GMT", gmcurtime->tm_mon+1, gmcurtime->tm_mday, gmcurtime->tm_year+1900, gmcurtime->tm_hour, gmcurtime->tm_min, gmcurtime->tm_sec);

}

// Add a hit to the log file.  The first argument is an
// unsigned long int, which pcrfreq_t is typedef'ed to, and
// the second argument is the signal strength, on a scale of
// 0 to 255.
void PLog :: LogFreq(pcrfreq_t curfreq, int sigstren)
{
	Freq2Str(temp, curfreq);
	fprintf(logfile_p, "%s\t", temp);
	CurrentTime(temp);
	fprintf(logfile_p, "%s\t", temp);
	fprintf(logfile_p, "%i\n", sigstren);
	fflush(logfile_p);
}

// Overloaded form of LogFreq.  Lets you specify no signal
// strength.  This should be eliminated later by a different
// function prototype (by setting a default value to sigstren),
// but no effect will be seen by the object caller.
void PLog :: LogFreq(pcrfreq_t curfreq)
{
	LogFreq(curfreq, 0);
}

// Private method to convert a frequency into something
// more human readable by placing a decimal point at
// the megahertz point.
void PLog :: Freq2Str(char *copyto, pcrfreq_t curfreq)
{
	bzero(copyto, sizeof(copyto));
	sprintf(copyto, "%i.%06i", curfreq/1000000, curfreq - ((int)(curfreq/1000000))*1000000);
}

// Similar to above.  Converts a human readable string form
// to a machine readable unsigned long int form for the frequency.
pcrfreq_t PLog :: Str2Freq(char *strfreq)
{
	pcrfreq_t tempfreq;
	char *mhz;
	char *khz;
	int i;

	mhz = strfreq;
	khz = strchr(strfreq, '.');

	if (khz == NULL) {
		if (strlen(mhz) > 4) {
			fprintf(stderr, "Malformed frequency: %s\n", strfreq);
			return 0;
		}
		tempfreq = atoi(mhz) * 1000000;
		return tempfreq;
	}
	
	*khz = '\0';
	khz++;
	tempfreq = atoi(khz);

	for (i = 5; i >= strlen(khz) ; i--) {
		tempfreq *= 10;
	}

	tempfreq += atoi(mhz) * 1000000;
	return tempfreq;
}

// Private method.  Counts the number of channels present in the
// log file.  The channel count can be returned to the caller by using
// int ChanCount()
void PLog :: CountChannels()
{
	if (readfile) {
		ChanNum = 0;
		while (fgets(temp, 80, logfile_p) != NULL) {
			if (strncmp(temp, "//", 2) != 0) {
				ChanNum++;
			}
		}
	}
	
	rewind(logfile_p);
}

// Private method to read the channels from the file into an array.
// This data can be returned later by using pcrfreq_t ReturnChan(int).
void PLog :: ReadChannels()
{
	char *freq, *tab;

	int i = 0;

	chanfreqs = (pcrfreq_t *) malloc(sizeof(pcrfreq_t) * ChanNum);

	while (fgets(temp, 80, logfile_p) != NULL) {
		if (strncmp(temp, "//", 2) != 0) {
			freq = temp;
			tab = strchr(temp, '\t');
			*tab = '\0';
			chanfreqs[i] = Str2Freq(freq);
	 		i++;
		}
	}
}

// Return the number of channels to the caller.
int PLog :: ChanCount()
{
	return ChanNum;
}

// Return a specific channel to the caller.
pcrfreq_t PLog :: ReturnChan(int CurChanNum)
{
	if (CurChanNum < 0 || CurChanNum > (ChanNum - 1)) {
		return 0;
	} 
	
	return chanfreqs[CurChanNum];
}
