#include <bickley/bkl-source-manager-client.h>
#include "hrn-source.h"
#include "hrn-source-manager.h"

enum {
    PROP_0,
};

enum {
    SOURCE_ADDED,
    SOURCE_REMOVED,
    READY,
    LAST_SIGNAL
};

struct _HrnSourceManagerPrivate {
    BklSourceManagerClient *manager;

    HrnSource *local_source;
    GList *transient_sources;

    guint source_count; /* The number of sources we have */
    guint source_replies; /* The number of replies (success or failure) that
                             we've received */
};

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), HRN_TYPE_SOURCE_MANAGER, HrnSourceManagerPrivate))
G_DEFINE_TYPE (HrnSourceManager, hrn_source_manager, G_TYPE_OBJECT);
static guint32 signals[LAST_SIGNAL] = {0, };

static void
hrn_source_manager_finalize (GObject *object)
{
    G_OBJECT_CLASS (hrn_source_manager_parent_class)->finalize (object);
}

static void
hrn_source_manager_dispose (GObject *object)
{
    G_OBJECT_CLASS (hrn_source_manager_parent_class)->dispose (object);
}

static void
hrn_source_manager_set_property (GObject      *object,
                                 guint         prop_id,
                                 const GValue *value,
                                 GParamSpec   *pspec)
{
    switch (prop_id) {

    default:
        break;
    }
}

static void
hrn_source_manager_get_property (GObject    *object,
                                 guint       prop_id,
                                 GValue     *value,
                                 GParamSpec *pspec)
{
    switch (prop_id) {

    default:
        break;
    }
}

static void
hrn_source_manager_class_init (HrnSourceManagerClass *klass)
{
    GObjectClass *o_class = (GObjectClass *) klass;

    o_class->dispose = hrn_source_manager_dispose;
    o_class->finalize = hrn_source_manager_finalize;
    o_class->set_property = hrn_source_manager_set_property;
    o_class->get_property = hrn_source_manager_get_property;

    g_type_class_add_private (klass, sizeof (HrnSourceManagerPrivate));

    signals[READY] = g_signal_new ("ready",
                                   G_TYPE_FROM_CLASS (klass),
                                   G_SIGNAL_RUN_FIRST |
                                   G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                   g_cclosure_marshal_VOID__VOID,
                                   G_TYPE_NONE, 0);
    signals[SOURCE_ADDED] = g_signal_new ("source-added",
                                          G_TYPE_FROM_CLASS (klass),
                                          G_SIGNAL_RUN_FIRST |
                                          G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                          g_cclosure_marshal_VOID__OBJECT,
                                          G_TYPE_NONE, 1, HRN_TYPE_SOURCE);
    signals[SOURCE_REMOVED] = g_signal_new ("source-removed",
                                            G_TYPE_FROM_CLASS (klass),
                                            G_SIGNAL_RUN_FIRST |
                                            G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                            g_cclosure_marshal_VOID__OBJECT,
                                            G_TYPE_NONE, 1, HRN_TYPE_SOURCE);
}

static void
check_all_sources_ready (HrnSourceManager *manager)
{
    HrnSourceManagerPrivate *priv = manager->priv;

    if (priv->source_replies == priv->source_count &&
        priv->local_source != NULL) {
        g_signal_emit (manager, signals[READY], 0);
    }
}

static void
source_ready_cb (BklSourceClient  *source_client,
                 HrnSourceManager *manager)
{
    HrnSourceManagerPrivate *priv = manager->priv;
    HrnSource *source;

    priv->source_replies++;

    source = hrn_source_new (source_client);
    if (source) {
        priv->transient_sources = g_list_prepend (priv->transient_sources,
                                                  source);
        g_signal_emit (manager, signals[SOURCE_ADDED], 0, source);
    } else {
        g_warning ("Could not create source for '%s'",
                   bkl_source_client_get_path (source_client));
    }

    /* HrnSource takes its own reference, we want to unref whether
       hrn_source_new() succeeded or not */
    g_object_unref (source_client);

    check_all_sources_ready (manager);
}

static void
source_added_cb (BklSourceManagerClient *client,
                 const char             *object_path,
                 HrnSourceManager       *manager)
{
    BklSourceClient *source;

    source = bkl_source_client_new (object_path);
    g_signal_connect (source, "ready", G_CALLBACK (source_ready_cb), manager);
}

static void
source_removed_cb (BklSourceManagerClient *client,
                   const char             *object_path,
                   HrnSourceManager       *manager)
{
    HrnSourceManagerPrivate *priv = manager->priv;
    GList *s;

    for (s = priv->transient_sources; s; s = s->next) {
        HrnSource *source = s->data;

        if (g_str_equal (object_path,
                         hrn_source_get_object_path (source))) {

            g_signal_emit (manager, signals[SOURCE_REMOVED], 0, source);

            g_object_unref (source);
            priv->transient_sources = g_list_delete_link (priv->transient_sources, s);
            break;
        }
    }
}

static void
local_source_ready (BklSourceClient  *client,
                    HrnSourceManager *manager)
{
    HrnSourceManagerPrivate *priv = manager->priv;

    priv->local_source = hrn_source_new (client);
    if (priv->local_source) {
        g_signal_emit (manager, signals[SOURCE_ADDED], 0, priv->local_source);
    } else {
        g_warning ("Could not create a source for '%s'",
                   bkl_source_client_get_path (client));
    }

    /* HrnSource takes its own reference, we want to unref whether
       hrn_source_new() succeeded or not */
    g_object_unref (client);

    check_all_sources_ready (manager);
}

static void
get_sources_reply (BklSourceManagerClient *client,
                   GList                  *sources,
                   GError                 *error,
                   gpointer                userdata)
{
    HrnSourceManager *manager = (HrnSourceManager *) userdata;
    HrnSourceManagerPrivate *priv = manager->priv;
    BklSourceClient *ls;
    GList           *s;

    if (error != NULL) {
        g_warning ("Error getting sources: %s", error->message);
        g_error_free (error);
    }

    ls = bkl_source_client_new (BKL_LOCAL_SOURCE_PATH);
    g_signal_connect (ls, "ready", G_CALLBACK (local_source_ready), manager);

    for (s = sources; s; s = s->next) {
        priv->source_count++;
        source_added_cb (client, s->data, manager);
    }

    g_list_free (sources);
}

static void
bkl_manager_ready_cb (BklSourceManagerClient *manager_client,
                      HrnSourceManager       *manager)
{
    HrnSourceManagerPrivate *priv = manager->priv;

    bkl_source_manager_client_get_sources (priv->manager,
                                           get_sources_reply, manager);
}


static void
hrn_source_manager_init (HrnSourceManager *self)
{
    HrnSourceManagerPrivate *priv = GET_PRIVATE (self);

    self->priv = priv;

    priv->manager = g_object_new (BKL_TYPE_SOURCE_MANAGER_CLIENT, NULL);
    g_signal_connect (priv->manager, "ready",
                      G_CALLBACK (bkl_manager_ready_cb), self);
    g_signal_connect (priv->manager, "source-added",
                      G_CALLBACK (source_added_cb), self);
    g_signal_connect (priv->manager, "source-removed",
                      G_CALLBACK (source_removed_cb), self);
}

HrnSource *
hrn_source_manager_get_local_source (HrnSourceManager *manager)
{
    HrnSourceManagerPrivate *priv = manager->priv;

    return priv->local_source;
}

GList *
hrn_source_manager_get_transient_sources (HrnSourceManager *manager)
{
    HrnSourceManagerPrivate *priv = manager->priv;

    return priv->transient_sources;
}

HrnSource *
hrn_source_manager_get_source_for_path (HrnSourceManager *manager,
                                        const char       *path)
{
    HrnSourceManagerPrivate *priv = manager->priv;
    GList *s;

    for (s = priv->transient_sources; s; s = s->next) {
        HrnSource *source = s->data;

        if (g_str_equal (path, hrn_source_get_object_path (source))) {
            return source;
        }
    }

    return NULL;
}

BklItem *
hrn_source_manager_get_item_for_uri (HrnSourceManager *manager,
                                     const char       *uri,
                                     HrnSource       **ret_source)
{
    HrnSourceManagerPrivate *priv = manager->priv;
    BklItem *item = NULL;
    GList   *s;

    /* Check local source first */
    item = hrn_source_get_item (priv->local_source, uri);
    if (item) {
        if (ret_source) {
            *ret_source = priv->local_source;
        }
        return item;
    }

    /* Check transients */
    for (s = priv->transient_sources; s; s = s->next) {
        HrnSource *source = s->data;

        item = hrn_source_get_item (source, uri);
        if (item) {
            if (ret_source) {
                *ret_source = source;
            }
            return item;
        }
    }

    if (ret_source) {
        *ret_source = NULL;
    }
    return NULL;
}

gboolean
hrn_source_manager_is_ready (HrnSourceManager *manager)
{
    HrnSourceManagerPrivate *priv = manager->priv;

    if (priv->source_replies == priv->source_count &&
        priv->local_source != NULL) {
        return TRUE;
    }

    return FALSE;
}
