/* hexter DSSI software synthesizer plugin
 *
 * Copyright (C) 2004, 2006 Sean Bolton and others.
 *
 * Portions of this file may have come from Juan Linietsky's
 * rx-saturno, copyright (C) 2002 by Juan Linietsky.
 * Portions of this file may have come from Jeff Harrington's
 * dx72csound package, for which I've found no copyright
 * attribution.
 * Significant reverse-engineering contributions were made by
 * Jamie Bullock.
 * Other parts may have been inspired by Chowning/Bristow,
 * Pinkston, Yala Abdullah's website, Godric Wilkie' website;
 * see the AUTHORS file for more detail.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307, USA.
 */

#define _BSD_SOURCE    1
#define _SVID_SOURCE   1
#define _ISOC99_SOURCE 1

#include <math.h>

#include "dx7_voice.h"

static int dx7_voice_tables_initialized = 0;
int32_t    dx7_voice_sin_table[SINE_SIZE + 1];

extern int32_t dx7_voice_eg_ol_to_mod_index_table[257]; /* forward */
extern int32_t dx7_voice_eg_ol_to_amp_table[257];       /* forward */

int32_t       *dx7_voice_eg_ol_to_mod_index = &dx7_voice_eg_ol_to_mod_index_table[128];
int32_t       *dx7_voice_eg_ol_to_amp       = &dx7_voice_eg_ol_to_amp_table[128];

void dx7_voice_init_tables(void) {

    int i;
    double f;

    if (!dx7_voice_tables_initialized) {

        for (i = 0; i <= SINE_SIZE; i++) {

            /* observation of my TX7's output with oscillator sync on suggests
             * it uses cosine */
            f = cos((double)(i) / SINE_SIZE * (2 * M_PI));  /* index / index max * radian cycle */
            dx7_voice_sin_table[i] = DOUBLE_TO_FP(f);
        }

#if FP_SHIFT != 24
        /* The fixed-point tables below are all in s7.24 format.  Shift
         * them to match FP_SHIFT. */
        for (i = 0; i <= 256; i++) {
            dx7_voice_eg_ol_to_mod_index_table[i] >>= (24 - FP_SHIFT);
            dx7_voice_eg_ol_to_amp_table[i]       >>= (24 - FP_SHIFT);
        }
#endif

        dx7_voice_tables_initialized = 1;
    }
}

/* This table lists which operators of an algorithm are carriers.  Bit 0 (LSB)
 * is set if operator 1 is a carrier, and so on through bit 5 for operator 6.
 */
uint8_t dx7_voice_carriers[32] = {
    0x05, /* algorithm 1, operators 1 and 3 */
    0x05,
    0x09, /* algorithm 3, operators 1 and 4 */
    0x09,
    0x15, /* algorithm 5, operators 1, 3, and 5 */
    0x15,
    0x05,
    0x05,
    0x05,
    0x09,
    0x09,
    0x05,
    0x05,
    0x05,
    0x05,
    0x01, /* algorithm 16, operator 1 */
    0x01,
    0x01,
    0x19, /* algorithm 19, operators 1, 4, and 5 */
    0x0b, /* algorithm 20, operators 1, 2, and 4 */
    0x1b, /* algorithm 21, operators 1, 2, 4, and 5 */
    0x1d, /* algorithm 22, operators 1, 3, 4, and 5 */
    0x1b,
    0x1f, /* algorithm 24, operators 1 through 5 */
    0x1f,
    0x0b,
    0x0b,
    0x25, /* algorithm 28, operators 1, 3, and 6 */
    0x17, /* algorithm 29, operators 1, 2, 3, and 5 */
    0x27, /* algorithm 30, operators 1, 2, 3, and 6 */
    0x1f,
    0x2f, /* algorithm 32, all operators */
};

float dx7_voice_carrier_count[32] = {
    2.0f, 2.0f, 2.0f, 2.0f, 3.0f, 3.0f, 2.0f, 2.0f,
    2.0f, 2.0f, 2.0f, 2.0f, 2.0f, 2.0f, 2.0f, 1.0f,
    1.0f, 1.0f, 3.0f, 3.0f, 4.0f, 4.0f, 4.0f, 5.0f,
    5.0f, 3.0f, 3.0f, 3.0f, 4.0f, 4.0f, 5.0f, 6.0f
};

float dx7_voice_eg_rate_rise_duration[128] = {  /* generated from my f04new */

     39.638000,  37.013000,  34.388000,  31.763000,  27.210500,
     22.658000,  20.408000,  18.158000,  15.908000,  14.557000,
     13.206000,  12.108333,  11.010667,   9.913000,   8.921000,
      7.929000,   7.171333,   6.413667,   5.656000,   5.307000,
      4.958000,   4.405667,   3.853333,   3.301000,   2.889000,
      2.477000,   2.313000,   2.149000,   1.985000,   1.700500,
      1.416000,   1.274333,   1.132667,   0.991000,   0.909000,
      0.827000,   0.758000,   0.689000,   0.620000,   0.558000,
      0.496000,   0.448667,   0.401333,   0.354000,   0.332000,
      0.310000,   0.275667,   0.241333,   0.207000,   0.180950,
      0.154900,   0.144567,   0.134233,   0.123900,   0.106200,
      0.088500,   0.079667,   0.070833,   0.062000,   0.056800,
      0.051600,   0.047300,   0.043000,   0.038700,   0.034800,
      0.030900,   0.028000,   0.025100,   0.022200,   0.020815,
      0.019430,   0.017237,   0.015043,   0.012850,   0.011230,
      0.009610,   0.009077,   0.008543,   0.008010,   0.006960,
      0.005910,   0.005357,   0.004803,   0.004250,   0.003960,
      0.003670,   0.003310,   0.002950,   0.002590,   0.002420,
      0.002250,   0.002000,   0.001749,   0.001499,   0.001443,
      0.001387,   0.001242,   0.001096,   0.000951,   0.000815,
      0.000815,   0.000815,   0.000815,   0.000815,   0.000815,
      0.000815,   0.000815,   0.000815,   0.000815,   0.000815,
      0.000815,   0.000815,   0.000815,   0.000815,   0.000815,
      0.000815,   0.000815,   0.000815,   0.000815,   0.000815,
      0.000815,   0.000815,   0.000815,   0.000815,   0.000815,
      0.000815,   0.000815,   0.000815

};

float dx7_voice_eg_rate_decay_duration[128] = {  /* generated from my f06new */

    317.487000, 285.764500, 254.042000, 229.857000, 205.672000,
    181.487000, 170.154000, 158.821000, 141.150667, 123.480333,
    105.810000,  98.382500,  90.955000,  81.804667,  72.654333,
     63.504000,  58.217000,  52.930000,  48.512333,  44.094667,
     39.677000,  33.089000,  26.501000,  24.283333,  22.065667,
     19.848000,  17.881500,  15.915000,  14.389667,  12.864333,
     11.339000,  10.641000,   9.943000,   8.833333,   7.723667,
      6.614000,   6.149500,   5.685000,   5.112667,   4.540333,
      3.968000,   3.639000,   3.310000,   3.033667,   2.757333,
      2.481000,   2.069500,   1.658000,   1.518667,   1.379333,
      1.240000,   1.116500,   0.993000,   0.898333,   0.803667,
      0.709000,   0.665500,   0.622000,   0.552667,   0.483333,
      0.414000,   0.384500,   0.355000,   0.319333,   0.283667,
      0.248000,   0.228000,   0.208000,   0.190600,   0.173200,
      0.155800,   0.129900,   0.104000,   0.095400,   0.086800,
      0.078200,   0.070350,   0.062500,   0.056600,   0.050700,
      0.044800,   0.042000,   0.039200,   0.034833,   0.030467,
      0.026100,   0.024250,   0.022400,   0.020147,   0.017893,
      0.015640,   0.014305,   0.012970,   0.011973,   0.010977,
      0.009980,   0.008310,   0.006640,   0.006190,   0.005740,
      0.005740,   0.005740,   0.005740,   0.005740,   0.005740,
      0.005740,   0.005740,   0.005740,   0.005740,   0.005740,
      0.005740,   0.005740,   0.005740,   0.005740,   0.005740,
      0.005740,   0.005740,   0.005740,   0.005740,   0.005740,
      0.005740,   0.005740,   0.005740,   0.005740,   0.005740,
      0.005740,   0.005740,   0.005740

};

float dx7_voice_eg_rate_decay_percent[128] = {  /* generated from P/H/Op f07 */

    0.000010, 0.025009, 0.050008, 0.075007, 0.100006,
    0.125005, 0.150004, 0.175003, 0.200002, 0.225001,
    0.250000, 0.260000, 0.270000, 0.280000, 0.290000,
    0.300000, 0.310000, 0.320000, 0.330000, 0.340000,
    0.350000, 0.358000, 0.366000, 0.374000, 0.382000,
    0.390000, 0.398000, 0.406000, 0.414000, 0.422000,
    0.430000, 0.439000, 0.448000, 0.457000, 0.466000,
    0.475000, 0.484000, 0.493000, 0.502000, 0.511000,
    0.520000, 0.527000, 0.534000, 0.541000, 0.548000,
    0.555000, 0.562000, 0.569000, 0.576000, 0.583000,
    0.590000, 0.601000, 0.612000, 0.623000, 0.634000,
    0.645000, 0.656000, 0.667000, 0.678000, 0.689000,
    0.700000, 0.707000, 0.714000, 0.721000, 0.728000,
    0.735000, 0.742000, 0.749000, 0.756000, 0.763000,
    0.770000, 0.777000, 0.784000, 0.791000, 0.798000,
    0.805000, 0.812000, 0.819000, 0.826000, 0.833000,
    0.840000, 0.848000, 0.856000, 0.864000, 0.872000,
    0.880000, 0.888000, 0.896000, 0.904000, 0.912000,
    0.920000, 0.928889, 0.937778, 0.946667, 0.955556,
    0.964444, 0.973333, 0.982222, 0.991111, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000

};

float dx7_voice_eg_rate_rise_percent[128] = {  /* checked, matches P/H/Op f05 */

    0.000010, 0.000010, 0.000010, 0.000010, 0.000010,
    0.000010, 0.000010, 0.000010, 0.000010, 0.000010,
    0.000010, 0.000010, 0.000010, 0.000010, 0.000010,
    0.000010, 0.000010, 0.000010, 0.000010, 0.000010,
    0.000010, 0.000010, 0.000010, 0.000010, 0.000010,
    0.000010, 0.000010, 0.000010, 0.000010, 0.000010,
    0.000010, 0.000010, 0.005007, 0.010005, 0.015003,
    0.020000, 0.028000, 0.036000, 0.044000, 0.052000,
    0.060000, 0.068000, 0.076000, 0.084000, 0.092000,
    0.100000, 0.108000, 0.116000, 0.124000, 0.132000,
    0.140000, 0.150000, 0.160000, 0.170000, 0.180000,
    0.190000, 0.200000, 0.210000, 0.220000, 0.230000,
    0.240000, 0.251000, 0.262000, 0.273000, 0.284000,
    0.295000, 0.306000, 0.317000, 0.328000, 0.339000,
    0.350000, 0.365000, 0.380000, 0.395000, 0.410000,
    0.425000, 0.440000, 0.455000, 0.470000, 0.485000,
    0.500000, 0.520000, 0.540000, 0.560000, 0.580000,
    0.600000, 0.620000, 0.640000, 0.660000, 0.680000,
    0.700000, 0.732000, 0.764000, 0.796000, 0.828000,
    0.860000, 0.895000, 0.930000, 0.965000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000, 1.000000, 1.000000,
    1.000000, 1.000000, 1.000000

};

/* This table converts an output level of 0 to 99 into a phase modulation
 * index of 0 to ~2.089 periods, expressed in s7.24 fixed point.  It
 * actually extends below 0 and beyond 99, since amplitude modulation can
 * produce 'negative' output levels, and velocities above 100 can produce
 * output levels above 99, plus it includes a 257th 'guard' point.
 * Table index 128 corresponds to output level 0, and index 227 to OL 99.
 * I believe this is based on information from the Chowning/Bristow
 * book (see the CREDITS file), filtered down to me through the work of
 * Pinkston, Harrington, and Abdullah as I found it on the Internet.  The
 * code used to calculate it looks something like this:
 *
 *    // DX7 output level to TL translation table
 *    int tl_table[128] = {
 *        127, 122, 118, 114, 110, 107, 104, 102, 100, 98, 96, 94, 92, 90,
 *        88, 86, 85, 84, 82, 81, 79, 78, 77, 76, 75, 74, 73, 72, 71,
 *        70, 69, 68, 67, 66, 65, 64, 63, 62, 61, 60, 59, 58, 57, 56, 55,
 *        54, 53, 52, 51, 50, 49, 48, 47, 46, 45, 44, 43, 42, 41, 40, 39,
 *        38, 37, 36, 35, 34, 33, 32, 31, 30, 29, 28, 27, 26, 25, 24, 23,
 *        22, 21, 20, 19, 18, 17, 16, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6,
 *        5, 4, 3, 2, 1, 0, -1, -2, -3, -4, -5, -6, -7, -8, -9, -10, -11,
 *        -12, -13, -14, -15, -16, -17, -18, -19, -20, -21, -22, -23, -24,
 *        -25, -26, -27, -28
 *    };
 *
 *    int ol;
 *    double mi;
 *
 *    for (ol = 0; ol < 128; ol++) {
 *        if (ol < 5) {    // smoothly ramp from 0.0 at 0 to the proper value at 5
 *            mi = pow(2.0, ( (33.0/16.0) - ((double)tl_table[5]/8.0) - 1.0));
 *            mi = mi * ((double)ol / 5.0);
 *        } else {
 *            mi = pow(2.0, ( (33.0/16.0) - ((double)tl_table[ol]/8.0) - 1.0));
 *        }
 *        printf(" %6d,", DOUBLE_TO_FP(mi));
 *    }
 */
int32_t dx7_voice_eg_ol_to_mod_index_table[257] = {

    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 659, 1319, 1978, 2638, 3298, 4277, 5086, 6049, 7193, 8554,
    10173, 12098, 14387, 17109, 20346, 22188, 24196, 28774, 31378,
    37315, 40693, 44376, 48392, 52772, 57548, 62757, 68437, 74631,
    81386, 88752, 96785, 105545, 115097, 125514, 136875, 149263,
    162772, 177504, 193570, 211090, 230195, 251029, 273750, 298526,
    325545, 355009, 387141, 422180, 460390, 502059, 547500, 597053,
    651091, 710019, 774282, 844360, 920781, 1004119, 1095000,
    1194106, 1302182, 1420039, 1548564, 1688721, 1841563, 2008239,
    2190000, 2388212, 2604364, 2840079, 3097128, 3377443, 3683127,
    4016479, 4380001, 4776425, 5208729, 5680159, 6194257, 6754886,
    7366255, 8032958, 8760003, 9552851, 10417458, 11360318,
    12388515, 13509772, 14732510, 16065917, 17520006, 19105702,
    20834916, 22720637, 24777031, 27019544, 29465021, 32131834,
    35040013, 38211405, 41669833, 45441275, 49554062, 54039088,
    58930043, 64263668, 70080027, 76422811, 83339667, 90882551,
    99108124, 108078176, 117860087, 128527336, 140160054, 152845623,
    166679334, 181765102, 198216249, 216156353, 235720174,
    257054673, 280320108, 305691246, 333358668, 363530205,
    396432499, 396432499

};

/* This table converts an output level of 0 to 99 into an output amplitude
 * scale factor of 0 to 1 in s7.24 fixed point.  It converts a linear 0-99
 * envelope to an exponential volume, and is just a normalized version of
 * dx7_voice_eg_ol_to_mod_index_table[] (see its description above). */
int32_t dx7_voice_eg_ol_to_amp_table[257] = {

    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 315, 631, 947, 1263, 1579, 2048, 2435, 2896, 3444, 4096,
    4870, 5792, 6888, 8192, 9741, 10623, 11585, 13777, 15024, 17866,
    19483, 21247, 23170, 25267, 27554, 30048, 32768, 35733, 38967,
    42494, 46340, 50535, 55108, 60096, 65536, 71467, 77935, 84989,
    92681, 101070, 110217, 120193, 131072, 142935, 155871, 169979,
    185363, 202140, 220435, 240387, 262144, 285870, 311743, 339958,
    370727, 404281, 440871, 480774, 524288, 571740, 623487, 679917,
    741455, 808562, 881743, 961548, 1048576, 1143480, 1246974,
    1359834, 1482910, 1617125, 1763487, 1923096, 2097152, 2286960,
    2493948, 2719669, 2965820, 3234250, 3526975, 3846193, 4194304,
    4573920, 4987896, 5439339, 5931641, 6468501, 7053950, 7692387,
    8388608, 9147841, 9975792, 10878678, 11863283, 12937002,
    14107900, 15384774, 16777216, 18295683, 19951584, 21757357,
    23726566, 25874004, 28215801, 30769549, 33554432, 36591367,
    39903169, 43514714, 47453132, 51748008, 56431603, 61539099,
    67108864, 73182735, 79806338, 87029429, 94906265, 103496016,
    112863206, 123078199, 134217728, 146365470, 159612677,
    174058858, 189812531, 189812531

};

/* This table lists the output level adjustment needed for a certain
 * velocity, expressed in output level units per unit of velocity
 * sensitivity. It is based on measurements I took from my TX7. */
float dx7_voice_velocity_ol_adjustment[128] = {

    -99.0,    -10.295511, -9.709229, -9.372207,
    -9.121093, -8.629703, -8.441805, -8.205647,
    -7.810857, -7.653259, -7.299901, -7.242308,
    -6.934396, -6.727051, -6.594723, -6.427755,
    -6.275133, -6.015212, -5.843023, -5.828787,
    -5.725659, -5.443202, -5.421110, -5.222133,
    -5.160615, -5.038265, -4.948225, -4.812105,
    -4.632120, -4.511531, -4.488645, -4.370043,
    -4.370610, -4.058591, -4.066902, -3.952988,
    -3.909686, -3.810096, -3.691883, -3.621306,
    -3.527286, -3.437519, -3.373512, -3.339195,
    -3.195983, -3.167622, -3.094788, -2.984045,
    -2.937463, -2.890713, -2.890660, -2.691874,
    -2.649229, -2.544696, -2.498147, -2.462573,
    -2.396637, -2.399795, -2.236338, -2.217625,
    -2.158336, -2.135569, -1.978521, -1.913965,
    -1.937082, -1.752275, -1.704013, -1.640514,
    -1.598791, -1.553859, -1.512187, -1.448088,
    -1.450443, -1.220567, -1.182340, -1.123139,
    -1.098469, -1.020642, -0.973039, -0.933279,
    -0.938035, -0.757380, -0.740860, -0.669721,
    -0.681526, -0.555390, -0.519321, -0.509318,
    -0.456936, -0.460622, -0.290578, -0.264393,
    -0.252716, -0.194141, -0.153566, -0.067842,
    -0.033402, -0.054947,  0.012860,  0.000000,
    -0.009715,  0.236054,  0.273956,  0.271968,
     0.330177,  0.345427,  0.352333,  0.433861,
     0.442952,  0.476411,  0.539632,  0.525355,
     0.526115,  0.707022,  0.701551,  0.734875,
     0.739149,  0.794320,  0.801578,  0.814225,
     0.818939,  0.897102,  0.895082,  0.927998,
     0.929797,  0.956112,  0.956789,  0.958121

};

/* This table converts pitch envelope level parameters into the
 * actual pitch shift in semitones.  For levels [17,85], this is
 * just ((level - 50) / 32 * 12), but at the outer edges the shift
 * is exagerated to 0 = -48 and 99 => 47.624.  This is based on
 * measurements I took from my TX7. */
double dx7_voice_pitch_level_to_shift[128] = {

    -48.000000, -43.497081, -38.995993, -35.626132, -31.873615,
    -28.495880, -25.500672, -22.872620, -20.998167, -19.496961,
    -18.373238, -17.251065, -16.122139, -15.375956, -14.624487,
    -13.876516, -13.126351, -12.375000, -12.000000, -11.625000,
    -11.250000, -10.875000, -10.500000, -10.125000, -9.750000,
    -9.375000, -9.000000, -8.625000, -8.250000, -7.875000,
    -7.500000, -7.125000, -6.750000, -6.375000, -6.000000,
    -5.625000, -5.250000, -4.875000, -4.500000, -4.125000,
    -3.750000, -3.375000, -3.000000, -2.625000, -2.250000,
    -1.875000, -1.500000, -1.125000, -0.750000, -0.375000, 0.000000,
    0.375000, 0.750000, 1.125000, 1.500000, 1.875000, 2.250000,
    2.625000, 3.000000, 3.375000, 3.750000, 4.125000, 4.500000,
    4.875000, 5.250000, 5.625000, 6.000000, 6.375000, 6.750000,
    7.125000, 7.500000, 7.875000, 8.250000, 8.625000, 9.000000,
    9.375000, 9.750000, 10.125000, 10.500000, 10.875000, 11.250000,
    11.625000, 12.000000, 12.375000, 12.750000, 13.125000,
    14.251187, 15.001922, 16.126327, 17.250917, 18.375718,
    19.877643, 21.753528, 24.373913, 27.378021, 30.748956,
    34.499234, 38.627888, 43.122335, 47.624065, 48.0, 48.0, 48.0,
    48.0, 48.0, 48.0, 48.0, 48.0, 48.0, 48.0, 48.0, 48.0, 48.0,
    48.0, 48.0, 48.0, 48.0, 48.0, 48.0, 48.0, 48.0, 48.0, 48.0,
    48.0, 48.0, 48.0, 48.0, 48.0

};

/* This table converts LFO speed to frequency in Hz. It is based on
 * interpolation of Jamie Bullock's measurements. */
float dx7_voice_lfo_frequency[128] = {
     0.062506,  0.124815,  0.311474,  0.435381,  0.619784,
     0.744396,  0.930495,  1.116390,  1.284220,  1.496880,
     1.567830,  1.738994,  1.910158,  2.081322,  2.252486,
     2.423650,  2.580668,  2.737686,  2.894704,  3.051722,
     3.208740,  3.366820,  3.524900,  3.682980,  3.841060,
     3.999140,  4.159420,  4.319700,  4.479980,  4.640260,
     4.800540,  4.953584,  5.106628,  5.259672,  5.412716,
     5.565760,  5.724918,  5.884076,  6.043234,  6.202392,
     6.361550,  6.520044,  6.678538,  6.837032,  6.995526,
     7.154020,  7.300500,  7.446980,  7.593460,  7.739940,
     7.886420,  8.020588,  8.154756,  8.288924,  8.423092,
     8.557260,  8.712624,  8.867988,  9.023352,  9.178716,
     9.334080,  9.669644, 10.005208, 10.340772, 10.676336,
    11.011900, 11.963680, 12.915460, 13.867240, 14.819020,
    15.770800, 16.640240, 17.509680, 18.379120, 19.248560,
    20.118000, 21.040700, 21.963400, 22.886100, 23.808800,
    24.731500, 25.759740, 26.787980, 27.816220, 28.844460,
    29.872700, 31.228200, 32.583700, 33.939200, 35.294700,
    36.650200, 37.812480, 38.974760, 40.137040, 41.299320,
    42.461600, 43.639800, 44.818000, 45.996200, 47.174400,
    47.174400, 47.174400, 47.174400, 47.174400, 47.174400,
    47.174400, 47.174400, 47.174400, 47.174400, 47.174400,
    47.174400, 47.174400, 47.174400, 47.174400, 47.174400,
    47.174400, 47.174400, 47.174400, 47.174400, 47.174400,
    47.174400, 47.174400, 47.174400, 47.174400, 47.174400,
    47.174400, 47.174400, 47.174400
};

/* This table converts pitch modulation sensitivity to semitones at full
 * modulation (assuming a perfectly linear pitch mod depth to pitch
 * relationship).  It is from a simple averaging of Jamie Bullock's
 * TX-data-1/PMD and TX-data-2/ENV data, and ignores the apparent ~0.1
 * semitone positive bias that Jamie observed. [-FIX- smbolton: my
 * inclination would be to call this bias, if it's reproducible, a
 * non-desirable 'bug', and _not_ implement it in hexter. And, at
 * least for my own personal build, I'd change that PMS=7 value to a
 * full octave, since that's one thing that's always bugged me about
 * my TX7.  Thoughts? ] */
float dx7_voice_pms_to_semitones[8] = {
    0.0, 0.450584, 0.900392, 1.474744,
    2.587385, 4.232292, 6.982097, /* 11.722111 */ 12.0
};

/* This table converts amplitude modulation depth to output level
 * reduction at full modulation with an amplitude modulation sensitivity
 * of 3.  It was constructed from regression of a very few data points,
 * using this code:
 *   perl -e 'for ($i = 0; $i <= 99; $i++) { printf " %f,\n", exp($i * 0.0428993 - 0.285189); }' >x.c
 * and is probably rather rough in its accuracy. -FIX- */
float dx7_voice_amd_to_ol_adjustment[100] = {
    0.0, 0.784829, 0.819230, 0.855139, 0.892622, 0.931748,
    0.972589, 1.015221, 1.059721, 1.106171, 1.154658, 1.205270,
    1.258100, 1.313246, 1.370809, 1.430896, 1.493616, 1.559085,
    1.627424, 1.698759, 1.773220, 1.850945, 1.932077, 2.016765,
    2.105166, 2.197441, 2.293761, 2.394303, 2.499252, 2.608801,
    2.723152, 2.842515, 2.967111, 3.097167, 3.232925, 3.374633,
    3.522552, 3.676956, 3.838127, 4.006362, 4.181972, 4.365280,
    4.556622, 4.756352, 4.964836, 5.182458, 5.409620, 5.646738,
    5.894251, 6.152612, 6.422298, 6.703805, 6.997652, 7.304378,
    7.624549, 7.958754, 8.307609, 8.671754, 9.051861, 9.448629,
    9.862789, 10.295103, 10.746365, 11.217408, 11.709099,
    12.222341, 12.758080, 13.317302, 13.901036, 14.510357,
    15.146387, 15.810295, 16.503304, 17.226690, 17.981783,
    18.769975, 19.592715, 20.451518, 21.347965, 22.283705,
    23.260462, 24.280032, 25.344294, 26.455204, 27.614809,
    28.825243, 30.088734, 31.407606, 32.784289, 34.221315,
    35.721330, 37.287095, 38.921492, 40.627529, 42.408347,
    44.267222, 46.207578, 48.232984, 50.347169, 52.75
};

/* This table converts modulation source sensitivity (e.g. 'foot
 * controller sensitivity') into output level reduction at full modulation
 * with amplitude modulation sensitivity 3.  It's basically just the above
 * table scaled for 0 to 15 instead of 0 to 99. */
float dx7_voice_mss_to_ol_adjustment[16] = {
    0.0, 0.997948, 1.324562, 1.758071, 2.333461, 3.097167, 4.110823,
    5.456233, 7.241976, 9.612164, 12.758080, 16.933606, 22.475719,
    29.831681, 39.595137, 52.75
};
