// ****************************************************************************
//  Project:        GUYMAGER
// ****************************************************************************
//  Programmer:     Guy Voncken
//                  Police Grand-Ducale
//                  Service de Police Judiciaire
//                  Section Nouvelles Technologies
// ****************************************************************************
//  Module:         Thread for reading data.
// ****************************************************************************

#include <errno.h>

#include <QtCore>

#include "common.h"
#include "config.h"
#include "device.h"
#include "threadread.h"
#include "threadwrite.h"

#include "fcntl.h"
//##define _GNU_SOURCE

const int THREADREAD_CHECK_CONNECTED_SLEEP = 1000;
const int THREADREAD_SLOWDOWN_SLEEP        =  700;


class t_ThreadReadLocal
{
   public:
      t_pDevice     pDevice;
      bool         *pSlowDownRequest;
};

t_ThreadRead::t_ThreadRead(void)
{
   CHK_EXIT (ERROR_THREADREAD_CONSTRUCTOR_NOT_SUPPORTED)
} //lint !e1401 not initialised

t_ThreadRead::t_ThreadRead (t_pDevice pDevice, bool *pSlowDownRequest)
{
   static bool Initialised = false;

   if (!Initialised)
   {
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADREAD_FILESRC_ALREADY_OPEN     ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADREAD_NO_DATA                  ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADREAD_DEVICE_DISCONNECTED      ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADREAD_UNEXPECTED_FAILURE       ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADREAD_CONSTRUCTOR_NOT_SUPPORTED))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADREAD_BLOCKSIZE                ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADREAD_DEVICE_ABORTED           ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADREAD_LIBEWF_FAILED            ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADREAD_BAD_FILE_HANDLE          ))
      Initialised = true;
   }

   pOwn = new t_ThreadReadLocal;
   pOwn->pDevice          = pDevice;
   pOwn->pSlowDownRequest = pSlowDownRequest;

   CHK_QT_EXIT (connect (this, SIGNAL(finished()), this, SLOT(SlotFinished())))
}

t_ThreadRead::~t_ThreadRead (void)
{
   delete pOwn;
}

inline bool ThreadReadEOF (t_pcDevice pDevice)
{
//   LOG_INFO ("EOF check %Ld %Ld", pDevice->CurrentPos, pDevice->Size)
   return (pDevice->CurrentReadPos >= pDevice->Size);
}

static APIRET ThreadReadCheckDeviceExists (t_pcDevice pDevice, bool &DeviceExists)
{
   FILE *pFileTest;

   pFileTest = fopen64 (QSTR_TO_PSZ(pDevice->LinuxDevice), "r");
   DeviceExists = (pFileTest != NULL);
   if (DeviceExists)
      (void) fclose (pFileTest);

   return NO_ERROR;
}

static APIRET ThreadReadDeviceDisconnected (t_pDevice pDevice)
{
   if (pDevice->pFileSrc)
   {
      (void) fclose (pDevice->pFileSrc);
      pDevice->pFileSrc = NULL;
   }
   if (pDevice->State == t_Device::Acquire)
        pDevice->State = t_Device::AcquirePaused;
   else pDevice->State = t_Device::VerifyPaused;
   LOG_INFO ("Device %s disconnected, switching device state to %s", QSTR_TO_PSZ(pDevice->LinuxDevice), pDevice->StateStr())

   return ERROR_THREADREAD_DEVICE_DISCONNECTED;
}

static APIRET ThreadReadAdjustSeek (t_pDevice pDevice)        // According to the GNU Clib docu, a seek should be done after read errors
{
   int rc;

   if (pDevice->pFileSrc == NULL)
      CHK (ERROR_THREADREAD_FILESRC_NOT_OPEN)

   if (ThreadReadEOF (pDevice))
      return NO_ERROR;

   rc = fseeko64 (pDevice->pFileSrc, (off64_t)pDevice->CurrentReadPos, SEEK_SET);
   if (rc)
      CHK_RET (ThreadReadDeviceDisconnected (pDevice))

   return NO_ERROR;
}

//static APIRET ThreadReadDirectMode (FILE *pFileSrc, bool Enable)
//{
//   int OldFlags, NewFlags, rc;
//   int FileDescriptor;
//
//   FileDescriptor = fileno (pFileSrc);
//   if (FileDescriptor == -1)
//      CHK (ERROR_THREADREAD_BAD_FILE_HANDLE)
//
//   OldFlags = fcntl (FileDescriptor, F_GETFL);
//   if (OldFlags > 0)
//   {
//      NewFlags = (Enable ? (OldFlags |  O_DIRECT)
//                         : (OldFlags & ~O_DIRECT));
//      if (NewFlags != OldFlags)
//      {
//         rc = fcntl (FileDescriptor, F_SETFL, NewFlags);
//         if (rc == -1)
//            LOG_INFO ("Setting flags with fcntl failed. Direct mode state unknown.")
//      }
//      LOG_INFO ("Direct mode %s", Enable ? "enabled" : "disabled")
//   }
//   else
//   {
//      LOG_INFO ("Reading flags with fcntl failed. Direct mode remains unchanged")
//   }
//
//   return NO_ERROR;
//}

static APIRET ThreadReadBlock0 (t_pDevice pDevice, unsigned char *pBuffer, unsigned int Size)
{
   bool DeviceExists;
   int  Read;

   errno = 0;
   if (pDevice->pFileSrc == NULL)
   {
      pDevice->pFileSrc = fopen64 (QSTR_TO_PSZ(pDevice->LinuxDevice), "r");
      if (!pDevice->pFileSrc)
         CHK_RET (ThreadReadDeviceDisconnected (pDevice))
      CHK_RET (ThreadReadAdjustSeek (pDevice))

//      CHK (ThreadReadDirectMode (pDevice->pFileSrc, true))
   }

   Read = (int) fread (pBuffer, Size, 1, pDevice->pFileSrc); //lint !e732 !e712 loss of sign/precision arg 2
   // ev. zusaetzlich ferror abfragen

// Bad sector simulation
//   if ((pDevice->CurrentReadPos > 500000) && (pDevice->CurrentReadPos < 600000) ||
//       (pDevice->CurrentReadPos > 700000) && (pDevice->CurrentReadPos < 740000) ||
//       (pDevice->CurrentReadPos == 0) || (pDevice->CurrentReadPos == 512) || (pDevice->CurrentReadPos == 1024))
//      Read = 0;

   switch (Read)
   {
      case  1: pDevice->CurrentReadPos += Size;
               return NO_ERROR;

      case  0:
//               LOG_INFO ("Read error at position %lld", pDevice->CurrentReadPos)

               CHK (ThreadReadCheckDeviceExists (pDevice, DeviceExists))
               if (DeviceExists)
               {
//                  LOG_DEBUG ("Device still exists")
                  CHK_RET (ThreadReadAdjustSeek (pDevice))
//                  LOG_DEBUG ("Seek adjusted")
                  return ERROR_THREADREAD_NO_DATA;
               }
               else
               {
                  CHK_RET (ThreadReadDeviceDisconnected (pDevice))
               }
               break;

      default: CHK_CONST (ERROR_THREADREAD_UNEXPECTED_FAILURE)
   }

   return NO_ERROR;
}


APIRET t_ThreadRead::ThreadReadBlock (t_pFifoBlock &pFifoBlock)
{
   t_pDevice      pDevice;
   unsigned char *pBuffer;
   quint64         BytesToRead;
   quint64         BytesRead = 0;
   quint64         RemainingSize;
   quint64         Sector;
   quint64         PrevReadPos;
   unsigned int    ReadTry;
   APIRET          RcRead;
   APIRET          RcSeek;

   pDevice       = pOwn->pDevice;
   PrevReadPos   = pDevice->CurrentReadPos;
   RemainingSize = pDevice->Size - pDevice->CurrentReadPos;
   if ((quint64)pDevice->FifoBlockSize > RemainingSize)
        BytesToRead = RemainingSize;
   else BytesToRead = pDevice->FifoBlockSize;

   if (BytesToRead > INT_MAX)
      CHK (ERROR_THREADREAD_BLOCKSIZE)

   if (pDevice->HasCompressionThreads())
        CHK (t_Fifo::CreateCompressionOptimised (pFifoBlock, pDevice->FifoBlockSize))
   else CHK (t_Fifo::Create (pFifoBlock, (unsigned int) BytesToRead))

   pFifoBlock->DataSize = (unsigned int) BytesToRead;
   pBuffer = pFifoBlock->Buffer;

   if (pDevice->FallbackMode)
        ReadTry = pDevice->SectorSize;
   else ReadTry = BytesToRead;
   do
   {
      RcRead = ThreadReadBlock0 (pDevice, pBuffer, ReadTry);
      switch (RcRead)
      {
         case NO_ERROR:
            BytesRead += ReadTry;
            pBuffer   += ReadTry;
            break;

         case ERROR_THREADREAD_NO_DATA:
            if (!pDevice->FallbackMode)
            {
               pDevice->FallbackMode = true;
//               ReadTry = (int) std::min (pDevice->SectorSize, pDevice->SectorSizePhys);
               ReadTry = pDevice->SectorSize;
//               CHK (ThreadReadDirectMode (pDevice->pFileSrc, true))
               LOG_INFO ("Device read error, switching %s to slow fallback mode. Reading sectors individually from now on.", QSTR_TO_PSZ(pDevice->LinuxDevice))
            }
            else
            {
               Sector = pDevice->CurrentReadPos / pDevice->SectorSize;
               pDevice->CurrentReadPos += ReadTry;
               RcSeek = ThreadReadAdjustSeek (pDevice); // If the seek still succeeds, then there's probably just a bad sector on
               switch (RcSeek)                          // the source device, let's replace it by a zero sector.
               {
                  case NO_ERROR:
                     if ((pDevice->Acquisition.Format == t_File::AFF) && (CONFIG (AffMarkBadSectors)))
                          CHK (AaffCopyBadSectorMarker (pBuffer, ReadTry))
                     else memset (pBuffer, 0, ReadTry);
                     BytesRead += ReadTry;
                     pBuffer   += ReadTry;
                     LOG_INFO ("Read error on %s, sector %Ld, bad data replaced by zero block", QSTR_TO_PSZ(pDevice->LinuxDevice), Sector)
                     pDevice->AddBadSector (Sector);
                     break;
                  case ERROR_THREADREAD_DEVICE_DISCONNECTED:
                     break;
                  default:
                     CHK (RcSeek)
               }
            }
            break;

         case ERROR_THREADREAD_DEVICE_DISCONNECTED:
            break;

         default:
            CHK (RcRead)
      }
   } while ((BytesRead < BytesToRead) && (pDevice->State != t_Device::AcquirePaused)
                                      && (pDevice->State != t_Device::VerifyPaused )
                                      && !pDevice->AbortRequest);

   if ((pDevice->State == t_Device::AcquirePaused) ||
       (pDevice->State == t_Device::VerifyPaused ))
   {
      CHK (t_Fifo::Destroy (pFifoBlock))
      pDevice->CurrentReadPos = PrevReadPos;

      return ERROR_THREADREAD_DEVICE_DISCONNECTED;
   }
   else if (pDevice->AbortRequest)
   {
      CHK (t_Fifo::Destroy (pFifoBlock))

      return ERROR_THREADREAD_DEVICE_ABORTED;
   }
   else if ((pDevice->FallbackMode) && (RcRead == NO_ERROR))
   {
      pDevice->FallbackMode = false;  // Switch to fast reading mode if last read was good
      LOG_INFO ("Device read ok again. Switching %s to fast block read.", QSTR_TO_PSZ(pDevice->LinuxDevice))
//      CHK (ThreadReadDirectMode (pDevice->pFileSrc, false))
   }

   pFifoBlock->LastBlock = ThreadReadEOF (pDevice);

   return NO_ERROR;
}

void t_ThreadRead::run (void)
{
   t_pDevice          pDevice;
   t_pFifoBlock       pFifoBlock;
   APIRET              rc;
   int                 rcf;
   bool                CalcHashes;
   quint64           *pBlocks;
   quint64             BlocksRead     = 0;
   quint64             BlocksVerified = 0;
   quint64             BytesRead      = 0;
   quint64             BytesVerified  = 0;
   t_HashContextMD5    HashContextMD5;
   t_HashContextSHA256 HashContextSHA256;

   LOG_INFO ("Acquisition of %s: Reading thread started", QSTR_TO_PSZ (pOwn->pDevice->LinuxDevice))
   pDevice = pOwn->pDevice;
   CalcHashes = pDevice->Acquisition.CalcHashes && !CONFIG (UseSeparateHashThread);
   pBlocks = &BlocksRead;
   pDevice->State = t_Device::Acquire;
   for (;;)
   {
      pDevice->pFileSrc       = NULL;
      pDevice->CurrentReadPos = 0;
      pDevice->FallbackMode   = false;
      *pBlocks = 0;
      if (CalcHashes)
      {
         CHK_EXIT (HashMD5Init    (&HashContextMD5   ))
         CHK_EXIT (HashSHA256Init (&HashContextSHA256))
      }
      do
      {
         if (*(pOwn->pSlowDownRequest))
            msleep (THREADREAD_SLOWDOWN_SLEEP);

         rc = ThreadReadBlock (pFifoBlock);
         switch (rc)
         {
            case NO_ERROR:
               pFifoBlock->Nr = (*pBlocks);

               if (CalcHashes)
               {
                  CHK_EXIT (HashMD5Append    (&HashContextMD5   , pFifoBlock->Buffer, pFifoBlock->DataSize))
                  CHK_EXIT (HashSHA256Append (&HashContextSHA256, pFifoBlock->Buffer, pFifoBlock->DataSize))
               }
               if ((pDevice->State == t_Device::Verify) && CalcHashes)
               {
                  pDevice->IncCurrentVerifyPos(pFifoBlock->DataSize);
                  CHK_EXIT (t_Fifo::Destroy (pFifoBlock))
               }
               else
               {
                  CHK_EXIT (pDevice->pFifoRead->Insert (pFifoBlock))
               }
               (*pBlocks)++;
               break;

            case ERROR_THREADREAD_DEVICE_DISCONNECTED:
               while (!pDevice->AbortRequest && ((pDevice->State == t_Device::AcquirePaused) ||
                                                 (pDevice->State == t_Device::VerifyPaused )) )
                  msleep (THREADREAD_CHECK_CONNECTED_SLEEP);
               break;

            case ERROR_THREADREAD_DEVICE_ABORTED:
               break;
            default:
               LOG_ERROR ("Unexpected return code ThreadReadBlock")
               CHK_EXIT (rc)
         }
      } while (!ThreadReadEOF(pDevice) && !pDevice->AbortRequest);

      if (pDevice->State == t_Device::Acquire)
           BytesRead     = pDevice->CurrentReadPos;
      else BytesVerified = pDevice->CurrentReadPos;

      if (pDevice->AbortRequest)
         break;

      if (pDevice->State == t_Device::Verify)
      {
         if (CalcHashes)
         {
            CHK_EXIT (HashMD5Digest    (&HashContextMD5   , &pDevice->MD5DigestVerify   ))
            CHK_EXIT (HashSHA256Digest (&HashContextSHA256, &pDevice->SHA256DigestVerify))
         }
         LOG_INFO ("Source verification completed (device %s)", QSTR_TO_PSZ (pDevice->LinuxDevice))
         break;
      }

      if (CalcHashes)
      {
         CHK_EXIT (HashMD5Digest    (&HashContextMD5   , &pDevice->MD5Digest   ))
         CHK_EXIT (HashSHA256Digest (&HashContextSHA256, &pDevice->SHA256Digest))
      }

      if (pDevice->Acquisition.VerifySource)
      {
         (void) fclose (pDevice->pFileSrc);
         LOG_INFO ("All data has been read (device %s), now re-reading for source verification", QSTR_TO_PSZ (pDevice->LinuxDevice))
         pDevice->State = t_Device::Verify;
         pDevice->StartTimestampVerify = QDateTime::currentDateTime();
         pBlocks = &BlocksVerified;
         if (!CalcHashes)
            CHK_EXIT (pDevice->pFifoRead->InsertDummy ())
      }
      else
      {
         LOG_INFO ("All data has been read (device %s)", QSTR_TO_PSZ (pDevice->LinuxDevice))
         break;
      }
   }

   if (pDevice->pFileSrc)
   {
      rcf = fclose (pDevice->pFileSrc);
      pDevice->pFileSrc = NULL;
      if (rcf)
      {
         LOG_INFO ("Unexpected error on fclose, errno = %d", errno)
         pDevice->AbortReason  = t_Device::ThreadReadFileError;
         pDevice->AbortRequest = true;
      }
   }

   LOG_INFO ("Reading thread exits now (device %s, %Ld blocks read, %llu bytes)", QSTR_TO_PSZ (pDevice->LinuxDevice), BlocksRead, BytesRead)
   if (pDevice->Acquisition.VerifySource)
   LOG_INFO ("                    (for the verification: %Ld blocks read, %llu bytes)", BlocksVerified, BytesVerified)
}

void t_ThreadRead::SlotFinished (void)
{
   emit SignalEnded (pOwn->pDevice);
}

