/* ----------------------------------------------------------------------------
 * audio.c
 * funtions for play audio samples
 *
 * Copyright 2002 Matthias Grimm
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 * ----------------------------------------------------------------------------*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#ifdef HAVE_SOUND

#include <pthread.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <malloc.h>
#include <linux/soundcard.h>
#include <audiofile.h>
#include <time.h>

#include "audio.h"

struct sample*
load_sample (char *samplename)
{
	AFfilehandle affd;     /* filehandle for soundfile from libaudiofile */
	AFframecount framecount;
	int dummy, channels, byteorder, framesize, precision, err = 0;
	struct sample *sample;

	if ((sample = (struct sample *) malloc(sizeof(struct sample))) == NULL)
		return NULL;

	if ((affd = afOpenFile(samplename, "r", 0))) {
		afGetSampleFormat(affd, AF_DEFAULT_TRACK, &dummy, &precision);
		channels = afGetChannels(affd, AF_DEFAULT_TRACK);
		byteorder = afGetVirtualByteOrder(affd, AF_DEFAULT_TRACK);
		framesize = (int) afGetFrameSize(affd, AF_DEFAULT_TRACK, 0);
		framecount = afGetFrameCount(affd, AF_DEFAULT_TRACK);
		sample->speed = (int) afGetRate(affd, AF_DEFAULT_TRACK);

		if (channels <= 2)
			sample->stereo = (channels == 2);
		else
			err = -1;

		if (precision == 8)
			sample->format = AFMT_U8;
		else if (precision == 16) {
			if (byteorder == AF_BYTEORDER_LITTLEENDIAN)
				sample->format = AFMT_S16_LE;
			else
				sample->format = AFMT_S16_BE;
		} else
			err = -1;

		if (err == 0) {
			sample->audiodatalen = framecount * framesize;
			if ((sample->audiodata = (char *) malloc(sample->audiodatalen))) {
				if ((afReadFrames(affd, AF_DEFAULT_TRACK, sample->audiodata, framecount)) != framecount) {
					free(sample->audiodata);
					err = -1;
				}
			}
		}
		afCloseFile(affd);
	} else
		err = -1;

	if (err == -1) {
		free(sample);
		return NULL;
	}

	return sample;
}

void
cleanup_audio(struct dspdata *dsp)
{
	if (dsp->active) {
		pthread_mutex_destroy(&dsp->mutex);
		pthread_cond_destroy(&dsp->cond);
	}
}

/* This function ist the sound playing thread.
 */
void *
play_sample (void *arg)
{
	struct dspdata *dsp = (struct dspdata *) arg;
	int fd = -1, err = -1;
	int playlen, t;
	char *playptr;

	dsp->playbeep = 0;
	for (;;) {
		if (dsp->playbeep == 0) {
			pthread_mutex_lock(&dsp->mutex);
			pthread_cond_wait(&dsp->cond, &dsp->mutex);
			pthread_mutex_unlock(&dsp->mutex);
		}
		dsp->playbeep = 0;
		if (fd == -1)
			if ((fd = open(dsp->audiodev, O_WRONLY, 0)) > 0)
				if ((err = ioctl(fd, SNDCTL_DSP_RESET, 0)) == 0)
					if ((err = ioctl(fd, SNDCTL_DSP_SETFMT, &dsp->sample->format)) == 0)
						if ((err = ioctl(fd, SNDCTL_DSP_STEREO, &dsp->sample->stereo)) == 0)
							if ((err = ioctl(fd, SNDCTL_DSP_SPEED, &dsp->sample->speed)) == 0)
								if ((err = ioctl(fd, SNDCTL_DSP_GETBLKSIZE, &dsp->maxblocksize)) == 0)
									if (dsp->maxblocksize > 32768)
										dsp->maxblocksize = 32768; /* limit blocksize */

		if (err == 0) {
			playptr = dsp->sample->audiodata;
			playlen = dsp->sample->audiodatalen;
			do {
				t = playlen < dsp->maxblocksize ? playlen : dsp->maxblocksize;
				if (write(fd, playptr, t) != t) {
					break;
				}
				playptr += t; playlen -= t;
			} while (playlen > 0);

			if (dsp->playbeep == 0) { /* no other request received? */
				close(fd);            /* ok, then close the audio device */
				fd = -1;
			} else
				ioctl(fd, SNDCTL_DSP_SYNC, 0); /* clear audio buffer */

			/* Closing or syncing the alsa 0.5 awacs sounddriver
			 * causes the speakers to crackle. We can't do
			 * anything against that because it is a problem in
			 * the reset routine of the alsa sounddrivers. There
			 * is no problem wirh the kernel OSS sounddriver.
			 */
		}
	}
	pthread_exit(NULL);
}

/* This function wakes the sound playing thread to do
 * a beep. If the audio subsystem is disabled
 * (dsp->active = 0) it does nothing.
 */
void
trigger_beep(struct dspdata *dsp, struct sample *sample)
{
	if (dsp->active)        /* audio subsystem active and valid? */
		if ((dsp->sample = sample)) {  /* is there a sample to play? */
			pthread_mutex_lock(&dsp->mutex);   /* ok, give me one ping */
			dsp->playbeep = 1;
			pthread_cond_signal(&dsp->cond);
			pthread_mutex_unlock(&dsp->mutex);
		}
}

/* This function sets up the sound playing thread.
 * It starts the thread or if an error occur cleans
 * up all the audio stuff
 */
int
init_sound_thread(struct dspdata *dsp)
{
	pthread_attr_t attr;
	int rc = 0;

	dsp->maxblocksize = 0;
	dsp->thread = 0;
	pthread_mutex_init(&dsp->mutex, NULL);
	pthread_cond_init (&dsp->cond, NULL);
	pthread_attr_init(&attr);
	pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
	if ((pthread_create(&dsp->thread, &attr, play_sample, (void *) dsp)) != 0) {
		cleanup_audio(dsp);
		rc = -1;
	}
	pthread_attr_destroy(&attr);
	return rc;
}

#endif
