/* gtask.c
 *
 * Copyright (C) 2008 Christian Hergert <chris@dronelabs.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston MA 
 * 02110-1301 USA
 */

/**
 * SECTION:gtask
 * @short_description: Asynchronous and callback-based execution
 *
 * #GTask is a #GObject sublcass that encapsulates a work item.  Tasks
 * remove the concept of a standard stack, but more on that in a minute.
 *
 * The power of #GTask comes from the callback and errback handling
 * strategy.  You can, ahead of time, create complex callback chains
 * for handling normal and extraordinary flow within your application.
 *
 * Callbacks are run sequentially after task execution has completed. You
 * may also have errbacks, which are like a callback but used to handle
 * an error in task execution or previous callback.
 *
 * If a callback creates an error, no more callbacks will be executed
 * until we reach another errback.
 * 
 * This documentation sucks, I know it.  I'll come back later and improve
 * it.  Read the section on twisted deferreds for now.
 */

#include "gtask.h"
#include "gtaskpriv.h"
#include "gtaskmarshal.h"

G_DEFINE_TYPE (GTask, g_task, G_TYPE_OBJECT);

enum {
	STATE_CHANGED,
	LAST_SIGNAL
};

enum {
	PROP_0,
	PROP_STATE
};

static gulong signals[LAST_SIGNAL] = { 0, };

GQuark
g_task_error_quark (void)
{
	return g_quark_from_static_string ("g_task_error-quark");
}

/**
 * g_task_new:
 * @func: A #GTaskFunc to be executed during g_task_execute()
 * @func_data: The user data for @func
 * @notify: The destroy notification for @func_data
 *
 * Creates a new instance of #GTask and sets the #GTaskFunc to use when
 * g_task_execute() is called.  @func will be called from a thread.
 *
 * Return value: A new instance of #GTask.
 */
GTask*
g_task_new (GTaskFunc func, gpointer func_data, GDestroyNotify notify)
{
	GClosure *closure = g_cclosure_new (G_CALLBACK (func), func_data, (GClosureNotify)notify);
	g_closure_set_marshal (closure, _gtask_marshal_POINTER__VOID);
	return g_task_new_from_closure (closure);
}

/**
 * g_task_new_from_closure:
 * @closure: A #GClosure
 *
 * Creates a new instance of #GTask and sets up the task to invoke
 * @closure when the task is executed. The closure will be called
 * from a thread.
 *
 * Return value: A new instance of #GTask.
 */
GTask*
g_task_new_from_closure (GClosure *closure)
{
	GTask *self = g_object_new (G_TYPE_TASK, NULL);

	if (closure)
	{
		self->priv->closure = g_closure_ref (closure);
		g_closure_sink (closure);
	}

	return self;
}

/*
 * Handles the result from a closure call.  The inner value will
 * be unwrapped and inspected for a new task to execute.  If one
 * was found (meaning there was no real result) the task will
 * be unwrapped and returned.
 *
 * If a non-GTask value was returned, the current value for the
 * task will be updated.
 */
static GTask*
_g_task_handle_closure_result (GTask *self, const GValue *result)
{
	if (!result)
		return NULL;

	/* 
	 * The GValue should always be a pointer type.  It should point
	 * to another GValue for which we own the reference on.  Inside
	 * that value is either a GTask or the result of the task.
	 */

	GValue *real_result = NULL;

	if (G_VALUE_HOLDS_POINTER (result))
	{
		real_result = g_value_get_pointer (result);

		if (real_result && G_VALUE_HOLDS_OBJECT (real_result))
		{
			GObject *object_v = g_value_get_object (real_result);
			if (object_v && G_IS_TASK (object_v))
			{
				/* unset the current result */
				g_task_set_result (self, NULL);

				/* get a copy of the GTask */
				object_v = g_object_ref (object_v);

				/* free the value (its a pointer anyway) */
				g_value_unset (real_result);

				/* clean up dynamically allocated value */
				g_free (real_result);

				return G_TASK (object_v);
			}
		}

		g_task_set_result (self, real_result);

		if (real_result)
		{
			g_value_unset (real_result);
			g_free (real_result);
		}
	}

	return NULL;
}

/**
 * g_task_execute:
 * @self: A #GTask
 * @error: A location to return an error
 * @see_also: g_task_callback()
 *
 * Synchronously performs the task's worker delegate.  Tasks that have more
 * work to complete will return a new #GTask.  If they have completed then
 * NULL is returned.
 *
 * Scheduler implementors are highly encouraged to immediately execute
 * the potentially returned task to reduce scheduling overhead.
 *
 * It is also the #GTaskScheduler<!-- -->s responsibility to invoke the callback
 * chain after execution via g_task_callback().
 *
 * Return value: a new #GTask to perform if additional work is required or NULL.
 */
GTask*
g_task_execute (GTask *self)
{
	g_return_val_if_fail (G_IS_TASK (self), NULL);

	GValue *result_value = NULL;
	GTask  *result_task  = NULL;

	if (G_UNLIKELY (!G_TASK_GET_CLASS (self)->execute))
		return NULL;

	result_value = G_TASK_GET_CLASS (self)->execute (self);
	result_task = _g_task_handle_closure_result (self, result_value);

	if (!self->priv->is_async)
		g_task_set_state (self, G_TASK_CALLBACKS);

	g_value_unset (result_value);
	g_free (result_value);

	return result_task;
}

/**
 * g_task_set_result:
 * @self: A #GTask
 * @value: a #GValue which will be copied
 *
 * Sets the current result for the task.  The value is copied and stored
 * to be passed as a parameter to the next callback.
 */
void
g_task_set_result (GTask *self, const GValue *value)
{
	g_return_if_fail (G_IS_TASK (self));

	g_mutex_lock (self->priv->mutex);

	if (self->priv->result && G_VALUE_TYPE (self->priv->result) != 0)
		g_value_unset (self->priv->result);
	else if (!self->priv->result)
		self->priv->result = g_new0 (GValue, 1);

	if (value && G_VALUE_TYPE (value))
	{
		g_value_init (self->priv->result, G_VALUE_TYPE (value));
		g_value_copy (value, self->priv->result);
	}

	g_mutex_unlock (self->priv->mutex);
}

/**
 * g_task_get_result:
 * @self: A #GTask
 *
 * Retrieves the current value for the task.  The value should not be
 * modified or freed.
 *
 * Return value: The current result.
 */
const GValue*
g_task_get_result (GTask *self)
{
	g_return_val_if_fail (G_IS_TASK (self), NULL);

	g_mutex_lock (self->priv->mutex);
	GValue *result = self->priv->result;
	g_mutex_unlock (self->priv->mutex);

	return result;
}

/**
 * g_task_cancel:
 * @self: A #GTask
 *
 * Cancels a task.  If the task has not started, it will not be executed
 * and tasks depending on it will be notified.  If the task is being
 * executed, it is up to the executing worker delegate to check for the
 * cancellation request and terminate execution.
 *
 * Callbacks and errbacks may still perform if they handle the error that
 * will be created for the task.
 */
void
g_task_cancel (GTask *self)
{
	g_return_if_fail (G_IS_TASK (self));

	if (self->priv->state != G_TASK_WAITING &&
	    self->priv->state != G_TASK_READY   &&
	    self->priv->state != G_TASK_EXECUTING)
		return;

	g_task_set_state (self, G_TASK_CANCELLED);
}

/**
 * g_task_get_state:
 * @self: A #GTask
 *
 * Retrieves the current state of a task.
 *
 * Return value: a #GTaskState representing the current state.
 */
GTaskState
g_task_get_state (GTask *self)
{
	g_return_val_if_fail (G_IS_TASK (self), -1);

	g_mutex_lock (self->priv->mutex);
	GTaskState state = self->priv->state;
	g_mutex_unlock (self->priv->mutex);

	return state;
}

/**
 * g_task_get_async:
 * @self: A #GTask
 *
 * Whether or not this task is asynchronous.
 *
 * Return value: TRUE if the task is asynchronous
 */
gboolean
g_task_get_async (GTask *self)
{
	g_return_val_if_fail (G_IS_TASK (self), FALSE);

	return self->priv->is_async;
}

/**
 * g_task_set_async:
 * @self: A #GTask
 * @is_async: if the task is asynchronous
 *
 * Asynchronous tasks are tasks whose execution closure will return
 * immediately but not be finished processing until later.
 *
 * Note that asynchronous tasks need to update their state to
 * @G_TASK_FINISHED using g_task_set_state().
 */
void
g_task_set_async (GTask *self, gboolean is_async)
{
	g_return_if_fail (G_IS_TASK (self));

	g_mutex_lock (self->priv->mutex);

	/* you can't change async state after its started executing */
	if (self->priv->state == G_TASK_READY || self->priv->state == G_TASK_WAITING)
		self->priv->is_async = is_async;

	g_mutex_unlock (self->priv->mutex);
}

/*
 * This method is attached to a dependent tasks state-changed
 * signal.  It is used to dispatch tasks when dependent tasks
 * have finished.
 */
static void
_g_task_on_state_changed (GTask      *dependency,
                          GTaskState  state,
                          gpointer    user_data)
{
	g_return_if_fail (G_IS_TASK (dependency));
	g_return_if_fail (G_IS_TASK (user_data));

	GTask *self = G_TASK (user_data);

	switch (state) {
	case G_TASK_FINISHED:
		g_task_remove_dependency (self, dependency);
		break;
	case G_TASK_CANCELLED:
		g_task_cancel (self);
		break;
	case G_TASK_CALLBACKS:
	case G_TASK_WAITING:
	case G_TASK_READY:
	case G_TASK_EXECUTING:
	default:
		break;
	}
}

/**
 * g_task_add_dependency:
 * @self: A #GTask
 * @dependency: A #GTask which must complete before execution
 *
 * Adds a #GTask to the list of tasks that must be completed before this
 * task may execute.
 */
void
g_task_add_dependency (GTask *self, GTask *dependency)
{
	g_return_if_fail (G_IS_TASK (self));
	g_return_if_fail (G_IS_TASK (dependency));

	if (self->priv->state > G_TASK_READY)
		return;
	else if (g_task_get_state (dependency) == G_TASK_FINISHED)
		return;

	g_mutex_lock (self->priv->mutex);

	self->priv->dependencies = g_list_prepend (self->priv->dependencies,
	                                           g_object_ref (dependency));
	self->priv->state = G_TASK_WAITING;

	g_mutex_unlock (self->priv->mutex);

	g_signal_connect (dependency, "state-changed",
	                  G_CALLBACK (_g_task_on_state_changed), self);
}

/**
 * g_task_remove_dependency:
 * @self: A #GTask
 * @dependency: A #GTask that is currently a dependency of @self
 *
 * Removes a currently dependent task. If all dependencies have been
 * met, the tasks state will move to @G_TASK_READY.
 *
 * If the task has a scheduler associated with it (priv->scheduler)
 * then the task will be re-scheduled for execution on that scheduler.
 */
void
g_task_remove_dependency (GTask *self, GTask *dependency)
{
	g_return_if_fail (G_IS_TASK (self));
	g_return_if_fail (G_IS_TASK (dependency));

	g_mutex_lock (self->priv->mutex);
	self->priv->dependencies = g_list_remove (self->priv->dependencies, dependency);
	g_mutex_unlock (self->priv->mutex);

	g_signal_handlers_disconnect_by_func (dependency, "state-changed", self);
	g_object_unref (dependency);

	if (!self->priv->dependencies && self->priv->state == G_TASK_WAITING)
		g_task_set_state (self, G_TASK_READY);

	if (self->priv->state == G_TASK_READY && self->priv->scheduler)
		g_task_scheduler_schedule (self->priv->scheduler, self);
}

/*
 * Adds a task handler to the callback/errback execution chain.  This is used
 * to DRY the addition process from errback and callback addition methods.
 */
static GTask*
_g_task_add_handler (GTask *self, GClosure *callback, GClosure *errback)
{
	g_return_val_if_fail (G_IS_TASK (self), NULL);

	GTaskHandler *handler = g_new0 (GTaskHandler, 1);

	if (callback)
	{
		handler->callback = g_closure_ref (callback);
		g_closure_sink (callback);
	}

	if (errback)
	{
		handler->errback = g_closure_ref (errback);
		g_closure_sink (errback);
	}

	self->priv->handlers = g_list_append (self->priv->handlers, handler);

	return self;
}

/**
 * g_task_add_callback:
 * @self: A #GTask
 * @callback: A #GTaskFunc
 * @user_data: user data for @callback
 * @notify: destroy notification for @user_data
 *
 * Adds a callback handler to the processing chain.  This is executed
 * in turn during the callback phase.  If the callback generates an error,
 * the first error handler left in the chain will run.  If the callback does
 * not generate an error, further callbacks may be called.
 */
void
g_task_add_callback (GTask          *self,
                     GTaskCallback   callback,
                     gpointer        user_data,
		     GDestroyNotify  notify)
{
	GClosure *closure = g_cclosure_new (G_CALLBACK (callback), user_data, (GClosureNotify)notify);
	g_closure_set_marshal (closure, _gtask_marshal_POINTER__POINTER);
	_g_task_add_handler (self, closure, NULL);
}

/**
 * g_task_add_callback_closure:
 * @self: A #GTask
 * @closure: A #GClosure to execute
 *
 * This method is similar to g_task_add_callback() except it allows
 * the use of a closure.  Binding authors should use this method to
 * allow for greater control and flexibility.
 */
void
g_task_add_callback_closure (GTask *self, GClosure *closure)
{
	_g_task_add_handler (self, closure, NULL);
}

/**
 * g_task_add_errback:
 * @self: A #GTask
 * @errback: A #GTaskFunc
 * @user_data: user data for @errback
 * @notify: destroy notification for @user_data
 *
 * Adds an errback handler to the processing chain.  This is executed in
 * turn during the callbacks phase.  If the errback unsets the error, then
 * callbacks added after the errback may continue to process.  If the
 * errback does not unset the error, further errback handlers may be
 * processed.
 *
 * To unset an error during an errback, use g_task_set_error() with
 * a NULL error.
 */
void
g_task_add_errback (GTask          *self,
                    GTaskErrback    errback,
                    gpointer        user_data,
		    GDestroyNotify  notify)
{
	GClosure *closure = g_cclosure_new (G_CALLBACK (errback), user_data, (GClosureNotify)notify);
	g_closure_set_marshal (closure, _gtask_marshal_POINTER__POINTER);
	_g_task_add_handler (self, NULL, closure);
}

/**
 * g_task_add_errback_closure:
 * @self: A #GTask
 * @closure: A #GClosure
 *
 * This method is similar to g_task_add_errback() except it allows
 * the use of a closure.  Binding authors should be using this method
 * to allow for greater control and flexibility.
 */
void
g_task_add_errback_closure (GTask *self, GClosure *closure)
{
	_g_task_add_handler (self, NULL, closure);
}

/**
 * g_task_add_both:
 * @self: A #GTask
 * @callback: A #GTaskCallback
 * @errback: A #GTaskErrback
 * @user_data: user data for @callback and @errback
 * @notify: destroy notification for @user_data
 *
 * Adds both an errback and a callback to a handler in the callbacks phase.
 * This means that one of these methods is garunteed to be executed during
 * the callbacks phase.  When the handler is reached, if the task currently
 * has an error, @errback will be called, otherwise, @callback will be called.
 */
void
g_task_add_both (GTask          *self,
                 GTaskCallback   callback,
                 GTaskErrback    errback,
		 gpointer        user_data,
		 GDestroyNotify  notify)
{
	GClosure *callback_closure = g_cclosure_new (G_CALLBACK (callback), user_data, (GClosureNotify)notify);
	GClosure *errback_closure = g_cclosure_new (G_CALLBACK (errback), user_data, (GClosureNotify)notify);

	g_closure_set_marshal (callback_closure, _gtask_marshal_POINTER__POINTER);
	g_closure_set_marshal (errback_closure, _gtask_marshal_POINTER__POINTER);

	_g_task_add_handler (self, callback_closure, errback_closure);
}

/**
 * g_task_add_both_closure:
 * @self: A #GTask
 * @callback: a #GClosure for the GTaskCallback.
 * @errback: a #GClosure for the GTaskErrback.
 *
 * This method is identical to g_task_add_both() except it allows the use
 * of closures directly.
 *
 * Use this for simple branching control of exceptions.
 */
void
g_task_add_both_closure (GTask *self, GClosure *callback, GClosure *errback)
{
	_g_task_add_handler (self, callback, errback);
}

/**
 * g_task_set_state:
 * @self: A #GTask
 * @state: The new #GTaskState for @self
 *
 * Updates the current state of a #GTask.  Changing the state will
 * emit the state-changed signal.
 */
void
g_task_set_state (GTask *self, GTaskState state)
{
	g_return_if_fail (G_IS_TASK (self));

	if (self->priv->state == state)
		return;

	g_mutex_lock (self->priv->mutex);
	self->priv->state = state;
	g_mutex_unlock (self->priv->mutex);

	g_signal_emit (self, signals[STATE_CHANGED], 0, state);
}

/*
 * Looses our reference on a callback or errback handler. The memory
 * consumed by handler is freed.
 */
static void
_g_task_handler_free (GTaskHandler *handler)
{
	if (handler->callback)
		g_closure_unref (handler->callback);
	if (handler->errback)
		g_closure_unref (handler->errback);
	g_free (handler);
}

/*
 * Executes a closure expecting a closure signature of a #GTaskFunc.
 * 
 * Return value: the result of the closure, unmodified, and owned by the
 *   caller.
 */
static GValue*
_g_task_func_closure_invoke (GTask *self, GClosure *closure)
{
	g_return_val_if_fail (G_IS_TASK (self), NULL);

	GValue *return_value = g_new0 (GValue, 1);
	GValue  param_values[1] = {{0}};

	g_value_init (&param_values[0], G_TYPE_OBJECT);
	g_value_set_object (&param_values[0], self);

	/* pointer, because its encapsulated. (gvalue<gvalue>) */
	g_value_init (return_value, G_TYPE_POINTER);

	g_closure_invoke (closure, return_value, 1, param_values, NULL);

	return return_value;
}

/*
 * Executes a closure expecting a closure signature of a #GTaskCallback.
 *
 * Return value: the result of the closure, unmodified, and owned by the
 *   caller.
 */
static GValue*
_g_task_callback_closure_invoke (GTask *self, GClosure *closure)
{
	g_return_val_if_fail (G_IS_TASK (self), NULL);

	GValue *return_value = g_new0 (GValue, 1);
	GValue  param_values[2] = {{0}, {0}};

	g_value_init (return_value, G_TYPE_POINTER);

	g_value_init (&param_values[0], G_TYPE_OBJECT);
	g_value_set_object (&param_values[0], self);

	g_value_init (&param_values[1], G_TYPE_POINTER);
	g_value_set_pointer (&param_values[1], (gpointer)g_task_get_result (self));

	g_closure_invoke (closure, return_value, 2, &param_values[0], NULL);

	return return_value;
}

/*
 * Executes a closure expecting a closure signature of a #GTaskErrback.
 *
 * Return value: the result of the closure, unmodified, and owned by the
 *   caller.
 */
static GValue*
_g_task_errback_closure_invoke (GTask *self, GClosure *closure)
{
	g_return_val_if_fail (G_IS_TASK (self), NULL);

	GValue *return_value = g_new0 (GValue, 1);
	GValue  param_values[2] = {{0}, {0}};

	g_value_init (return_value, G_TYPE_POINTER);

	g_value_init (&param_values[0], G_TYPE_OBJECT);
	g_value_set_object (&param_values[0], self);

	g_assert (self->priv->error != NULL);

	g_value_init (&param_values[1], G_TYPE_POINTER);
	g_value_set_pointer (&param_values[1], (gpointer)g_task_get_error (self));

	g_closure_invoke (closure, return_value, 2, param_values, NULL);

	return return_value;
}

/**
 * g_task_callback:
 * @self: A #GTask
 *
 * Works towards completing the execution of the callback chain. If the
 * task contains an error, we will try to resolve the error by jumping
 * to the next errback handler.
 *
 * If a callback or errback return a new #GTask to execute, the callback
 * chain will pause and wait for this new task to be completed.
 *
 * When the resulted task has been completed, the scheduler implementation
 * should recall this method to work towards finishing the execution of
 * the callbacks.
 *
 * Return value: A new #GTask or NULL if the callback chain has completed.
 */
GTask*
g_task_callback (GTask *self)
{
	g_return_val_if_fail (G_IS_TASK (self), NULL);

	GTask        *result_task  = NULL;
	GValue       *result_value = NULL;
	GTaskHandler *handler      = NULL;

	g_return_val_if_fail (self->priv->state == G_TASK_FINISHED  ||
	                      self->priv->state == G_TASK_CALLBACKS ||
	                      self->priv->state == G_TASK_CANCELLED,
	                      NULL);

	if (self->priv->state == G_TASK_FINISHED || self->priv->state == G_TASK_CANCELLED)
		g_task_set_state (self, G_TASK_CALLBACKS);

	while (G_LIKELY (self->priv->handlers != NULL))
	{
		gboolean has_error = g_task_has_error (self);

		if (has_error)
			handler = g_task_move_next_errback (self);
		else
			handler = g_task_move_next_callback (self);

		if (G_LIKELY (handler))
		{
			if (has_error)
				result_value = _g_task_errback_closure_invoke (self, handler->errback);
			else
				result_value = _g_task_callback_closure_invoke (self, handler->callback);

			result_task = _g_task_handle_closure_result (self, result_value);

			_g_task_handler_free (handler);
			g_free (result_value);

			if (result_task)
				return result_task;
		}
		else break;
	}

	/* NOTE: Propagate our error or our result to the parent task
	 *   if there is a parent task. */
	if (self->priv->previous)
	{
		if (self->priv->error)
			g_task_set_error (self->priv->previous, self->priv->error);
		else
			g_task_set_result (self->priv->previous, g_task_get_result (self));
	}

	return NULL;
}

/*
 * Retrieves the current handler and increments the list to the next
 * position.  The caller owns the reference and must free it with
 * g_free().
 *
 * Return value: a caller owned #GTaskHandler or NULL.
 */
static GTaskHandler*
_g_task_move_next_unlocked (GTask *self)
{
	g_return_val_if_fail (G_IS_TASK (self), NULL);
	GTaskHandler *handler = NULL;

	if (!self->priv->handlers)
		return NULL;

	handler = self->priv->handlers->data;
	self->priv->handlers = g_list_delete_link (self->priv->handlers, self->priv->handlers);

	return handler;
}

/**
 * g_task_move_next_callback:
 * @self: A #GTask
 *
 * This method will move to the next handler which is a callback
 * task handler.  The newly current #GTaskHandler will be returned.
 *
 * Return value: The next #GTaskHandler that is a callback or NULL.
 */
GTaskHandler*
g_task_move_next_callback (GTask *self)
{
	g_return_val_if_fail (G_IS_TASK (self), NULL);

	GTaskHandler *handler = NULL;

	if (!self->priv->handlers)
		return NULL;

	g_mutex_lock (self->priv->mutex);

	while (NULL != (handler = _g_task_move_next_unlocked (self)))
	{
		if (handler->callback)
			break;
		_g_task_handler_free (handler);
	}

	g_mutex_unlock (self->priv->mutex);

	return handler;
}

/**
 * g_task_move_next_errback:
 * @self: A #GTask
 *
 * This method will move to the next handler that is an errback.
 * The newly current #GTaskHandler will be returned.
 *
 * Return value: The next #GTaskHandler which is an errback or NULL.
 */
GTaskHandler*
g_task_move_next_errback (GTask *self)
{
	g_return_val_if_fail (G_IS_TASK (self), NULL);
	GTaskHandler *handler = NULL;

	if (!self->priv->handlers)
		return NULL;

	g_mutex_lock (self->priv->mutex);

	while (NULL != (handler = _g_task_move_next_unlocked (self)))
	{
		if (handler->errback)
			break;
		_g_task_handler_free (handler);
	}

	g_mutex_unlock (self->priv->mutex);

	return handler;
}

/**
 * _g_task_execute_real:
 * @self: A #GTask
 * @error: a location for an error
 *
 * The default execute implementation for a #GTask.  It handles both
 * synchronous and asynchronous task implementations.
 */
static GValue*
_g_task_execute_real (GTask *self)
{
	g_return_val_if_fail (G_IS_TASK (self), NULL);
	return _g_task_func_closure_invoke (self, self->priv->closure);
}

/**
 * g_task_has_error:
 * @self: A #GTask.
 *
 * This method checks to see if there is currently an error for the task.
 *
 * Return value: TRUE if there is an error
 */
gboolean
g_task_has_error (GTask *self)
{
	g_return_val_if_fail (G_IS_TASK (self), FALSE);
	return self->priv->error != NULL;
}

/**
 * g_task_get_error:
 * @self: A #GTask.
 *
 * This method will retrieve the current error for the task.  It should
 * not be modified directly.
 *
 * Return value: A #GError that should not be modified.
 */
G_CONST_RETURN GError*
g_task_get_error (GTask *self)
{
	g_return_val_if_fail (G_IS_TASK (self), NULL);
	return self->priv->error;
}

/**
 * g_task_set_error:
 * @self: A #GTask
 * @error: a #GError
 *
 * Sets the current error for the task.  @error is copied locally.
 */
void
g_task_set_error (GTask *self, const GError *error)
{
	g_return_if_fail (G_IS_TASK (self));

	g_mutex_lock (self->priv->mutex);

	if (self->priv->error)
		g_error_free (self->priv->error);

	if (error)
		self->priv->error = g_error_copy (error);
	else
		self->priv->error = NULL;

	g_mutex_unlock (self->priv->mutex);
}

/**
 * g_task_take_error:
 * @self: A #GTask
 * @error: A #GError
 *
 * Like g_task_set_error() but the error is used directly and steals
 * ownership of the error.
 */
void
g_task_take_error (GTask *self, GError *error)
{
	g_return_if_fail (G_IS_TASK (self));

	g_mutex_lock (self->priv->mutex);
	self->priv->error = error;
	g_mutex_unlock (self->priv->mutex);
}

static void
g_task_get_property (GObject    *object,
                     guint       property_id,
                     GValue     *value,
                     GParamSpec *pspec)
{
	switch (property_id) {
	case PROP_STATE:
		g_value_set_enum (value, g_task_get_state (G_TASK (object)));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
	}
}

static void
g_task_set_property (GObject      *object,
                     guint         property_id,
                     const GValue *value,
                     GParamSpec   *pspec)
{
	switch (property_id) {
	case PROP_STATE:
		g_task_set_state (G_TASK (object), g_value_get_enum (value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
	}
}

static void
g_task_dispose (GObject *object)
{
	g_return_if_fail (G_IS_TASK (object));

	GTask *self = G_TASK (object);

	if (G_LIKELY (self->priv->previous))
		g_object_unref (self->priv->previous);

	if (G_LIKELY (self->priv->closure))
		g_closure_unref (self->priv->closure);

	if (G_LIKELY (self->priv->scheduler))
		g_object_unref (self->priv->scheduler);

	if (G_OBJECT_CLASS (g_task_parent_class)->dispose)
		G_OBJECT_CLASS (g_task_parent_class)->dispose (object);
}

static void
g_task_finalize (GObject *object)
{
	g_return_if_fail (G_IS_TASK (object));

	if (G_TASK (object)->priv->error)
		g_error_free (G_TASK (object)->priv->error);

	G_OBJECT_CLASS (g_task_parent_class)->finalize (object);
}

static void
g_task_class_init (GTaskClass *task_class)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (task_class);

	g_type_class_add_private (task_class, sizeof (GTaskPrivate));

	task_class->execute         = _g_task_execute_real;
	gobject_class->get_property = g_task_get_property;
	gobject_class->set_property = g_task_set_property;
	gobject_class->dispose      = g_task_dispose;
	gobject_class->finalize     = g_task_finalize;

	/**
	 * GTask:state:
	 *
	 * The current task state.
	 */
	g_object_class_install_property (gobject_class,
	                                 PROP_STATE,
	                                 g_param_spec_enum ("state",
	                                                    "State",
	                                                    "Current state of the task",
	                                                    G_TYPE_TASK_STATE,
	                                                    G_TASK_READY,
	                                                    G_PARAM_READWRITE));

	/**
	 * GTask::state-changed:
	 * @task: the object which received the signal
	 * @state: the new #GTaskState
	 *
	 * The state-changed signal is emitted when a task moves into a new
	 * state.  This is particularly useful if you want to alter the state
	 * of dependent tasks when this task changes state.
	 */
	signals[STATE_CHANGED] = g_signal_new (g_intern_static_string ("state-changed"),
	                                       G_TYPE_FROM_CLASS (task_class),
	                                       G_SIGNAL_RUN_FIRST,
	                                       0, NULL, NULL,
	                                       g_cclosure_marshal_VOID__INT,
	                                       G_TYPE_NONE,
	                                       1, G_TYPE_INT);
}

static void
g_task_init (GTask *self)
{
	self->priv = G_TYPE_INSTANCE_GET_PRIVATE (self, G_TYPE_TASK, GTaskPrivate);
	self->priv->mutex = g_mutex_new ();
	self->priv->state = G_TASK_READY;
}

GType
g_task_state_get_type (void)
{
	static GType g_task_state_type_id = 0;
	if (G_UNLIKELY (g_task_state_type_id == 0))
	{
		static const GEnumValue values[] = {
			{G_TASK_WAITING,   "G_TASK_WAITING",   "waiting"},
			{G_TASK_READY,     "G_TASK_READY",     "ready"},
			{G_TASK_EXECUTING, "G_TASK_EXECUTING", "executing"},
			{G_TASK_CALLBACKS, "G_TASK_CALLBACKS", "callbacks"},
			{G_TASK_FINISHED,  "G_TASK_FINISHED",  "finished"},
			{G_TASK_CANCELLED, "G_TASK_CANCELLED", "cancelled"},
			{0, NULL, NULL}
		};
		g_task_state_type_id = g_enum_register_static ("GTaskState", values);
	}
	return g_task_state_type_id;
}
