/* gt_athena.c - widget routines for X version of etalk
 *
 * Copyright (C) 1997, 1999 Free Software Foundation
 * Copyright (C) 1995, 1996 Eric M. Ludlam
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 * 
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 * 
 * Please send bug reports, etc. to zappo@gnu.org.
 * 
 * Description:
 *   These routines manage the athena widget set specific interface,
 * where _x handles those routines which would be shared between
 * widget sets.  An _motif or _openwin may someday exist if there is
 * demand for it.
 *   
 * $Log: gt_athena.c,v $
 * Revision 1.20  1999/08/26 11:55:45  zappo
 * Added Ctxt to fix_user_windows.  Fixed all occurances of XRead to pass
 * down Ctxt.
 *
 * Revision 1.19  1998/01/04 13:30:23  zappo
 * Removed prototype for a function that doesn't exist.
 *
 * Revision 1.18  1997/12/14 19:14:49  zappo
 * Renamed package to gtalk, renamed symbols and files apropriately
 * Fixed copyright and email address.
 *
 * Revision 1.17  1997/10/25 02:44:03  zappo
 * Replaced the read w/ a flush in set_echo_label. Prevents recursion
 * and messing up the order of execution of typed characters.
 *
 * Revision 1.16  1997/10/23 01:57:14  zappo
 * Moved paste-code (except adding the translation) into the generic Xt
 * code in gt_x.c (and made it work there.)
 *
 * Revision 1.15  1997/10/18 02:46:13  zappo
 * Commented out the realize widget to fix startup size problems
 *
 * Revision 1.14  1997/09/19 19:06:31  zappo
 * Popup shell now appears over the etalk window.
 *
 * Revision 1.13  1997/07/27 17:14:23  zappo
 * Changed to use the generic list for all windows.
 *
 * Revision 1.12  1997/03/23 17:17:06  zappo
 * Fixed initial sizing problem
 *
 * Revision 1.11  1997/03/21 12:33:50  zappo
 * Fixed font identification problems.
 *
 * Revision 1.10  1997/03/21 12:09:56  zappo
 * Moved font decoding and sizing into this file.
 *
 * Revision 1.9  1997/02/23 03:20:37  zappo
 * Turned off autofill when creating a popup.
 *
 * Revision 1.8  1997/02/20 02:33:26  zappo
 * When menu items are highlighted, help text is printed in the minibuffer.
 *
 * Revision 1.7  1997/02/01  14:29:56  zappo
 * Changed minibuffer to be text widget instead of a label.
 * Added extra showcursor parameter to XW_set_echo_label, and the cursor
 * is turned off when it is false.
 * Added new translation/Action proc so menu buttons display help in the
 * minibuffer when that button is selected.
 *
 * Revision 1.6  1997/01/29  12:36:27  zappo
 * For the label under a text field, the colors are auto-reversed from
 * whatever the default resources are.  This way a user can specify
 * colors for just the pane, and the label comes out inverted.
 *
 * Revision 1.5  1996/02/26  00:23:27  zappo
 * fill-column code now works, and behaves like the curses when it comes
 * to wrapping words.  Added code for fonts so the window is always the
 * rights size.
 *
 * Revision 1.4  1995/12/09  23:26:23  zappo
 * Added a clear window command for startup blurb clearing
 *
 * Revision 1.3  1995/09/29  08:24:37  zappo
 * now uses DISP_window_displayed to flag out-of data windows, UserPanes
 * are now labeled after who they are connected to.  This allows the
 * setting of Xresources to modify the colors of individuals.
 *
 * History:
 * zappo   9/16/95    Created
 *
 * Tokens: ::Header:: gtalkx.h
 */
#include <X11/Xlib.h>

#include "gtalklib.h"
#include "gtalkc.h"
#include "gtalkx.h"

#include <X11/IntrinsicP.h>

#include <X11/Xaw/Dialog.h>

#include <X11/Xaw/Paned.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Form.h>

#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/AsciiTextP.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>

/* Now for some static local variables... */
static Widget echo_area = NULL;
static Widget gtalk_pane, etalk_pane, menu_box;
static Widget PopupText;

static void XW_show_message();


/*
 * Function: XW_build_mainwindow
 *
 *   Creates the top level windows needed to run etalk in X window mode.
 *
 * Returns:     Nothing
 * Parameters:  None
 *
 * History:
 * zappo   9/16/95    Created
 */
void XW_build_mainwindow(XCtxt)
     struct Xcontext *XCtxt;	/* x context to initialize */
{
  XFontStruct *font;
  static XtActionsRec act[] =
  { { "show-message", (XtActionProc)XW_show_message } };

  XtAppAddActions(XCtxt->app_context, act, sizeof(act)/sizeof(XtActionsRec));

  /* Now, initialize the shells we need to put our text widgets into */
  gtalk_pane = 
    XtVaCreateManagedWidget("GtalkPane", panedWidgetClass, XCtxt->topLevel,
			    XtNorientation, XtorientVertical,
			    NULL);

  /* Create a menu up on the top. */
  menu_box =
    XtVaCreateManagedWidget("MenuBox", boxWidgetClass, gtalk_pane,
			    XtNallowResize,       True,
			    XtNresizeToPreferred, True,
			    XtNshowGrip,          False,
			    XtNmin,               10,
			    XtNorientation,       XtorientHorizontal,
			    NULL);

  /* Make the menus */
  X_make_menus(menu_box);

  /* Create the form in which we will stuff the text widgets... */
  etalk_pane = 
    XtVaCreateWidget("EtalkPane", panedWidgetClass, gtalk_pane,
		     XtNallowResize,       True,
		     XtNresizeToPreferred, True,
		     XtNshowGrip,          False,
		     XtNwidth,             1,
		     XtNheight,            1,
		     NULL);

  /* Start with the echo area */
  echo_area = XtVaCreateManagedWidget("EchoArea", asciiTextWidgetClass,
				      gtalk_pane,
				      XtNallowResize,       True,
				      XtNmin,               10,
				      XtNresizeToPreferred, True,
				      XtNresize,            True,
				      XtNshowGrip,          False,
				      XtNskipAdjust,        True,
				      NULL);

  XtManageChild(etalk_pane);

  /* Put it all up
   * XtRealizeWidget (XCtxt->topLevel);
   * Actually, for linux this happens too soon.  If the font thingy
   * below fails, we may need to add the above line back in.  Unfortunatly
   * that also means we will get funky sized windows half the time.
   */

  /* Get the font in the window so we can make our windows the right size */
  XtVaGetValues(echo_area,
		XtNfont, &font,
		NULL);

  /* This is not true for all widget sets...

       Now resize the window so it matches a typical TTY size based on
       font, plus some arbitrary extra chars for scrollbar etc */
  XtVaSetValues(XCtxt->topLevel,
		XtNwidth, (Dimension) (font->max_bounds.width * 81 + 10),
		XtNheight, (Dimension) 
		(font->max_bounds.ascent + font->max_bounds.descent) *
		/* 10 copyright lines, 3=(menu+status+echo), 10 to spare */
		(11 + 3) + 20,
		NULL);

}


/*
 * Function: XW_fix_windows
 *
 *   This routine realigns windows when users are added or lost from
 * the window list.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt - Talk context
 *              list - Pointer toWindow list
 *
 * History:
 * zappo   9/18/95    Created
 */
void XW_fix_user_windows(Ctxt, list)
     struct TalkContext *Ctxt;
     struct WindowList *list;
{
  struct WindowList *loop;
  Dimension winsize;
  int count, done;
  
  /* count up the number of windows. In this loop, LOCAL won't count,
   * so start at 1. */
  for(loop = NextElement(list), count = 1; loop; loop=NextElement(loop))
    {
      if(DISP_window_displayed(loop))
	count++;
    }

  /* Now, lets see how much space we have */
  XtVaGetValues(etalk_pane, XtNheight, &winsize, NULL);

  if(winsize/count < 100) winsize = 100*count;
  XtVaSetValues(etalk_pane, XtNheight, winsize, NULL);

  /* rebuild the window list */
  for(loop = list, done = 0; loop; loop = NextElement(loop))
    {
      /* c short circuit protects us from a seg-fault */
      /* a non-user window is local guy, and all others have user pointers */
      /* if a user is cleaned, windows will have been cleaned */
      if(DISP_window_displayed(loop))
	{
	  if(loop->window)
	    {
	      XtVaSetValues(loop->group, XtNheight, winsize/count,
			    XtNpreferredPaneSize, winsize/count,
			    NULL);
	    }
	  else
	    {
	      char *mybuf;	/* text buffer to be used */
	      char *panename;
	      Pixel bg, fg;

	      if(loop->user)
		{
		  panename = malloc(strlen(loop->user->name) + 5);
		  sprintf(panename, "%sPane", loop->user->name);
		}
	      else
		panename = "localPane";

	      /* Create a new window... */
	      loop->group =
		XtVaCreateWidget(panename,
				 panedWidgetClass,     etalk_pane,
				 XtNorientation,       XtorientVertical,
				 XtNmin,               50,
				 XtNshowGrip,          True,
				 XtNallowResize,       True,
				 XtNheight,            winsize/count,
				 XtNpreferredPaneSize, winsize/count,
				 NULL);
	      mybuf = (char *)malloc(MAX_BUFFER_SIZE);
	      memset(mybuf, 0, MAX_BUFFER_SIZE);
	      loop->window = 
		XtVaCreateManagedWidget("UserText",
					asciiTextWidgetClass, loop->group,
					XtNstring,            mybuf,
					XtNscrollVertical,    XawtextScrollAlways,
					XtNscrollHorizontal,  XawtextScrollWhenNeeded,
					XtNallowResize,       True,
					XtNmin,               40,
					XtNshowGrip,          False,
					XtNskipAdjust,        False,
					NULL);

	      XtOverrideTranslations
		(loop->window,
		 XtParseTranslationTable
		 ("<Btn2Down>:paste-with-net()\n<Btn2Up>:paste-with-net()\n"));

	      loop->statusline =
		XtVaCreateManagedWidget("UserLabel", 
					labelWidgetClass, loop->group,
					XtNjustify,           XtJustifyLeft,
					XtNallowResize,       True,
					XtNmin,               10,
					XtNresizeToPreferred, True,
					XtNresize,            True,
					XtNshowGrip,          False,
					XtNskipAdjust,        True,
					NULL);
	      /* Invert the colors.  This way someone can just specify the
	       * pane, and inversion looks good.
	       */
	      XtVaGetValues(loop->statusline,
			    XtNforeground, &fg,
			    XtNbackground, &bg,
			    NULL);
	      XtVaSetValues(loop->statusline,
			    XtNforeground, bg,
			    XtNbackground, fg,
			    NULL);
	      XtManageChild(loop->group);
	    }
	}
      else
	{
	  /* We must free up the windows if they exist! */
	  if(loop->group)
	    {
	      X_delwin(loop->group);
	      loop->group = NULL;
	    }
	}
    }
  /* This routine is oft called from deep within other things which do
   * not call the X read routine, so call it here just to be sure */
  X_xread(Ctxt, NULL);
}


/*
 * Function: XW_popup
 *
 *   Creates a dialog shell, with a text widget in it just large
 * enough to hold the text specified by the parameters.  (Maybe, if I
 * feel like doing all that font-extents stuff.)  Returns as a void
 * pointer the shell widget.
 *
 * Returns:     Nothing
 * Parameters:  width  - Number of width
 *              height - Number of height
 * History:
 * zappo   9/18/95    Created
 */
Widget XW_popup(XCtxt, width, height)
     struct Xcontext *XCtxt;
     int width, height;
{
  Widget shell, dialog, button;
  int x, y;

  XtVaGetValues(XCtxt->topLevel, XtNx, &x, XtNy, &y, NULL);

  shell = XtVaCreatePopupShell("PopupInformationShell", 
			       transientShellWidgetClass, XCtxt->topLevel,
			       XtNtitle, "GNU Talk Information",
			       XtNx, x,
			       XtNy, y,
			       NULL);

  dialog = XtVaCreateManagedWidget("PopupInformationDialog",
				   formWidgetClass, shell, 
				   XtNwidth, 200,
				   XtNheight, 100,
				   NULL);

  PopupText
    = XtVaCreateManagedWidget("PopupInformationText",
			      asciiTextWidgetClass, dialog,
			      XtNresizable,         True,
			      XtNresize,            XawtextResizeBoth,
			      NULL);

  button =
    XtVaCreateManagedWidget("OkButton", commandWidgetClass, dialog,
			    XtNfromVert, PopupText,
			    XtNlabel,    "Ok",
			    NULL);

  XtAddCallback(button, XtNcallback, simulate_key_callback, "\033");

  /* Turn off the menu so they can't do any new dialogs */
  XtSetSensitive(menu_box, False);

  return shell;
}
void XW_kill_popup(XCtxt)
     struct Xcontext *XCtxt;
{
  PopupText = NULL;

  /* Fix the menus */
  XtSetSensitive(menu_box, True);
}

/*
 * Function: XW_popup_text, XW_add_text, and support functions
 *
 *   Adds text to an existing popup window or text window
 *
 * Returns:     Nothing
 * Parameters:  w    - Widget containing the text widget
 *              text - Pointer toCharacter of text
 *              len  - length of text
 * History:
 * zappo   9/19/95    Created
 */
void XW_popup_text(text, len)
     char *text;
     int   len;
{
  if(PopupText)
    {
      XW_add_text(PopupText, text, len);
      XW_add_text(PopupText, "\012", 1);
    }
}
static void add_text_engine(w, text, len)
     Widget w;
     char *text;
     int   len;
{
  XawTextPosition tp;
  XawTextBlock    tb;

  /* Give ourselves permission to add stuff */
  XtVaSetValues(w, XtNeditType, (XtArgVal) XawtextAppend, NULL);

  /* Get the last position */
  tp = ((TextWidget)w)->text.lastPos;
  if((len + tp) >= MAX_BUFFER_SIZE)
    {
      tb.firstPos = 0;
      tb.length = 0;
      tb.ptr = "";
      XawTextReplace(w, 0, MAX_BUFFER_SIZE - len, &tb);
      tp = ((TextWidget)w)->text.lastPos;
    }

  tb.firstPos = 0;
  tb.length = len;
  tb.ptr = text;

  XawTextReplace(w, tp, tp, &tb);
  tp = ((TextWidget)w)->text.lastPos;
  XawTextSetInsertionPoint(w, tp);

  /* Reset this value */
  XtVaSetValues(w, XtNeditType, (XtArgVal) XawtextRead, NULL);
}
int XW_winwidth(window)
     void *window;
{
  Dimension    pixwidth;
  XFontStruct *font;

  /* Get the pixel width and current font in the window */
  XtVaGetValues((Widget)window,
		XtNwidth, &pixwidth, 
		XtNfont, &font,
		NULL);

  return pixwidth / font->max_bounds.width;
}
static int current_column(window)
     void *window;
{
  XawTextPosition tp, np;
  char           *buff;
  int             linecolumn = 0;

  /* Find the current column of the "insertion point" */
  
  XtVaGetValues((Widget)window, XtNstring, &buff, NULL);

  tp = ((TextWidget)window)->text.lastPos;

  if(tp)
    {
      /* Scan over everything till we find a newline */
      for(np = tp; (np > 0) && (buff[np] != 10); np--);

      /* move back over the newline we just found */
      if(np != 0) np++;

      linecolumn = tp - np;
    }
  return linecolumn;
}
void XW_add_text(w, text, len)
     Widget w;
     char *text;
     int   len;
{
  add_text_engine(w, text, len);
  /* Don't auto-fill info boxes.  That would be silly. */
  if((w != PopupText) && (current_column(w)+2) >= XW_winwidth(w))
    add_text_engine(w, "\n", 1);
}

int XW_fillcolumn(window)
     void *window;
{
  /* Logic to determine if we should fill right now. */
  return (current_column(window) + 10) > XW_winwidth(window);
}
void XW_delchar(window)
     void *window;
{
  XawTextPosition tp;
  XawTextBlock    tb;
  char           *buff;

  XtVaGetValues((Widget)window, XtNstring, &buff, NULL);

  tp = ((TextWidget)window)->text.lastPos;

  /* Only delete back to the last line feed */
  if((tp > 0) && (buff[tp - 1] != 10))
    {
      /* Give ourselves permission to add stuff */
      XtVaSetValues((Widget)window, XtNeditType, (XtArgVal) XawtextAppend, NULL);

      tb.firstPos = 0;
      tb.length = 0;
      tb.ptr = "";
      XawTextReplace((Widget)window, tp-1, tp, &tb);
      tp = ((TextWidget)window)->text.lastPos;
      XawTextSetInsertionPoint((Widget)window, tp);

      /* Reset this value */
      XtVaSetValues((Widget)window, XtNeditType, (XtArgVal) XawtextRead, NULL);  
    }
}
void XW_delline(window)
     void *window;
{
  XawTextPosition tp, np;
  XawTextBlock    tb;
  char           *buff;

  XtVaGetValues((Widget)window, XtNstring, &buff, NULL);

  tp = ((TextWidget)window)->text.lastPos;

  if(tp)
    {
      /* Scan over everything till we find a newline */
      for(np = tp; (np > 0) && (buff[np] != 10); np--);
      /* move back over the newline we just found */
      if(np != 0) np++;
      
      /* Give ourselves permission to add stuff */
      XtVaSetValues((Widget)window, XtNeditType, (XtArgVal) XawtextAppend, NULL);
      
      tb.firstPos = 0;
      tb.length = 0;
      tb.ptr = "";
      XawTextReplace((Widget)window, np, tp, &tb);
      tp = ((TextWidget)window)->text.lastPos;
      XawTextSetInsertionPoint((Widget)window, tp);

      /* Reset this value */
      XtVaSetValues((Widget)window, XtNeditType, (XtArgVal) XawtextRead, NULL);  
    }
}
void XW_delword(window)
     void *window;
{
  XawTextPosition tp, np;
  XawTextBlock    tb;
  char           *buff;

  XtVaGetValues((Widget)window, XtNstring, &buff, NULL);

  tp = ((TextWidget)window)->text.lastPos;

  if(tp)
    {
      /* Scan over spaces first... */
      for(np = tp-1; (np > 0) && ((buff[np] == ' ') || (buff[np] == '\t')); np--);
      /* Now scan over the word... */
      for(;(np > 0) && (buff[np] != ' ') && (buff[np] != '\t') && (buff[np] != 10);
	  np--);
      /* move back over the space we just found */
      if(np != 0) np++;
      
      /* Give ourselves permission to add stuff */
      XtVaSetValues((Widget)window, XtNeditType, (XtArgVal) XawtextAppend, NULL);
      
      tb.firstPos = 0;
      tb.length = 0;
      tb.ptr = "";
      XawTextReplace((Widget)window, np, tp, &tb);
      tp = ((TextWidget)window)->text.lastPos;
      XawTextSetInsertionPoint((Widget)window, tp);

      /* Reset this value */
      XtVaSetValues((Widget)window, XtNeditType, (XtArgVal) XawtextRead, NULL);  
    }
}


/*
 * Function: XW_clearwin
 *
 *   This will clear one text window specified in WINDOW
 *
 * Returns:     Nothing
 * Parameters:  window - Widget to window
 *
 * History:
 * zappo   12/9/95    Created
 */
void XW_clearwin(window)
     void *window;
{
  XawTextPosition tp;
  XawTextBlock    tb;
  char           *buff;

  XtVaGetValues((Widget)window, XtNstring, &buff, NULL);
  tp = ((TextWidget)window)->text.lastPos;
  if(tp)
    {
      /* Give ourselves permission to add stuff */
      XtVaSetValues((Widget)window, XtNeditType, (XtArgVal) XawtextAppend, NULL);

      tb.firstPos = 0;
      tb.length = 0;
      tb.ptr = "";
      XawTextReplace((Widget)window, zeroPosition, tp, &tb);
      tp = ((TextWidget)window)->text.lastPos;
      XawTextSetInsertionPoint((Widget)window, tp);
      
      /* Reset this value */
      XtVaSetValues((Widget)window, XtNeditType, (XtArgVal) XawtextRead, NULL);  
    }
}


/*
 * Function: XW_setlabel
 *
 *   Sets a label's value.
 *
 * Returns:     Nothing
 * Parameters:  win   - EmptyPointer to the window to set
 *              label - Pointer toCharacter of label
 * History:
 * zappo   9/18/95    Created
 */
void XW_setlabel(win, label)
     void *win;
     char *label;
{
  XtVaSetValues((Widget)win, XtNlabel, label, NULL);
}

/*
 * Function: XW_build_menu
 *
 *   Use this to build one widget menu with callbacks.  All callbacks
 * are given data 
 *
 * Returns:     Nothing
 * Parameters:  parent   - Widget parent
 *              name     - Name of
 *              items    -  items
 *              numitems - Number of number
 * History:
 * zappo   9/16/95    Created
 */
void XW_build_menu(parent, name, items, numitems)
     Widget parent;
     char *name;
     struct MenuItemConstructor *items;
     int numitems;
{
  Widget mbutton, menu, item;
  int i;

  mbutton = XtVaCreateManagedWidget(name, menuButtonWidgetClass, parent,
				    XtNlabel,    name,
				    XtNmenuName, name,
				    XtNresize,   False,
				    NULL);

  menu = XtVaCreatePopupShell(name, simpleMenuWidgetClass, mbutton,
			      XtNlabel, name,
			      NULL);

  XtOverrideTranslations
    (menu,
     XtParseTranslationTable("<BtnMotion>: highlight() show-message()"));

  for (i=0; i < numitems; i++) {

    /* Awful hack Part 1:
     * Put the description (long) in the NAME slot of this widget.
     * This is because the name is never seen except by hackers, and
     * because there is no user data to put such useful information.
     */
    item = XtVaCreateManagedWidget(items[i].describe, smeBSBObjectClass,
				   menu, 
				   XtNlabel, items[i].button_name,
				   NULL);

    XtAddCallback(item, XtNcallback, items[i].callback, items[i].data);
  }
}

/*
 * Function: XW_set_echo_label
 *
 *   Set the value of the echo area label.
 *
 * Returns:     Nothing
 * Parameters:  str        - String Pointer  string
 *              showcursor - State of the cursor
 * History:
 * zappo   9/17/95    Created
 * zappo   1/30/97    Added parameters showcursor
 */
void XW_set_echo_label(str, showcursor)
     char *str;
     int showcursor;
{
  if(echo_area)
    {
      XtVaSetValues(echo_area, 
		    XtNstring, str,
		    XtNdisplayCaret, showcursor,
		    NULL);

      XawTextSetInsertionPoint(echo_area,
			       ((TextWidget)echo_area)->text.lastPos);

      XFlush(XtDisplay(echo_area));
    }
}

static void XW_show_message(w, event, prms, nprms)
     Widget    w;
     XEvent   *event;
     String   *prms;
     Cardinal *nprms;
{
  static Widget lastw = NULL;
  Widget neww;

  neww = XawSimpleMenuGetActiveEntry(w);

  if(neww != lastw)
    {
      if(neww)
	{
	  /* Awful hack Part 2:
	   * Extract the name, and display it in the minibuffer
	   */
	  XW_set_echo_label(XtName(neww), FALSE);
	}
      else
	{
	  XW_set_echo_label("", FALSE);
	}
      lastw = neww;
    }
}
