/* gt_assoc.c - Manages file/viewer associations
 *
 * Copyright (C) 1996, 1997 Eric M. Ludlam
 * Copyright (C) 1997 Free Software Foundation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 * 
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 * 
 * Please send bug reports, etc. to zappo@gnu.org.
 * 
 * 
 * Description:
 *   This file manages a simple list of file sub-strings and the
 * associated external viewer.  When a file needs to be viewed (due to
 * a download, or some such) then these routines are used to determine
 * how to display it.
 *   
 * $Log: gt_assoc.c,v $
 * Revision 1.9  1998/01/04 13:29:48  zappo
 * Fixed typo in fork_assoc causing errors w/ SunOS cc
 *
 * Revision 1.8  1997/12/14 19:14:43  zappo
 * Renamed package to gtalk, renamed symbols and files apropriately
 * Fixed copyright and email address.
 *
 * Revision 1.7  1997/07/27 17:37:12  zappo
 * Fixed compile warning
 *
 * Revision 1.6  1997/07/22 13:21:31  zappo
 * Changed to use the list library.
 *
 * Revision 1.5  1997/02/01 14:31:47  zappo
 * More changes to fix app_def calls to FORK_save_pid.
 *
 * Revision 1.4  1997/01/28  03:18:30  zappo
 * changed calls to FORK_save_pid
 *
 * Revision 1.3  1997/01/26  15:29:39  zappo
 * Fixed parameters to FORK_save_pid since they changed.
 *
 * Revision 1.2  1996/03/02  03:12:49  zappo
 * Fixed some -Wall warnings
 *
 * Revision 1.1  1996/01/25  02:48:33  zappo
 * Initial revision
 *
 * History:
 * zappo   1/21/96    Created
 *
 * Tokens: ::Header:: gtproc.h
 */

#include "gtalklib.h"
#include "gtalkc.h"
#include "gtproc.h"
#include "sitecnfg.h"

static MakeList(list);


/*
 * Function: ASSOC_add
 *
 *   Adds an association between a file type and an application exec
 * string for which any following parameters will be interpreted as
 * the file being viewed.
 *
 * Returns:     int  - 
 * Parameters:  Ctxt     - Context
 *              match    - Pointer toCharacter of match
 *              app_exec - Pointer toCharacter of app exec
 * History:
 * zappo   11/8/95    Created
 */
int ASSOC_add(Ctxt, match, app_exec)
     struct TalkContext *Ctxt;
     char *match;
     char *app_exec;
{
  struct assoc_node *new;

  if(!match || ! app_exec)
    {
      DISP_message(Ctxt, "Null pointer passed to ASSOC_add");
      return Fail;
    }

  new = (struct assoc_node *)LIST_alloc(&list, sizeof (struct assoc_node));

  new->match = strdup(match);
  new->app_exec = strdup(app_exec);
  if(!new->match || !new->app_exec)
    {
      gtalk_shutdown("strdup failed in ASSOC_add");
    }

  return Success;
}

/*
 * Function: ASSOC_number
 *
 *   Returns the number of associations.
 *
 * Returns:     int  - 
 * Parameters:  None
 *
 * History:
 * zappo   11/9/95    Created
 */
int ASSOC_number()
{
  return LIST_map(&list, NULL, NULL);
}

/*
 * Function: ASSOC_print
 *
 *   Prints out the association list.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt - Context
 *
 * History:
 * zappo   11/9/95    Created
 */
static void ASSOC_print_node(node, Ctxt)
     struct assoc_node *node;
     struct TalkContext *Ctxt;
{
  char buffer[200];

  sprintf(buffer, "%s\t\t%s", node->match, node->app_exec);
  DISP_message(Ctxt, buffer);
}

void ASSOC_print(Ctxt)
     struct TalkContext *Ctxt;
{
  DISP_message(Ctxt, "Suffix\t\tAssociated Application");

  LIST_map(&list, ASSOC_print_node, Ctxt);
}

/*
 * Function: ASSOC_fork_assoc
 *
 *   Fork an application based on the association list stored here.
 * If no match is found, then launch the default viewer.
 *
 * Returns:     int  - 
 * Parameters:  Ctxt - Context
 *              file - Pointer toCharacter of file
 * History:
 * zappo   11/8/95    Created
 */
int ASSOC_fork_assoc(Ctxt, file)
     struct TalkContext *Ctxt;
     char *file;
{
  struct assoc_node *loop;
  union app_def      app;
  char *env;
  pid_t kid;
  static struct assoc_node default_env = {{ NULL, NULL }, "Any", NULL };
  static struct assoc_node default_pg = {{ NULL, NULL }, "Any", DEFAULT_PAGER};

  /* look for an association.  Things like .gif and launch xv */
  for(loop = FirstElement(list); 
      loop && !strstr(file, loop->match);
      loop = NextElement(loop));
 
  if(loop)
    {
      kid = FORK_try(Ctxt, loop->app_exec, file, NULL, 0, 0);
      if(!kid)
	return Fail;
      app.assoc = loop;
      FORK_save_pid(kid, APP_ASSOC, app, file, NULL);
    }
  else if(Ctxt->runstate == Socket)
    {
      /* In this case, tell our parent about it in case they have a
	 better method of displaying files.  Since the only one so far
	 is emacs, I think this is a good bet. */
      printf("%c%c%s\n", ETALK_ESCAPE, TTY_SHOW_FILE, file);
      kid = 0;
    }
  else if((env = getenv(DEFAULT_PAGER_ENV)))
    {
      default_env.app_exec = env;
      kid = FORK_try(Ctxt, env, file, NULL, 0, 0);
      if(!kid)
	{
	  kid = FORK_try(Ctxt, DEFAULT_PAGER, file, NULL, 0, 0);
	  if(!kid)
	    return Fail;
	  app.assoc = &default_pg;
	  FORK_save_pid(kid, APP_ASSOC, app, file, NULL);
	}
      else
	{
	  app.assoc = &default_env;
	  FORK_save_pid(kid, APP_ASSOC, app, file, NULL);
	}
    }
  else
    {
      kid = FORK_try(Ctxt, DEFAULT_PAGER, file, NULL, 0, 0);
      if(!kid)
	return Fail;
      app.assoc = &default_pg;
      FORK_save_pid(kid, APP_ASSOC, app, file, NULL);
    }
  return Success;
}
