/* GStreamer process pipe
 * Copyright (C) 2006 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1307, USA.
 */

/**
 * SECTION:element-procpipe
 *
 * <refsect2>
 * <para>
 * Transforms a data stream by having it processed by a child process
 * (given by <link linkend="GstProcTrans--command">command</link>),
 * providing data to its stdin and receiving data from its stdout.
 * Note that output buffers are quite raw in that they have neither caps
 * nor any metadata (timestamps, &hellip;).
 * </para>
 * </refsect2>
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gst/gst.h>

#include "gstproctrans.h"

#define GST_TYPE_PROC_PIPE \
  gst_proc_pipe_get_type ()
#define GST_PROC_PIPE(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj), GST_TYPE_PROC_PIPE, GstProcPipe))
#define GST_PROC_PIPE_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass), GST_TYPE_PROC_PIPE, GstProcPipeClass))
#define GST_IS_PROC_PIPE(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj), GST_TYPE_PROC_PIPE))
#define GST_IS_PROC_PIPE_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass), GST_TYPE_PROC_PIPE))

typedef struct _GstProcPipe GstProcPipe;
typedef struct _GstProcPipeClass GstProcPipeClass;

struct _GstProcPipe
{
  GstProcTrans parent;

  gchar *args;
};

struct _GstProcPipeClass
{
  GstProcTransClass parent_class;
};

GST_DEBUG_CATEGORY_STATIC (proc_pipe_debug);
#define GST_CAT_DEFAULT proc_pipe_debug

static GstStaticPadTemplate sink_template =
GST_STATIC_PAD_TEMPLATE (GST_PROC_TRANS_SINK_NAME,
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS_ANY);

static GstStaticPadTemplate src_template =
GST_STATIC_PAD_TEMPLATE (GST_PROC_TRANS_SRC_NAME,
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS_ANY);

/* properties */
enum
{
  PROP_0,
  PROP_ARGS
};

#define DEFAULT_BLOCKSIZE  16384

static void gst_proc_pipe_finalize (GObject * object);
static gboolean gst_proc_pipe_set_caps (GstProcTrans * ptrans, GstCaps * incaps,
    GstCaps ** outcaps);
static GstStateChangeReturn gst_proc_pipe_change_state (GstElement * element,
    GstStateChange transition);

/* properties */
static void gst_proc_pipe_set_property (GObject * object,
    guint prop_id, const GValue * value, GParamSpec * pspec);
static void gst_proc_pipe_get_property (GObject * object,
    guint prop_id, GValue * value, GParamSpec * pspec);

GST_BOILERPLATE (GstProcPipe, gst_proc_pipe, GstProcTrans, GST_TYPE_PROC_TRANS);

static void
gst_proc_pipe_base_init (gpointer klass)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (klass);

  gst_element_class_set_details_simple (element_class, "Process Pipe",
      "Filter", "Feeds input into process stdin and pushes its stdout",
      "Mark Nauwelaerts <mnauw@users.sourceforge.net>");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&src_template));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&sink_template));
}

static void
gst_proc_pipe_class_init (GstProcPipeClass * klass)
{
  GstElementClass *element_class;
  GObjectClass *gobject_class;

  gobject_class = G_OBJECT_CLASS (klass);
  element_class = GST_ELEMENT_CLASS (klass);

  GST_DEBUG_CATEGORY_INIT (proc_pipe_debug, "procpipe", 0, "Process Pipe");

  gobject_class->finalize = GST_DEBUG_FUNCPTR (gst_proc_pipe_finalize);

  gobject_class->set_property = gst_proc_pipe_set_property;
  gobject_class->get_property = gst_proc_pipe_get_property;

  g_object_class_install_property (G_OBJECT_CLASS (klass), PROP_ARGS,
      g_param_spec_string ("args", "arguments",
          "Arguments for the command to execute", NULL, G_PARAM_READWRITE));

  element_class->change_state = GST_DEBUG_FUNCPTR (gst_proc_pipe_change_state);
}

static void
gst_proc_pipe_init (GstProcPipe * pipe, GstProcPipeClass * klass)
{
  GstProcTransClass *proctransclass = GST_PROC_TRANS_CLASS (klass);

  proctransclass->set_caps = GST_DEBUG_FUNCPTR (gst_proc_pipe_set_caps);

  /* most properties are initialized by base element */
  pipe->args = NULL;
}


static void
gst_proc_pipe_finalize (GObject * object)
{
  GstProcPipe *pipe = GST_PROC_PIPE (object);

  g_free (pipe->args);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

static gboolean
gst_proc_pipe_set_caps (GstProcTrans * ptrans, GstCaps * incaps,
    GstCaps ** outcaps)
{
  GstProcPipe *pipe = GST_PROC_PIPE (ptrans);

  if (pipe->args) {
    gint count;
    gchar **args;
    GError *err = NULL;

    if (!g_shell_parse_argv (pipe->args, &count, &args, &err)) {
      g_return_val_if_fail (err != NULL, FALSE);
      GST_ELEMENT_ERROR (pipe, RESOURCE, FAILED,
          ("glib error %s", err->message ? err->message : ""),
          ("parsing %s", pipe->args));
      return FALSE;
    }
    g_array_append_vals (ptrans->args, args, count);
    /* only free the list of pointers, not their contents */
    g_free (args);
  }

  return TRUE;
}


static void
gst_proc_pipe_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstProcPipe *pipe;

  g_return_if_fail (GST_IS_PROC_PIPE (object));

  pipe = GST_PROC_PIPE (object);

  switch (prop_id) {
    case PROP_ARGS:
      g_free (pipe->args);
      pipe->args = g_value_dup_string (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_proc_pipe_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstProcPipe *pipe;

  g_return_if_fail (GST_IS_PROC_PIPE (object));

  pipe = GST_PROC_PIPE (object);

  switch (prop_id) {
    case PROP_ARGS:
      g_value_take_string (value, g_strdup (pipe->args));
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static GstStateChangeReturn
gst_proc_pipe_change_state (GstElement * element, GstStateChange transition)
{
  GstStateChangeReturn ret;

  ret = GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);
  if (ret == GST_STATE_CHANGE_FAILURE)
    goto done;

  switch (transition) {
    case GST_STATE_CHANGE_PAUSED_TO_READY:
      break;
    default:
      break;
  }

done:
  return ret;
}
