/* GStreamer multi process pipe
 * Copyright (C) 2007 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1307, USA.
 */

 /**
 * SECTION:element-multiprocpipe
 *
 * <refsect2>
 * <para>
 * Transforms a data stream by spawning a child process
 * (given by <link linkend="GstMultProcTrans--command">command</link>) for
 * each buffer received, feeding this to the process' stdin
 * and pushing out the resulting stdout as a single buffer.
 * </para>
 * </refsect2>
 *
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gst/gst.h>

#include "gstmultiproctrans.h"

#include <string.h>

#define GST_TYPE_MULTI_PROC_PIPE \
  gst_multi_proc_pipe_get_type ()
#define GST_MULTI_PROC_PIPE(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj), GST_TYPE_MULTI_PROC_PIPE, GstMultiProcPipe))
#define GST_MULTI_PROC_PIPE_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass), GST_TYPE_MULTI_PROC_PIPE, GstMultiProcPipeClass))
#define GST_IS_MULTI_PROC_PIPE(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj), GST_TYPE_MULTI_PROC_PIPE))
#define GST_IS_MULTI_PROC_PIPE_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass), GST_TYPE_MULTI_PROC_PIPE))

typedef struct _GstMultiProcPipe GstMultiProcPipe;
typedef struct _GstMultiProcPipeClass GstMultiProcPipeClass;

struct _GstMultiProcPipe
{
  GstMultiProcTrans parent;

  gchar *args;
};

struct _GstMultiProcPipeClass
{
  GstMultiProcTransClass parent_class;
};

GST_DEBUG_CATEGORY_STATIC (multi_proc_pipe_debug);
#define GST_CAT_DEFAULT multi_proc_pipe_debug

static GstStaticPadTemplate sink_template =
GST_STATIC_PAD_TEMPLATE (GST_MULTI_PROC_TRANS_SINK_NAME,
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS_ANY);

static GstStaticPadTemplate src_template =
GST_STATIC_PAD_TEMPLATE (GST_MULTI_PROC_TRANS_SRC_NAME,
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS_ANY);

/* properties */
enum
{
  PROP_0,
  PROP_ARGS
};

static void gst_multi_proc_pipe_finalize (GObject * object);
static gboolean gst_multi_proc_pipe_set_caps (GstMultiProcTrans * ptrans,
    GstCaps * incaps, GstCaps ** outcaps);
static GstStateChangeReturn gst_multi_proc_pipe_change_state (GstElement *
    element, GstStateChange transition);

/* properties */
static void gst_multi_proc_pipe_set_property (GObject * object,
    guint prop_id, const GValue * value, GParamSpec * pspec);
static void gst_multi_proc_pipe_get_property (GObject * object,
    guint prop_id, GValue * value, GParamSpec * pspec);

GST_BOILERPLATE (GstMultiProcPipe, gst_multi_proc_pipe, GstMultiProcTrans,
    GST_TYPE_MULTI_PROC_TRANS);

static void
gst_multi_proc_pipe_base_init (gpointer klass)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (klass);

  gst_element_class_set_details_simple (element_class, "Multi Process Pipe",
      "Filter",
      "Feeds input into (a) process (per buffer) and pushes the output",
      "Mark Nauwelaerts <mnauw@users.sourceforge.net>");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&src_template));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&sink_template));
}

static void
gst_multi_proc_pipe_class_init (GstMultiProcPipeClass * klass)
{
  GstElementClass *element_class;
  GObjectClass *gobject_class;

  gobject_class = G_OBJECT_CLASS (klass);
  element_class = GST_ELEMENT_CLASS (klass);

  GST_DEBUG_CATEGORY_INIT (multi_proc_pipe_debug, "multiprocpipe", 0,
      "Multi Process Pipe");

  gobject_class->finalize = GST_DEBUG_FUNCPTR (gst_multi_proc_pipe_finalize);

  gobject_class->set_property = gst_multi_proc_pipe_set_property;
  gobject_class->get_property = gst_multi_proc_pipe_get_property;

  g_object_class_install_property (G_OBJECT_CLASS (klass), PROP_ARGS,
      g_param_spec_string ("args", "arguments",
          "Arguments for the command to execute", NULL,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  element_class->change_state =
      GST_DEBUG_FUNCPTR (gst_multi_proc_pipe_change_state);
}

static void
gst_multi_proc_pipe_init (GstMultiProcPipe * pipe,
    GstMultiProcPipeClass * klass)
{
  GstMultiProcTransClass *mptclass = GST_MULTI_PROC_TRANS_CLASS (klass);

  mptclass->set_caps = GST_DEBUG_FUNCPTR (gst_multi_proc_pipe_set_caps);

  /* most properties are initialized by base element */
  pipe->args = NULL;
}


static void
gst_multi_proc_pipe_finalize (GObject * object)
{
  GstMultiProcPipe *pipe = GST_MULTI_PROC_PIPE (object);

  g_free (pipe->args);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

static gboolean
gst_multi_proc_pipe_update_args (GstMultiProcPipe * pipe)
{
  GstMultiProcTrans *ptrans;

  ptrans = GST_MULTI_PROC_TRANS (pipe);

  if (pipe->args) {
    gint count;
    gchar **args;
    GError *err = NULL;

    GST_INFO ("%s", pipe->args);
    if (!g_shell_parse_argv (pipe->args, &count, &args, &err)) {
      g_return_val_if_fail (err != NULL, FALSE);
      GST_ELEMENT_ERROR (pipe, RESOURCE, FAILED,
          ("glib error %s", err->message ? err->message : ""),
          ("parsing %s", pipe->args));
      return FALSE;
    }
    g_array_append_vals (ptrans->args, args, count);
    /* only free the list of pointers, not their contents */
    g_free (args);
  }

  return TRUE;
}

static gboolean
gst_multi_proc_pipe_set_caps (GstMultiProcTrans * ptrans, GstCaps * incaps,
    GstCaps ** outcaps)
{
  GstMultiProcPipe *pipe = GST_MULTI_PROC_PIPE (ptrans);

  return gst_multi_proc_pipe_update_args (pipe);
}

static void
gst_multi_proc_pipe_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstMultiProcPipe *pipe;

  g_return_if_fail (GST_IS_MULTI_PROC_TRANS (object));

  pipe = GST_MULTI_PROC_PIPE (object);

  switch (prop_id) {
    case PROP_ARGS:
      if (!pipe->args || !strcmp (pipe->args, g_value_get_string (value))) {
        g_free (pipe->args);
        pipe->args = g_value_dup_string (value);
        gst_multi_proc_pipe_update_args (pipe);
      }
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_multi_proc_pipe_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec)
{
  GstMultiProcPipe *pipe;

  g_return_if_fail (GST_IS_MULTI_PROC_TRANS (object));

  pipe = GST_MULTI_PROC_PIPE (object);

  switch (prop_id) {
    case PROP_ARGS:
      g_value_take_string (value, g_strdup (pipe->args));
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static GstStateChangeReturn
gst_multi_proc_pipe_change_state (GstElement * element,
    GstStateChange transition)
{
  GstStateChangeReturn ret;

  ret = GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);
  if (ret == GST_STATE_CHANGE_FAILURE)
    goto done;

  switch (transition) {
    case GST_STATE_CHANGE_PAUSED_TO_READY:
      break;
    default:
      break;
  }

done:
  return ret;
}
