/* GStreamer Filter
 * Copyright (C) 2006 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * transcode filter:
 * Copyright (C) Tilmann Bitterberg - June 2002
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1307  USA
 */

/**
 * SECTION:element-csub
 *
 * <refsect2>
 * <para>
 * Subtract <link linkend="GstCsub--red-sub">red-sub</link> from the Cr chroma
 * component and/or <link linkend="GstCsub--blue-sub">blue-sub</link> from the
 * Cb chroma component.
 * </para>
 * <title>History</title>
 * <para>
 * <itemizedlist>
 * <listitem>
 * <para>
 * Transcode nored filter [Tilmann Bitterberg]
 * </para>
 * </listitem>
 * </itemizedlist>
 * </para>
 * </refsect2>
 *
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "plugin-tc.h"

#include <string.h>


#define GST_TYPE_CSUB \
  (gst_csub_get_type())
#define GST_CSUB(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj),GST_TYPE_CSUB,GstCsub))
#define GST_CSUB_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass),GST_TYPE_CSUB,GstCsubClass))
#define GST_IS_CSUB(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj),GST_TYPE_CSUB))
#define GST_IS_CSUB_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass),GST_TYPE_CSUB))


typedef struct _GstCsub GstCsub;
typedef struct _GstCsubClass GstCsubClass;

struct _GstCsub
{
  GstVideoFilter videofilter;

  gint width, height;

  /* properties */
  gint blue_sub, red_sub;
};


struct _GstCsubClass
{
  GstVideoFilterClass parent_class;
};

GST_DEBUG_CATEGORY_STATIC (csub_debug);
#define GST_CAT_DEFAULT csub_debug


/* signals and args */
enum
{
  /* FILL ME */
  LAST_SIGNAL
};

enum
{
  PROP_0,
  PROP_BLUE_SUB,
  PROP_RED_SUB
      /* FILL ME */
};

#define DEFAULT_BLUE_SUB      0
#define DEFAULT_RED_SUB       2

static GstStaticPadTemplate gst_csub_src_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SRC_NAME,
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420 }"))
    );

static GstStaticPadTemplate gst_csub_sink_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SINK_NAME,
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420 }"))
    );

static GstFlowReturn gst_csub_transform_ip (GstBaseTransform * btrans,
    GstBuffer * in);
static gboolean gst_csub_start (GstBaseTransform * btrans);
static gboolean gst_csub_stop (GstBaseTransform * btrans);

static void gst_csub_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_csub_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);

GST_BOILERPLATE (GstCsub, gst_csub, GstVideoFilter, GST_TYPE_VIDEO_FILTER);

GST_VIDEO_FILTER_SET_CAPS_BOILERPLATE (GstCsub, gst_csub);

static void
gst_csub_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_set_details_simple (element_class, "Csub",
      "Filter/Effect/Video", "Chroma subtracter; aka nored the image",
      "Mark Nauwelaerts <mnauw@users.sourceforge.net>,\n" "Tilmann Bitterberg");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_csub_sink_template));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_csub_src_template));
}

static void
gst_csub_class_init (GstCsubClass * g_class)
{
  GObjectClass *gobject_class;
  GstBaseTransformClass *trans_class;

  gobject_class = G_OBJECT_CLASS (g_class);
  trans_class = GST_BASE_TRANSFORM_CLASS (g_class);

  GST_DEBUG_CATEGORY_INIT (csub_debug, "csub", 0, "csub");

  gobject_class->set_property = gst_csub_set_property;
  gobject_class->get_property = gst_csub_get_property;

  g_object_class_install_property (gobject_class, PROP_BLUE_SUB,
      g_param_spec_int ("blue-sub", "Blue (Cb) Subtract",
          "Subtract blue from Cb",
          -127, 127, DEFAULT_BLUE_SUB,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_RED_SUB,
      g_param_spec_int ("red-sub", "Red (Cr) Subtract",
          "Subtract red from Cr",
          -127, 127, DEFAULT_RED_SUB,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  trans_class->set_caps = GST_DEBUG_FUNCPTR (gst_csub_set_caps);
  trans_class->transform_ip = GST_DEBUG_FUNCPTR (gst_csub_transform_ip);
  trans_class->start = GST_DEBUG_FUNCPTR (gst_csub_start);
  trans_class->stop = GST_DEBUG_FUNCPTR (gst_csub_stop);
}

static void
gst_csub_init (GstCsub * filter, GstCsubClass * g_class)
{
  filter->blue_sub = DEFAULT_BLUE_SUB;
  filter->red_sub = DEFAULT_RED_SUB;
}

static GstFlowReturn
gst_csub_transform_ip (GstBaseTransform * btrans, GstBuffer * in)
{
  GstCsub *filter;
  guint8 *src, *cubuf, *cvbuf;
  guint width, height, stride;
  guint y;

  gst_object_sync_values (G_OBJECT (btrans), GST_BUFFER_TIMESTAMP (in));

  filter = GST_CSUB (btrans);

  src = (guint8 *) GST_BUFFER_DATA (in);

  width = filter->width;
  height = filter->height;

  stride = GST_VIDEO_I420_V_ROWSTRIDE (width);

  cubuf = src + GST_VIDEO_I420_U_OFFSET (width, height);
  cvbuf = src + GST_VIDEO_I420_V_OFFSET (width, height);

  GST_DEBUG ("chain, %d ,%d", filter->blue_sub, filter->red_sub);

  for (y = 0; y < (height * stride) / 2; y++) {
    *cubuf = (*cubuf - filter->blue_sub) & 0xFF;
    *cvbuf = (*cvbuf - filter->red_sub) & 0xFF;
  }

  return GST_FLOW_OK;
}


static gboolean
gst_csub_start (GstBaseTransform * btrans)
{
  return TRUE;
}

static gboolean
gst_csub_stop (GstBaseTransform * btrans)
{
  return TRUE;
}

static void
gst_csub_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstCsub *src;

  g_return_if_fail (GST_IS_CSUB (object));
  src = GST_CSUB (object);

  switch (prop_id) {
    case PROP_BLUE_SUB:
      src->blue_sub = g_value_get_int (value);
      break;
    case PROP_RED_SUB:
      src->red_sub = g_value_get_int (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_csub_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstCsub *src;

  g_return_if_fail (GST_IS_CSUB (object));
  src = GST_CSUB (object);

  switch (prop_id) {
    case PROP_BLUE_SUB:
      g_value_set_int (value, src->blue_sub);
      break;
    case PROP_RED_SUB:
      g_value_set_int (value, src->red_sub);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}
